/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anthony Bussani - Initial API and implementation
 *     Thomas Gross - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard;

import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map;

import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.PasswordCallback;

import org.eclipse.higgins.registry.IConfiguration;

/**
 * Strategy for storing and retrieving data from a CardStore. The strategy must
 * be initialized to be functional.
 * 
 * The exportCards() method is to store the cards under the same paradigm the
 * strategy uses for storing cards. E.g., if the strategy stores ICards in a
 * encrypted CRDS file, the exportCards() method will also store the exported
 * cards in such a CRDS file.
 * 
 * @author Anthony Bussani
 * @author Thomas Gross
 */
public interface CardStoreStrategy {
    /**
         * Initializes the ICardStoreStrategy before it is used. The
         * ICardStoreStrategy SHOULD throw a ICardStoreException if other
         * functions are leveraged before the strategy is initialized.
         * 
         * If the File with filename does not exist the strategy assumes that
         * the CardStore needs to be newly generated and initiates a new
         * CardStore at the designated location.
         * 
         * @param authHandler
         *                CallbackHandler for handling password events
         * @param filename
         *                target location of the CardStore. Can refer to a
         *                directory for directory-based CardStores or a single
         *                File.
         * @param passwordCallback
         *                the password handler to use if password is required by
         *                the type of implemented store
         * @param config
         * @throws ICardStoreException
         * 
         * @author Anthony Bussani
         * @author Thomas Gross
         */
    public void initialize(CallbackHandler authHandler, ICardProvider cardProvider, String filename,
	    PasswordCallback passwordCallback, IConfiguration configuration)
	    throws CardStoreException;

    /**
         * Synchronizes the CardStore from the provided in-memory Map icards.
         * The icards are transfered to the disk CardStore.
         * 
         * @param authHandler
         * @param icards
         *                Map with the memory representation of the CardStore
         * @throws ICardStoreException
         * 
         * @author Anthony Bussani
         * @author Thomas Gross
         */
    public void synchFromMap(CallbackHandler authHandler, Map icards)
	    throws CardStoreException;

    /**
         * Synchronizes the CardStore from the disk representation. The icards
         * are transfered from the disk CardStore to the provided in-memory map.
         * 
         * @param authHandler
         * @param icards
         *                icards Map with the memory representation of the
         *                CardStore
         * @throws ICardStoreException
         * 
         * @author Anthony Bussani
         * @author Thomas Gross
         */
    public void synchFromStore(CallbackHandler authHandler, Map icards)
	    throws CardStoreException;

    /**
         * Exports the cards provided as Iterator cards to stream out. The
         * strategy leverages the very same paradigm for the export as for
         * normal storing of cards.
         * 
         * @param authHandler
         * @param cards
         *                Iterator of cards to be exported
         * @param out
         *                OutputStream where the cards are stored.
         * @throws ICardException
         * 
         * @author Anthony Bussani
         * @author Thomas Gross
         */
    public void exportCards(CallbackHandler authHandler, Iterator cards,
	    OutputStream out) throws CardException;

    /**
         * Determines whether this strategy is capable of protecting its ICards
         * by a password. Implicitly assumes that the password is changeable.
         * 
         * @return true if the strategy handles passwords.
         * 
         * @author Thomas Gross
         */
    public boolean isPasswordProtected();

    /**
         * Changes the password of a password protected CardStore.
         * 
         * @param oldPassword
         * @param newPassword
         * 
         * @throws UnsupportedOperationException
         *                 if the CardStore does not support passwords.
         * @throws ICardStoreException
         *                 if the oldPassword is not correct.
         * 
         * @author Thomas Gross
         */
    public void changePassword(char[] oldPassword, char[] newPassword)
	    throws CardStoreException;

    /**
         * This should be part of a future CardManager
         * 
         * @param asciiStore
         * @throws CardStoreException
         */
    public Map importStore(CallbackHandler authHandler, String asciiStore)
	    throws CardStoreException;
}
