/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard;

import java.util.Iterator;
import java.util.Properties;

import javax.security.auth.callback.CallbackHandler;

import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.registry.IServiceProvider;

public interface ICardProvider extends IServiceProvider {
	public static final String CONTEXT_REF_PROP_NAME = "context.uri";

	public static final String CONTEXT_SCHEMA_URL_PROP_NAME = "schema.url";

	public static final String CONTEXT_SCHEMA_FILE_PROP_NAME = "schema.file";

	public static final String IMAGE_URL_PROP_NAME = "image.url";

	/**
	 * Gets human readable name of this i-card provider.
	 */
	public String getName();

	/**
	 * Gets human readable description of this i-card provider.
	 */
	public String getDescription();

	/**
	 * Gets the type of i-cards this i-card provider manages (e.g. "uri",
	 * "cardspace", "idemix", "openid", etc.)
	 */
	//public String getType();
	
	/**
	 * Gets an array of <code>ICard</code> interfaces supported by this i-card
	 * provider (e.g. ICard, IInformationCard, IManagedInformationCard,
	 * IPersonalInformationCard, IURICard, etc.).
	 */
	public Class[] getSupportedTypes();

	/**
	 * Gets all available user's <code>ICard</code>s whose identity could be
	 * requested using provided callback handler.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @return the <code>Iterator</code> through all user's <code>ICard</code>s.
	 */
	public  Iterator getICards(CallbackHandler authHandler) throws CardException;
	
	/**
	 * Checks all available user's <code>ICard</code>s whose identity could
	 * be requested using provided callback handler against provided
	 * <code>policy</code> and returns all satisfied.
	 * 
	 * <p>
	 * Typically, when <code>polocy</code> parameter is not <code>null</code>,
	 * i-card provider uses {@link IPolicy#isSatisfiedBy(ICard)} to find matched
	 * <code>ICard</code>s.
	 * 
	 * <p>
	 * When <code>policy</code> parameter is <code>null</code> i-card
	 * provider returns all available <code>ICard</code>s.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param policy -
	 *            the policy to be used to select satisfied <code>ICard</code>s
	 *            only.
	 * 
	 * @return the <code>Iterator</code> through all satisfied
	 *         <code>ICard</code>s.
	 */
	public Iterator getICardsByPolicy(CallbackHandler authHandler, IPolicy policy) throws CardException;

	/**
	 * Gets all user's <code>ICard</code>s whose identity could be requested
	 * using provided callback handler that could be exported to
	 * <code>IElement</code> with the format described by
	 * <code>format</code> parameter.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param format -
	 *            the descriptor of the element for which export capabilities
	 *            need to be checked.
	 * 
	 * @return the <code>Iterator</code> through all satisfied
	 *         <code>ICard</code>s.
	 */
	public Iterator getICardsByFormat(CallbackHandler authHandler, IElementFormat format) throws CardException;
	
	/**
	 * Gets user's <code>ICard</code> whose identity could be requested using
	 * spesified callback handler which have the spesified unique identifier.
	 * <p>
	 * Card object returned by this method call may or may not have its claims
	 * retieved from endpoint service. When a card doesn't have its claims
	 * retieved ({@link ICard#isClaimsRetrieved()} returns <code>false</code>)
	 * it is necessary to make additional call to
	 * {@link #getICardByCUID(CallbackHandler, CUID, ICredential)} in order to
	 * obtain fully loaded card object.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param CUID -
	 *            the string representation of unique identifier of i-card which
	 *            is requested.
	 * @return the <code>ICard</code> instance.
	 * @throws CardException
	 *             if <code>ICard</code> with <code>CUID</code> couldn't be
	 *             retrieved.
	 */
	//public ICard getICardByCUID(CallbackHandler authHandler, String CUID) throws CardException;

	/**
	 * Gets user's <code>ICard</code> whose identity could be requested using
	 * spesified callback handler which have the spesified unique identifier.
	 * <p>
	 * Card object returned by this method call may or may not have its claims
	 * retieved from endpoint service. When a card doesn't have its claims
	 * retieved ({@link ICard#isClaimsRetrieved()} returns <code>false</code>)
	 * it is necessary to make additional call to
	 * {@link #getICardByCUID(CallbackHandler, CUID, ICredential)} in order to
	 * obtain fully loaded card object.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param cuid -
	 *            the unique identifier of i-card which is requested.
	 * @return the <code>ICard</code> instance.
	 * @throws CardException
	 *             if <code>ICard</code> with <code>CUID</code> couldn't be
	 *             retrieved.
	 */
	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException;

	/**
	 * Gets fully loaded user's <code>ICard</code> whose identity could be
	 * requested using spesified callback handler which have the spesified
	 * unique identifier.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param cuid -
	 *            the unique identifier of i-card which is requested.
	 * @param userCredential -
	 *            the credential object used to authenticate the user on
	 *            endpoint service.
	 * @return the <code>ICard</code> instance.
	 * @throws AuthenticationException
	 *             when an attempt to authenticate the user on endpoint service
	 *             has failed
	 * @throws CardException
	 *             if <code>ICard</code> with <code>CUID</code> couldn't be
	 *             retrieved.
	 */
	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid, ICredential userCredential) throws AuthenticationException, CardException;
	
	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props);

	//public boolean canImport(CallbackHandler authHandler, InputStream in);
	
	/**
	 * Tests whether this provider can import i-card represented by
	 * <code>card</code> parameter.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param card -
	 *            the element which holds i-card data
	 * @return true if <code>card</code> parameter holds valid i-card data and
	 *         this provider could import them as i-card.
	 */
	public boolean canImportICard(CallbackHandler authHandler, IElement card);

	/**
	 * Imports i-card data into backed store used by this provider and returns
	 * i-card object representing imported i-card.
	 * <p>
	 * Card object returned by this method call may or may not have its claims
	 * retieved from endpoint service. When a card doesn't have its claims
	 * retieved ({@link ICard#isClaimsRetrieved()} returns <code>false</code>)
	 * it is necessary to make additional call to
	 * {@link #getICardByCUID(CallbackHandler, CUID, ICredential)} in order to
	 * obtain fully loaded card object.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param card -
	 *            the element which holds i-card data
	 * @return the i-card object which represents imported i-card
	 * @throws CardException
	 */
	public ICard importICard(CallbackHandler authHandler, IElement card) throws CardException;

	/**
	 * Performs export of <code>cards</code> into array of
	 * <code>IElement</code>s. Each element in returned array must conform
	 * with the requirements specified by <code>elementDescriptor</code>
	 * parameter.
	 * 
	 * @param authHandler -
	 *            the callback handler to be used to request user's identity.
	 * @param format -
	 *            the element descriptor which defines the format of element to
	 *            export
	 * @param cards -
	 *            the array of CUIDs of cards to be exported
	 * @throws CardException
	 */
	public IElement[] exportICards(CallbackHandler authHandler, IElementFormat format, CUID[] cards) throws CardException;
	
	/**
	 * @param props
	 * @return
	 */
	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException;

	//public ICard importCard(CallbackHandler authHandler, String fileName) throws CardException;

	//public void exportCards(CallbackHandler authHandler, Iterator cards, OutputStream out) throws CardException;

	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException;

	//
	/*
	 * public Iterable<IProtocol> getSupportedProtocols(); public boolean
	 * supportProtocol(IProtocol protocol); // public IRPToken getToken(ICard
	 * card, IProtocol protocol);
	 */

	/**
	 * @param authHandler
	 * @return <code>ICardTemplate</code> which describe data required by this provider to create a card
	 */
	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler);
	
	/**
	 * @param authHandler
	 * @return <code>ICardTemplate</code> which describe data required by this provider to create a card
	 */
	public ICardTemplate getCardCreationTemplatesByID(CallbackHandler authHandler, String ID);

	/**
	 * @param authHandler
	 * @param template
	 * @return
	 * @throws CardException
	 */
	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException;

}
