/**
 Copyright (c) 2007 Parity Communications, Inc. 
 All rights reserved. This program and the accompanying materials 
 are made available under the terms of the Eclipse Public License v1.0 
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html

 Contributors:
 Alexander Yuhimenko - implementation
 */

package org.eclipse.higgins.ics.ws.client;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.rpc.ServiceException;

import org.apache.axis.EngineConfiguration;
import org.apache.axis.configuration.EngineConfigurationFactoryFinder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.Card;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.CardCredential;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.CardInformation;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.CardTemplate;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.CardUpdate;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.CardUsage;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.Category;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.ErrorKeys;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.ListEntity;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.Response;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.UserData;
import org.eclipse.higgins.ics.gwt.base.client.model.entity.UserProfile;
import org.eclipse.higgins.ics.ws.wsdl.CardInformationTO;
import org.eclipse.higgins.ics.ws.wsdl.CardUsageTO;
import org.eclipse.higgins.ics.ws.wsdl.CategoryTO;
import org.eclipse.higgins.ics.ws.wsdl.FormatDescriptorTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardResponseTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardTemplateTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardUpdateResponseTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardUpdateTO;
import org.eclipse.higgins.ics.ws.wsdl.ICardsAndCategoryTO;
import org.eclipse.higgins.ics.ws.wsdl.MapTO;
import org.eclipse.higgins.ics.ws.wsdl.RPPSServiceLocator;
import org.eclipse.higgins.ics.ws.wsdl.RPPSServiceSEI;
import org.eclipse.higgins.ics.ws.wsdl.ResponseMessage;
import org.eclipse.higgins.ics.ws.wsdl.UITokenServiceCredentialTO;
import org.eclipse.higgins.ics.ws.wsdl.UserProfileTO;

/**
 * @author Alexander Yuhimenko
 * 
 */
public final class WSClient {

	private static final Log log = LogFactory.getLog(WSClient.class);

	private static String RPPSEndpointAddress;

	static {
		RPPSEndpointAddress = System.getProperty("org.eclipse.higgins.ics.ws.RPPSEndpointAddress");
		log.info(RPPSEndpointAddress);
	}

	private RPPSServiceLocator service;

	public WSClient() {
		initService();
	}

	/**
	 * @return
	 * @throws ServiceException
	 */
	private final void initService() {
		EngineConfiguration defConfig = EngineConfigurationFactoryFinder.newFactory().getClientEngineConfig();
		// //SimpleProvider clientConfig = new SimpleProvider();
		// SimpleSessionHandler sessionHandler = new SimpleSessionHandler();
		// sessionHandler.setDefaultSessionTimeout(60 * 30);
		// SimpleChain reqHandler = new SimpleChain();
		// SimpleChain respHandler = new SimpleChain();
		// reqHandler.addHandler(sessionHandler);
		// respHandler.addHandler(sessionHandler);
		// Handler pivot = (Handler) new HTTPSender();
		// Handler transport = new SimpleTargetedChain(reqHandler, pivot, respHandler);
		// //clientConfig.deployTransport(HTTPTransport.DEFAULT_TRANSPORT_NAME, transport);

		this.service = new RPPSServiceLocator();
		this.service.setEngineConfiguration(defConfig);
		// ((org.apache.axis.client.Service) this.service).setEngine(new AxisClient(defConfig));
		this.service.setMaintainSession(true);

		if (RPPSEndpointAddress != null) {
			service.setRPPSServiceSEIPortEndpointAddress(RPPSEndpointAddress);
		}

	}

	/**
	 * @param data
	 * @param userId
	 * @param userPassword
	 * @return
	 */
	/*
	 * public final Response importICards(byte[] data, String userId, String userPassword) { Response response = new
	 * Response(); try { ICardTO[] cardTOs = getServiceImpl().importICardAsByteArray(userId, userPassword, data);
	 * response.setResult(cardTOs); response.setStatusCode(new Integer(0)); } catch (RemoteException e) {
	 * response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException); response.setStatusMessage(e.getMessage()); } catch
	 * (ServiceException e) { response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
	 * response.setStatusMessage(e.getMessage()); } catch (Exception e) {
	 * response.setStatusCode(ErrorKeys.ERROR_CODE_Exception); response.setStatusMessage(e.getMessage()); } return
	 * response; }
	 */

	/**
	 * @param cuid
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response deleteICard(String cuid, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().deleteICard(userIdentifier, userPassword, cuid);
			response.setStatusCode(new Integer(0));
			response.setStatusMessage("Ok");
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userId
	 * @param userPassword
	 * @return Card[]
	 */
	// public final Response getCards(String userId, String userPassword) {
	// Response response = new Response();
	// try {
	// ICardResponseTO cardResponseTO = getServiceImpl().getICards(userId, userPassword);
	// ICardTO[] cardTOs = cardResponseTO.getICardList();
	// if (cardTOs != null) {
	// Card[] cards = new Card[cardTOs.length];
	// for (int i = 0; i < cardTOs.length; i++) {
	// cards[i] = Utils.ICardTO2Card(cardTOs[i], new Card());
	// }
	// response.setResult(cards);
	// } else {
	// response.setResult(new Card[0]);
	// }
	// response.setStatusCode(new Integer(0));
	// } catch (RemoteException re) {
	// response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
	// response.setStatusMessage(re.getMessage());
	// } catch (ServiceException se) {
	// response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
	// response.setStatusMessage(se.getMessage());
	// } catch (Exception e) {
	// response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
	// response.setStatusMessage(e.getMessage());
	// }
	// return response;
	// }
	public final Response getUserData(String userId, String userPassword) {
		Response response = new Response();
		try {
			UserData data = new UserData();
			ICardsAndCategoryTO cardsAndCategoryTO = getServiceImpl().getICardsAndCategoriesByUser(userId, userPassword);
			if (cardsAndCategoryTO != null) {

				MapTO[] relations = null;

				if (cardsAndCategoryTO.getRelationList() != null) {
					relations = cardsAndCategoryTO.getRelationList();
				}

				if (cardsAndCategoryTO.getICardList() != null) {
					ICardTO[] cardTOs = cardsAndCategoryTO.getICardList();
					Card[] cards = new Card[cardTOs.length];
					for (int i = 0; i < cardTOs.length; i++) {
						if (cardTOs[i] != null) {
							cards[i] = Utils.ICardTO2Card(cardTOs[i], new Card());
							if (relations != null) {
								for (int j = 0; j < relations.length; j++) {
									if (cards[i].getCuid().equals(relations[j].getKey())) {
										cards[i].setRelatedCards(relations[j].getValues());
									}
								}
							}
						}
					}
					data.setCards(cards);
				}

				if (cardsAndCategoryTO.getCategoryList() != null && cardsAndCategoryTO.getCategoryList().length > 0) {
					CategoryTO[] categoryTOs = cardsAndCategoryTO.getCategoryList();
					ArrayList categoryList = new ArrayList();
					for (int i = 0; i < categoryTOs.length; i++) {
						if (categoryTOs[i] != null)
							categoryList.add(Utils.categoryTO2category(categoryTOs[i]));
					}
					data.setCategoryList(categoryList);
				}

			}
			response.setResult(data);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userId
	 * @param userPassword
	 * @return
	 */
	public final Response getCardByCredential(Card card, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			ICardTO cardTO = null;
			if (card.getCardCredential() != null && card.getCardCredential().length > 0) {
				cardTO = getServiceImpl().getICardByIdAndTokenServiceCredential(userIdentifier, userPassword, card.getCuid(),
						Utils.cardCredential2UITokenServiceCredentialTO(card.getCardCredential()[0]));
			} else {
				cardTO = getServiceImpl().getICardByIdAndTokenServiceCredential(userIdentifier, userPassword, card.getCuid(), null);
			}
			Card newCard = Utils.ICardTO2Card(cardTO, new Card());
			newCard.setCardCredential(card.getCardCredential());
			newCard.setValueAvailable(true);
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));

		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}
	
	//ACHTUNG HARDCODE     ACHTUNG HARDCODE   ACHTUNG HARDCODE   ACHTUNG HARDCODE   ACHTUNG HARDCODE
	
	
	
	public final Response createPwmCardByTemplate(ICardTemplateTO template, String userId, String userPassword) {
		Response response = new Response();
		
		for (int i=0; i<template.getElements().length; i++) {
			template.getElements()[i].setValue("password");
			if (template.getElements()[i].getId().equals("cardPicture"))
				template.getElements()[i].setValue("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");
		}
		
		try {
			ICardTO cardTO = getServiceImpl().createICardByTemplate(userId, userPassword, template);
			cardTO.getType();
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		
		return response;
	}
	
	
	
	
	//ACHTUNG HARDCODE   ACHTUNG HARDCODE   ACHTUNG HARDCODE   ACHTUNG HARDCODE   ACHTUNG HARDCODE

	/**
	 * @param userId
	 * @param userPassword
	 * @return
	 */
	public final Response getICardTemplate(String userId, String userPassword) {
		Response response = new Response();
		try {
			ICardTemplateTO[] cardTemplateTOs = getServiceImpl().getICardCreationTemplate(userId, userPassword);
			CardTemplate cardTemplate = null;
			for (int i=0; i<cardTemplateTOs.length; i++) {
				if (cardTemplateTOs[i].getCardType().equals("org.eclipse.higgins.icard.IPersonalInformationCard")) 
					cardTemplate = Utils.cardTemplateTO2cardTemplate(cardTemplateTOs[i]);
				/*
				if (cardTemplateTOs[i].getCardType().equals("org.eclipse.higgins.icard.pwm.IPwmCard")) 
					createPwmCardByTemplate(cardTemplateTOs[i], userId, userPassword);*/
				
			}
			if (cardTemplate == null) 
				cardTemplate = Utils.cardTemplateTO2cardTemplate(cardTemplateTOs[0]);
			response.setResult(cardTemplate);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param cardTemplateTO
	 * @param userId
	 * @param userPassword
	 * @return
	 */
	public final Response createICard(CardTemplate cardTemplate, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			ICardTO cardTO = getServiceImpl().createICardByTemplate(userIdentifier, userPassword,
					Utils.cardTemplate2cardTemplateTO(cardTemplate));
			response.setResult(Utils.ICardTO2Card(cardTO, new Card()));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}

		return response;
	}

	/**
	 * @param userProfile
	 * @return
	 */
	public final Response addUserProfile(UserProfile userProfile) {
		Response response = new Response();
		try {
			ResponseMessage responseMessage = getServiceImpl().addUserProfile(userProfile.getLoginName(), userProfile.getPassword(),
					Utils.UserProfile2UserProfileTO(userProfile));
			response.setResult(userProfile);
			response.setStatusCode(new Integer(responseMessage.getCode() == 0 ? 0 : 1000));
			response.setStatusMessage("0");
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response getUserProfile(String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			UserProfileTO userProfile = getServiceImpl().getUserProfile(userIdentifier, userPassword);
			UserProfile user = Utils.UserProfileTO2UserProfile(userProfile);
			user.setLoginName(userIdentifier);
			user.setPassword(userPassword);
			response.setResult(user);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userProfile
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response modifyUserProfile(UserProfile userProfile, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			UserProfileTO userProfileTO = getServiceImpl().modifyUserProfile(userProfile.getLoginName(), userProfile.getPassword(),
					Utils.UserProfile2UserProfileTO(userProfile));
			response.setStatusMessage("0");
			response.setStatusCode(new Integer(0));
			response.setResult(Utils.UserProfileTO2UserProfile(userProfileTO));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userIdentifier
	 * @param userpassword
	 * @return
	 */
	public final Response deleteUserProfile(String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().deleteUserProfile(userIdentifier, userPassword);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());

		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response modifyICard(Card card, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			Card newCard = Utils.ICardTO2Card(getServiceImpl().updateICard(userIdentifier, userPassword,
					Utils.card2ICardTO(card, new ICardTO())), new Card());
			newCard.setCardCredential(card.getCardCredential());
			if (newCard.isProtected()) {
				newCard.setClaimItems(card.getClaimItems());
			}
			newCard.setValueAvailable(true);
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @return
	 * @throws ServiceException
	 */
	public final RPPSServiceSEI getServiceImpl() throws ServiceException {
		return service.getRPPSServiceSEIPort();
	}

	public final Response getCardById(String cuid, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			Card newCard = Utils.ICardTO2Card(getServiceImpl().getICardById(userIdentifier, userPassword, cuid), new Card());
			// newCard.setProtected(true);
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	//
	// public static final RPPSServiceSEI getServiceImpl() throws ServiceException {
	// RPPSServiceLocator service = new RPPSServiceLocator();
	// if (RPPSEndpointAddress != null) {
	// service.setRPPSServiceSEIPortEndpointAddress(RPPSEndpointAddress);
	// }
	//
	// return service.getRPPSServiceSEIPort();
	// }

	/**
	 * @param cardId
	 * @param pin
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response setPinCode(String cuid, String pin, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().setPinCode(userIdentifier, userPassword, cuid, pin);
			Card newCard = Utils.ICardTO2Card(getServiceImpl().getICardById(userIdentifier, userPassword, cuid), new Card());
			// newCard.setProtected(true);
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode) {
		Response response = new Response();
		try {
			Card newCard = Utils.ICardTO2Card(getServiceImpl().editPinCode(userId, password, cuid, oldPinCode, newPinCode), new Card());
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param cardId
	 * @param pin
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response resetPinCode(String cuid, String pin, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().resetPinCode(userIdentifier, userPassword, cuid, pin);
			Card newCard = Utils.ICardTO2Card(getServiceImpl().getICardById(userIdentifier, userPassword, cuid), new Card());
			// newCard.setProtected(true);
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param cardId
	 * @param cardName
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response createDuplicateICard(String cuid, String cardName, String pinCode, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			Card newCard = Utils.ICardTO2Card(getServiceImpl().createDuplicateICard(userIdentifier, userPassword, cardName, cuid, pinCode),
					new Card());
			response.setResult(newCard);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param category
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response createCategory(Category category, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			response.setResult(Utils.categoryTO2category(getServiceImpl().addCategory(userIdentifier, userPassword,
					Utils.category2categoryTO(category))));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param category
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response modifyCategory(Category category, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			response.setResult(Utils.categoryTO2category(getServiceImpl().modifyCategory(userIdentifier, userPassword,
					Utils.category2categoryTO(category))));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param categoryId
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response deleteCategory(String categoryId, String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().deleteCategory(userIdentifier, userPassword, categoryId);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @param userIdentifier
	 * @param userPassword
	 * @return
	 */
	public final Response deleteAllCard(String userIdentifier, String userPassword) {
		Response response = new Response();
		try {
			getServiceImpl().deleteAllICard(userIdentifier, userPassword);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response importICards(String userIdentifier, String userPassword, byte[] fileData) {
		Response response = new Response();
		try {
			ICardResponseTO responseTO = getServiceImpl().importICardAsByteArray(userIdentifier, userPassword, fileData);
			ICardTO[] icards = responseTO.getICardList();
			Card[] cards = new Card[icards.length];
			for (int i = 0; i < icards.length; i++) {
				cards[i] = Utils.ICardTO2Card(icards[i], new Card());
			}
			response.setResult(cards);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response getICardsFromRestoreAndFormatDescriptor(String userIdentifier, String userPassword, String formatID,
			byte[] fileData, UITokenServiceCredentialTO uitscTO) {
		Response response = new Response();
		try {
			ICardResponseTO resp = getServiceImpl().importICards(userIdentifier, userPassword, fileData, formatID, uitscTO);
			ICardTO[] icards = resp.getICardList();
			ArrayList cards = new ArrayList(icards.length);
			ListEntity list = new ListEntity();
			for (int i = 0; i < icards.length; i++) {
				cards.add(Utils.ICardTO2Card(icards[i], new Card()));
			}
			list.setList(cards);
			response.setResult(list);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}

		return response;
	}

	public final Response getCardInformation(String cuid, String userIdentifier, String userPassword, int size) {
		Response response = new Response();
		try {
			CardInformationTO ciTO = getServiceImpl().getCardInformation(userIdentifier, userPassword, cuid, size);
			response.setResult(Utils.cardInformationTO2CardInformation(ciTO, new CardInformation()));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response clearCardCredential(String userId, String password, String cuid)

	{
		Response response = new Response();
		try {
			ICardTO ciTO = getServiceImpl().clearCardCredential(userId, password, cuid);
			response.setResult(Utils.ICardTO2Card(ciTO, new Card()));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response clearCardHistory(String userId, String password, String cuid) {
		Response response = new Response();
		try {
			boolean result = getServiceImpl().clearCardHistory(userId, password, cuid);
			response.setStatusCode(result ? new Integer(0) : ErrorKeys.ERROR_CODE_ServiceException);
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response setUrlsCardRelation(String userId, String password, String cuid, String[] urls) {
		Response response = new Response();
		try {
			/*
			 * CardInformationTO ciTO = getServiceImpl().setUrlsCardRelation(userId, password, cuid, urls);
			 * response.setResult(Utils.CardInformationTO2CardInformation(ciTO, new CardInformation()));
			 * response.setStatusCode(new Integer(0)); } catch (RemoteException e) {
			 * response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException); response.setStatusMessage(e.getMessage()); }
			 * catch (ServiceException e) { response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			 * response.setStatusMessage(e.getMessage());
			 */
		} catch (Exception e) {
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response deleteUrlsCardRelation(String userId, String password, String cuid, String[] urls) {
		Response response = new Response();
		try {
			boolean ciTO = getServiceImpl().deleteUrlCardRelation(userId, password, cuid, urls[0]);
			// response.setResult(Utils.cardInformationTO2CardInformation(ciTO, new CardInformation()));
			if (ciTO)
				response.setStatusCode(new Integer(0));
			else
				response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response modifyCategories(String userId, String password, Category[] categories) {
		Response response = new Response();
		CategoryTO[] categoriesTO = new CategoryTO[categories.length];
		for (int i = 0; i < categories.length; i++)
			categoriesTO[i] = Utils.category2categoryTO(categories[i]);

		try {
			ResponseMessage responseMessage = getServiceImpl().modifyCategories(userId, password, categoriesTO);
			response.setStatusMessage(responseMessage.getMessage());
			response.setStatusCode(new Integer(responseMessage.getCode() == 0 ? 0 : 1000));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response checkImportStream(String userId, String password, byte[] crd) {
		Response response = new Response();
		try {
			FormatDescriptorTO fdTO = getServiceImpl().checkImportStream(userId, password, crd);
			response.setResult(Utils.formatDescriptorTO2FormatDescriptor(fdTO));
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response setCardCredential(String userId, String password, String cuid, CardCredential tokenCredential) {
		Response response = new Response();
		try {
			boolean bResponse = getServiceImpl().setCardCredential(userId, password, cuid,
					Utils.cardCredential2UITokenServiceCredentialTO(tokenCredential));
			if (bResponse)
				response.setStatusCode(new Integer(0));
			else
				response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response getCardHistory(String userId, String password, String cuid, int startFrom, int capacity, String orderBy,
			String orderDirection) {
		Response response = new Response();
		try {
			CardUsageTO[] cuTOs = getServiceImpl().getCardHistory(userId, password, cuid, startFrom, capacity, orderBy, orderDirection);
			ArrayList cus = new ArrayList(0);
			ListEntity list = new ListEntity();
			for (int i = 0; i < cuTOs.length; i++) {
				cus.add(Utils.cardUsageTO2CardUsage(cuTOs[i], new CardUsage()));
			}
			list.setList(cus);
			response.setResult(list);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response getOutputFormats(String userId, String password) {
		Response response = new Response();
		try {
			FormatDescriptorTO[] fdTOs = getServiceImpl().getOutputFormats(userId, password);
			ArrayList fds = new ArrayList(0);
			ListEntity list = new ListEntity();
			for (int i = 0; i < fdTOs.length; i++) {
				fds.add(Utils.formatDescriptorTO2FormatDescriptor(fdTOs[i]));
			}
			list.setList(fds);
			response.setResult(list);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response getICardsByFormat(String userId, String password, String formatID) {
		Response response = new Response();
		try {
			ICardTO[] cardTOs = getServiceImpl().getICardsByFormat(userId, password, formatID);
			ArrayList cards = new ArrayList(0);
			ListEntity list = new ListEntity();
			for (int i = 0; i < cardTOs.length; i++) {
				cards.add(Utils.ICardTO2Card(cardTOs[i], new Card()));
			}
			list.setList(cards);
			response.setResult(list);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response exportICards(String userId, String password, String formatID, String[] cards, CardCredential credential) {
		Response response = new Response();
		try {
			byte[] file = getServiceImpl().exportICards(userId, password, formatID, cards,
					Utils.cardCredential2UITokenServiceCredentialTO(credential));
			response.setResult(file);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	public final Response getMRU(String userId, String password) {
		Response response = new Response();
		try {
			MapTO[] mrutos = getServiceImpl().getClaimValuesMRU(userId, password);
			List mrus = new ArrayList();
			for (int i = 0; i < mrutos.length; i++) {
				mrus.add(Utils.MRUTO2MRU(mrutos[i]));
			}
			ListEntity entity = new ListEntity();
			entity.setList(mrus);
			response.setResult(entity);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;

	}

	public final Response updateClaimAcrossCards(String userId, String password, String claimType, String claimValue,
			CardUpdate[] cardUpdate) {
		Response response = new Response();
		try {
			ICardUpdateTO[] to = new ICardUpdateTO[cardUpdate.length];
			for (int i = 0; i < cardUpdate.length; i++) {
				to[i] = Utils.cardUpdate2ICardUpdateTO(cardUpdate[i]);
			}
			ICardUpdateResponseTO[] responseTO = getServiceImpl().updateClaimAcrossCards(userId, password, claimType, claimValue, to);
			List cuids = new ArrayList();
			for (int i = 0; i < responseTO.length; i++) {
				if (responseTO[i].getCode() == 0)
					cuids.add(responseTO[i].getCuid());
			}
			ListEntity entity = new ListEntity();
			entity.setList(cuids);
			response.setResult(entity);
			response.setStatusCode(new Integer(0));
		} catch (RemoteException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_RemoteException);
			response.setStatusMessage(e.getMessage());
		} catch (ServiceException e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_ServiceException);
			response.setStatusMessage(e.getMessage());
		} catch (Exception e) {
			log.error(e, e);
			response.setStatusCode(ErrorKeys.ERROR_CODE_Exception);
			response.setStatusMessage(e.getMessage());
		}
		return response;
	}

	/**
	 * @return the service
	 */
	public RPPSServiceLocator getService() {
		return this.service;
	}

	/**
	 * @param service
	 *            the service to set
	 */
	public void setService(RPPSServiceLocator service) {
		this.service = service;
	}
}
