package org.eclipse.higgins.ics.gwt.base.client.util;

import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.user.client.rpc.IsSerializable;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.MultiWordSuggestOracle;
import com.google.gwt.user.client.ui.SuggestOracle;

public final class HigginsSuggestionOracle extends SuggestOracle {
	
	static class PrefixTree extends AbstractCollection {

		  /**
		   * Iterates over the structure of a PrefixTree. No concurrency checks are
		   * made. This Iterator will output anything new added to the tree if the new
		   * entry is after the current position of the Iterator.
		   *
		   * This Iterator implementation is iterative and uses an internal stack object
		   * to avoid call-stack limitations and invocation overhead.
		   */
		  private static class PrefixTreeIterator implements Iterator {

		    private JavaScriptObject stack;

		    /**
		     * Constructor.
		     *
		     * @param tree The base of the PrefixTree to iterate over
		     */
		    public PrefixTreeIterator(PrefixTree tree) {
		      init();
		      addTree(tree, "");
		    }

		    public boolean hasNext() {
		      // Have nextImpl peek at the next value that would be returned.
		      return nextImpl(true) != null;
		    }

		    /**
		     * {@inheritDoc} Wraps the native implementation with some sanity checking.
		     */
		    public Object next() {
		      final Object toReturn = nextImpl(false);

		      // A null response indicates that there was no data to be had.
		      if (toReturn == null) {
		        // Sanity check.
		        if (!hasNext()) {
		          throw new NoSuchElementException("No more elements in the iterator");
		        } else {
		          throw new RuntimeException(
		              "nextImpl() returned null, but hasNext says otherwise");
		        }
		      }

		      return toReturn;
		    }

		    public void remove() {
		      throw new UnsupportedOperationException("PrefixTree does not support "
		          + "removal.  Use clear()");
		    }

		    /**
		     * Add a frame to the work stack.
		     *
		     * <pre>
		     *  frame := {suffixNames, subtrees, prefix, index}
		     *  suffixNames := All suffixes in the target PrefixTree
		     *  subtrees := All subtrees in the target PrefixTree
		     *  prefix := A string that next() will prepend to output from the frame
		     *  index := Stores which suffix was last output
		     * </pre>
		     *
		     * @param tree The tree to add
		     * @param prefix The prefix to prepend to values in tree
		     */
		    private native void addTree(PrefixTree tree, String prefix) /*-{
		      var suffixes = [];
		      for (suffix in tree.@com.google.gwt.user.client.ui.PrefixTree::suffixes) {
		        suffixes.push(suffix);
		      }

		      var frame = {
		        suffixNames: suffixes,
		        subtrees: tree.@com.google.gwt.user.client.ui.PrefixTree::subtrees,
		        prefix: prefix,
		        index: 0
		      };

		      var stack = this.@com.google.gwt.user.client.ui.PrefixTree$PrefixTreeIterator::stack;
		      stack.push(frame);
		    }-*/;

		    /**
		     * Initialize JSNI objects.
		     */
		    private native void init() /*-{
		      this.@com.google.gwt.user.client.ui.PrefixTree$PrefixTreeIterator::stack = [];
		    }-*/;

		    /**
		     * Access JSNI structures.
		     *
		     * @param peek If this is true, don't advance the iteration, just return the
		     *          value that next() would return if it were called
		     * @return The next object, or null if there is an error
		     */
		    private native Object nextImpl(boolean peek) /*-{
		      var stack = this.@com.google.gwt.user.client.ui.PrefixTree$PrefixTreeIterator::stack;
		      var safe = @com.google.gwt.user.client.ui.PrefixTree::safe(Ljava/lang/String;)
		      var unsafe = @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)

		      // Put this in a while loop to handle descent into subtrees without recursion.
		      while (stack.length > 0) {
		        var frame = stack.pop();

		        // Check to see if there are any remaining suffixes to output.
		        if (frame.index < frame.suffixNames.length) {
		          var toReturn = frame.prefix + unsafe(frame.suffixNames[frame.index]);

		          if (!peek) {
		            frame.index++;
		          }

		          // If the current frame has more suffixes, retain it on the stack.
		          if (frame.index < frame.suffixNames.length) {
		            stack.push(frame);

		            // Otherwise, put all of the subtrees on the stack.
		          } else {
		            for (key in frame.subtrees) {
		              var target = frame.prefix + unsafe(key);
		              var subtree = frame.subtrees[key];
		              this.@com.google.gwt.user.client.ui.PrefixTree$PrefixTreeIterator::addTree(Lcom/google/gwt/user/client/ui/PrefixTree;Ljava/lang/String;)(subtree, target);
		            }
		          }

		          return toReturn;

		       // Put all subframes on the stack, and return to top of loop.
		       } else {
		         for (key in frame.subtrees) {
		           var target = frame.prefix + unsafe(key);
		           var subtree = frame.subtrees[key];

		           this.@com.google.gwt.user.client.ui.PrefixTree$PrefixTreeIterator::addTree(Lcom/google/gwt/user/client/ui/PrefixTree;Ljava/lang/String;)(subtree, target);
		         }
		       }
		     }

		     // This would indicate that next() was called on an empty iterator.
		     // Will throw an exception from next().
		     return null;
		    }-*/;
		  }

		  /**
		   * Used by native methods to create an appropriately blessed PrefixTree.
		   * 
		   * @param prefixLength Smaller prefix length equals faster, more direct
		   *          searches, at a cost of setup time
		   * @return a newly constructed prefix tree
		   */
		  protected static PrefixTree createPrefixTree(int prefixLength) {
		    return new PrefixTree(prefixLength);
		  }

		  /**
		   *  Ensure that a String can be safely used as a key to an associative-array
		   *  JavaScript object by prepending a prefix.
		   *  
		   *  @param s The String to make safe
		   *  @return A safe version of <code>s</code>
		   */
		  private static String safe(String s) {
		    return ':' + s;
		  }

		  /**
		   *  Undo the operation performed by safe().
		   *  
		   *  @param s A String returned from safe()
		   *  @return The original String passed into safe()
		   */
		  private static String unsafe(String s) {
		    return s.substring(1);
		  }

		  /**
		   * Stores the requested prefix length.
		   */
		  protected final int prefixLength;

		  /**
		   * Field to store terminal nodes in.
		   */
		  protected JavaScriptObject suffixes;

		  /**
		   * Field to store subtrees in.
		   */
		  protected JavaScriptObject subtrees;

		  /**
		   * Store the number of elements contained by this PrefixTree and its
		   * sub-trees.
		   */
		  protected int size = 0;

		  /**
		   * Constructor.
		   */
		  public PrefixTree() {
		    this(2, null);
		  }

		  /**
		   * Constructor.
		   *
		   * @param source Initialize from another collection
		   */
		  public PrefixTree(Collection source) {
		    this(2, source);
		  }

		  /**
		   * Constructor.
		   *
		   * @param prefixLength Smaller prefix length equals faster, more direct
		   *          searches, at a cost of setup time.
		   */
		  public PrefixTree(final int prefixLength) {
		    this(prefixLength, null);
		  }

		  /**
		   * Constructor.
		   *
		   * @param prefixLength Smaller prefix length equals faster, more direct
		   *          searches, at a cost of setup time.
		   * @param source Initialize from another collection
		   */
		  public PrefixTree(final int prefixLength, final Collection source) {
		    this.prefixLength = prefixLength;
		    clear();

		    if (source != null) {
		      addAll(source);
		    }
		  }

		  /**
		   * Adds an object to the PrefixTree.
		   *
		   * @param o The object to add
		   * @throws UnsupportedOperationException if the object is not a String
		   *  @return <code>true</code> if the string was added, <code>false</code>
		   *          otherwise
		   */
		  public boolean add(Object o) throws UnsupportedOperationException {
		    if (o instanceof String) {
		      return add((String) o);
		    } else {
		      throw new UnsupportedOperationException(
		          "Cannot add non-Strings to PrefixTree");
		    }
		  }

		  /**
		   * Add a String to the PrefixTree.
		   *
		   * @param s The data to add
		   * @return <code>true</code> if the string was added, <code>false</code>
		   *         otherwise
		   */
		  public native boolean add(String s) /*-{
		    var suffixes =
		        this.@com.google.gwt.user.client.ui.PrefixTree::suffixes;
		    var subtrees =
		        this.@com.google.gwt.user.client.ui.PrefixTree::subtrees;
		    var prefixLength =
		        this.@com.google.gwt.user.client.ui.PrefixTree::prefixLength;

		    // This would indicate a mis-use of the code.
		    if ((s == null) || (s.length == 0)) {
		      return false;
		    }

		    // Use <= so that strings that are exactly prefixLength long don't
		    // require some kind of null token.
		    if (s.length <= prefixLength) {
		      var safeKey = @com.google.gwt.user.client.ui.PrefixTree::safe(Ljava/lang/String;)(s);
		      if (suffixes.hasOwnProperty(safeKey)) {
		        return false;
		      } else {
		        // Each tree keeps a count of how large it and its children are.
		        this.@com.google.gwt.user.client.ui.PrefixTree::size++;

		        suffixes[safeKey] = true;
		        return true;
		      }

		    // Add the string to the appropriate PrefixTree.
		    } else {
		      var prefix = @com.google.gwt.user.client.ui.PrefixTree::safe(Ljava/lang/String;)(s.slice(0, prefixLength));
		      var theTree;

		      if (subtrees.hasOwnProperty(prefix)) {
		        theTree = subtrees[prefix];
		      } else {
		        theTree = @com.google.gwt.user.client.ui.PrefixTree::createPrefixTree(I)(prefixLength * 2);
		        subtrees[prefix] = theTree;
		      }

		      var slice = s.slice(prefixLength);
		      if (theTree.@com.google.gwt.user.client.ui.PrefixTree::add(Ljava/lang/String;)(slice)) {
		        // The size of the subtree increased, so we need to update the local count.
		        this.@com.google.gwt.user.client.ui.PrefixTree::size++;
		        return true;
		      } else {
		        return false;
		      }
		    }
		  }-*/;

		  /**
		   * Initialize native state.
		   */
		  public native void clear() /*-{
		    this.@com.google.gwt.user.client.ui.PrefixTree::size = 0;
		    this.@com.google.gwt.user.client.ui.PrefixTree::subtrees = {};
		    this.@com.google.gwt.user.client.ui.PrefixTree::suffixes = {};
		  }-*/;

		  public boolean contains(Object o) {
		    if (o instanceof String) {
		      return contains((String) o);
		    } else {
		      return false;
		    }
		  }

		  public boolean contains(String s) {
		    return (getSuggestions(s, 1)).contains(s);
		  }

		  /**
		   * Retrieve suggestions from the PrefixTree. The number of items returned from
		   * getSuggesstions may slightly exceed <code>limit</code> so that all
		   * suffixes and partial stems will be returned. This prevents the search space
		   * from changing size if the PrefixTree is used in an interactive manner.
		   * <br/> The returned List is guaranteed to be safe; changing its contents
		   * will not affect the PrefixTree.
		   *
		   * @param search The prefix to search for
		   * @param limit The desired number of results to retrieve
		   * @return A List of suggestions
		   */
		  public List/* <String> */getSuggestions(String search, int limit) {
		    final List toReturn = new ArrayList();
		    if ((search != null) && (limit > 0)) {
		      suggestImpl(search, "", toReturn, limit);
		    }
		    return toReturn;
		  }

		  public Iterator iterator() {
		    return new PrefixTreeIterator(this);
		  }

		  /**
		   * Get the number of all elements contained within the PrefixTree.
		   * 
		   * @return the size of the prefix tree
		   */
		  public int size() {
		    return size;
		  }

		  protected native void suggestImpl(String search, String prefix,
		      Collection output, int limit) /*-{
		    var suffixes =
		        this.@com.google.gwt.user.client.ui.PrefixTree::suffixes;
		    var subtrees =
		        this.@com.google.gwt.user.client.ui.PrefixTree::subtrees;
		    var prefixLength =
		        this.@com.google.gwt.user.client.ui.PrefixTree::prefixLength;

		    // Search is too big to be found in current tree, just recurse.
		    if (search.length > prefix.length + prefixLength) {
		      var key = @com.google.gwt.user.client.ui.PrefixTree::safe(Ljava/lang/String;)(search.slice(prefix.length, prefix.length + prefixLength));

		      // Just pick the correct subtree, if it exists, and call suggestImpl.
		      if (subtrees.hasOwnProperty(key)) {
		        var subtree = subtrees[key];
		        var target = prefix + @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)(key);
		        subtree.@com.google.gwt.user.client.ui.PrefixTree::suggestImpl(Ljava/lang/String;Ljava/lang/String;Ljava/util/Collection;I)(search, target, output, limit);
		      }

		    // The answer can only exist in this tree's suffixes or subtree keys.
		    } else {
		     // Check local suffixes.
		     for (suffix in suffixes) {
		       var target = prefix + @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)(suffix);
		       if (target.indexOf(search) == 0) {
		         output.@java.util.Collection::add(Ljava/lang/Object;)(target);
		       }

		       if (output.@java.util.Collection::size()() >= limit) {
		         return;
		       }
		     }

		     // Check the subtree keys.  If the key matches, that implies that all
		     // elements of the subtree would match.
		     for (var key in subtrees) {
		       var target = prefix + @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)(key);
		       var subtree = subtrees[key];

		       // See if the prefix gives us a match.
		       if (target.indexOf(search) == 0) {

		         // Provide as many suggestions as will fit into the remaining limit.
		         // If there is only one suggestion, include it.
		         if ((subtree.@com.google.gwt.user.client.ui.PrefixTree::size <= limit - output.@java.util.Collection::size()()) ||
		             (subtree.@com.google.gwt.user.client.ui.PrefixTree::size == 1)) {
		           subtree.@com.google.gwt.user.client.ui.PrefixTree::dump(Ljava/util/Collection;Ljava/lang/String;)(output, target);

		         // Otherwise, include as many answers as we can by truncating the suffix
		         } else {

		           // Always fully-specify suffixes.
		           for (var suffix in subtree.@com.google.gwt.user.client.ui.PrefixTree::suffixes) {
		             output.@java.util.Collection::add(Ljava/lang/Object;)(target + @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)(suffix));
		           }

		           // Give the keys of the subtree.
		           for (var subkey in subtree.@com.google.gwt.user.client.ui.PrefixTree::subtrees) {
		             output.@java.util.Collection::add(Ljava/lang/Object;)(target + @com.google.gwt.user.client.ui.PrefixTree::unsafe(Ljava/lang/String;)(subkey) + "...");
		           }
		         }
		       }
		     }
		   }
		  }-*/;

		  /**
		   * Put all contents of the PrefixTree into a Collection.
		   * 
		   * @param output the collection into which the prefixes will be dumped
		   * @param prefix the prefix to filter with
		   */
		  private void dump(Collection output, String prefix) {
		    for (final Iterator i = iterator(); i.hasNext();) {
		      output.add(prefix + (String) i.next());
		    }
		  }
		}

  /**
   * Suggestion class for {@link MultiWordSuggestOracle}.
   */
  public static class MultiWordSuggestion implements Suggestion, IsSerializable {
    private String displayString;
    private String replacementString;

    /**
     * Constructor used by RPC.
     */
    public MultiWordSuggestion() {
    }
    
    /**
     * Constructor for <code>MultiWordSuggestion</code>.
     * 
     * @param replacementString the string to enter into the SuggestBox's
     *                          text box if the suggestion is chosen
     * @param displayString the display string
     */
    public MultiWordSuggestion(String replacementString, String displayString) {
      this.replacementString = replacementString;
      this.displayString = displayString;
    }

    public String getDisplayString() {
      return displayString;
    }

    public String getReplacementString() {
      return replacementString;
    }
  }

  private static final char WHITESPACE_CHAR = ' ';
  private static final String WHITESPACE_STRING = " ";

  /**
   * Regular expression used to collapse all whitespace in a query string.
   */
  private static final String NORMALIZE_TO_SINGLE_WHITE_SPACE = "\\s+";

  private static HTML convertMe = new HTML();

  /**
   * Associates substrings with words.
   */
  private final PrefixTree tree = new PrefixTree();

  /**
   * Associates individual words with candidates.
   */
  private HashMap toCandidates = new HashMap();

  /**
   * Associates candidates with their formatted suggestions.
   */
  private HashMap toRealSuggestions = new HashMap();

  /**
   * The whitespace masks used to prevent matching and replacing of the given
   * substrings.
   */
  private char[] whitespaceChars;
  
  private List listCandidates = new ArrayList();

  /**
   * Constructor for <code>MultiWordSuggestOracle</code>. This uses a space as
   * the whitespace character.
   * 
   * @see #MultiWordSuggestOracle(String)
   */
  public HigginsSuggestionOracle() {
    this(" ");
  }

  /**
   * Constructor for <code>MultiWordSuggestOracle</code> which takes in a set
   * of whitespace chars that filter its input.
   * <p>
   * Example: If <code>".,"</code> is passed in as whitespace, then the string
   * "foo.bar" would match the queries "foo", "bar", "foo.bar", "foo...bar", and
   * "foo, bar". If the empty string is used, then all characters are used in
   * matching. For example, the query "bar" would match "bar", but not "foo
   * bar".
   * </p>
   * 
   * @param whitespaceChars the characters to treat as word separators
   */
  public HigginsSuggestionOracle(String whitespaceChars) {
    this.whitespaceChars = new char[whitespaceChars.length()];
    for (int i = 0; i < whitespaceChars.length(); i++) {
      this.whitespaceChars[i] = whitespaceChars.charAt(i);
    }
  }

  /**
   * Adds a suggestion to the oracle. Each suggestion must be plain text.
   * 
   * @param suggestion the suggestion
   */
  public void add(String suggestion) {
	  listCandidates.add(suggestion);
	 String candidate = normalizeSuggestion(suggestion);
    // candidates --> real suggestions.
    toRealSuggestions.put(candidate, suggestion);

    // word fragments --> candidates.
    String[] words = candidate.split(WHITESPACE_STRING);
    for (int i = 0; i < words.length; i++) {
      String word = words[i];
      tree.add(word);
      HashSet l = (HashSet) toCandidates.get(word);
      if (l == null) {
        l = new HashSet();
        toCandidates.put(word, l);
      }
      l.add(candidate);
    }
  }

  /**
   * Adds all suggestions specified. Each suggestion must be plain text.
   * 
   * @param collection the collection
   */
  public void addAll(Collection collection) {
	  listCandidates.addAll(collection);
	 Iterator suggestions = collection.iterator();
    while (suggestions.hasNext()) {
      add((String) suggestions.next());
    }
  }

  /**
   * Removes all of the suggestions from the oracle.
   */
  public void clear() {
    tree.clear();
    toCandidates.clear();
    toRealSuggestions.clear();
  }

  public boolean isDisplayStringHTML() {
    return true;
  }

  public void requestSuggestions(Request request, Callback callback) {
    final List suggestions = computeItemsFor(request.getQuery(), request
      .getLimit());
    Response response = new Response(suggestions);
    callback.onSuggestionsReady(request, response);
  }

  String escapeText(String escapeMe) {
    convertMe.setText(escapeMe);
    String escaped = convertMe.getHTML();
    return escaped;
  }

  /**
   * Compute the suggestions that are matches for a given query.
   * 
   * @param query search string
   * @param limit limit
   * @return matching suggestions
   */
  private List computeItemsFor(String query, int limit) {
    query = normalizeSearch(query);

    // Get candidates from search words.
    List candidates = createCandidatesFromSearch(query, limit);

    // Convert candidates to suggestions.
    return convertToFormattedSuggestions(query, candidates);
  }

  /**
   * Returns real suggestions with the given query in <code>strong</code> html
   * font.
   * 
   * @param query query string
   * @param candidates candidates
   * @return real suggestions
   */
  private List convertToFormattedSuggestions(String query, List candidates) {
    List suggestions = new ArrayList();

    for (int i = 0; i < candidates.size(); i++) {
      String candidate = (String) candidates.get(i);
      int index = 0;
      int cursor = 0;
      // Use real suggestion for assembly.
      String formattedSuggestion = (String) toRealSuggestions.get(candidate);

      // Create strong search string.
      StringBuffer accum = new StringBuffer();

      while (true) {
        index = candidate.indexOf(query, index);
        if (index == -1) {
          break;
        }
        int endIndex = index + query.length();
        if (index == 0 || (WHITESPACE_CHAR == candidate.charAt(index - 1))) {
          String part1 = escapeText(formattedSuggestion
            .substring(cursor, index));
          String part2 = escapeText(formattedSuggestion.substring(index,
            endIndex));
          cursor = endIndex;
          accum.append(part1).append("<strong>").append(part2).append(
            "</strong>");
        }
        index = endIndex;
      }

      // Check to make sure the search was found in the string.
      if (cursor == 0) {
        continue;
      }

      // Finish creating the formatted string.
      String end = escapeText(formattedSuggestion.substring(cursor));
      accum.append(end);
      MultiWordSuggestion suggestion = new MultiWordSuggestion(
        formattedSuggestion, accum.toString());
      suggestions.add(suggestion);
    }
    return suggestions;
  }

  /**
   * Find the sorted list of candidates that are matches for the given query.
   */
  private List createCandidatesFromSearch(String query, int limit) {
    ArrayList candidates = new ArrayList();

    if (query.length() == 0) {
      return listCandidates;
    }

    // Find all words to search for.
    String[] searchWords = query.split(WHITESPACE_STRING);
    HashSet candidateSet = null;
    for (int i = 0; i < searchWords.length; i++) {
      String word = searchWords[i];

      // Eliminate bogus word choices.
      if (word.length() == 0 || word.matches(WHITESPACE_STRING)) {
        continue;
      }

      // Find the set of candidates that are associated with all the
      // searchWords.
      HashSet thisWordChoices = createCandidatesFromWord(word);
      if (candidateSet == null) {
        candidateSet = thisWordChoices;
      } else {
        candidateSet.retainAll(thisWordChoices);

        if (candidateSet.size() < 2) {
          // If there is only one candidate, on average it is cheaper to
          // check if that candidate contains our search string than to
          // continue intersecting suggestion sets.
          break;
        }
      }
    }
    if (candidateSet != null) {
      candidates.addAll(candidateSet);
      Collections.sort(candidates);
      // Respect limit for number of choices.
      for (int i = candidates.size() - 1; i > limit; i--) {
        candidates.remove(i);
      }
    }
    return candidates;
  }

  /**
   * Creates a set of potential candidates that match the given query.
   *
   * @param query query string
   * @return possible candidates
   */
  private HashSet createCandidatesFromWord(String query) {
    HashSet candidateSet = new HashSet();
    List words = tree.getSuggestions(query, Integer.MAX_VALUE);
    if (words != null) {
      // Find all candidates that contain the given word the search is a
      // subset of.
      for (int i = 0; i < words.size(); i++) {
        Collection belongsTo = (Collection) toCandidates.get(words.get(i));
        if (belongsTo != null) {
          candidateSet.addAll(belongsTo);
        }
      }
    }
    return candidateSet;
  }

  /**
   * Normalize the search key by making it lower case, removing multiple spaces,
   * apply whitespace masks, and make it lower case.
   */
  private String normalizeSearch(String search) {
    // Use the same whitespace masks and case normalization for the search
    // string as was used with the candidate values.
    search = normalizeSuggestion(search);

    // Remove all excess whitespace from the search string.
    search = search.replaceAll(NORMALIZE_TO_SINGLE_WHITE_SPACE,
      WHITESPACE_STRING);

    return search.trim();
  }

  /**
   * Takes the formatted suggestion, makes it lower case and blanks out any
   * existing whitespace for searching.
   */
  private String normalizeSuggestion(String formattedSuggestion) {
    // Formatted suggestions should already have normalized whitespace. So we
    // can skip that step.

    // Lower case suggestion.
    formattedSuggestion = formattedSuggestion.toLowerCase();

    // Apply whitespace.
    if (whitespaceChars != null) {
      for (int i = 0; i < whitespaceChars.length; i++) {
        char ignore = whitespaceChars[i];
        formattedSuggestion = formattedSuggestion.replace(ignore,
          WHITESPACE_CHAR);
      }
    }
    return formattedSuggestion;
  }
}

