/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut of interfaces agreed upon 20060707
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IModel;

/**
 * As defined at <a href="http://spwiki.editme.com/Context">http://spwiki.editme.com/Context</a> and <a href="http://spwiki.editme.com/ContextProvider">http://spwiki.editme.com/ContextProvider</a>, This interface holds a set of {@link IDigitalSubject} objects.<p>
 * 
 */
public interface IContext
{
	/*
	 * Context-wide methods
	 */

	/**
	 * Opens this Context using the provided authentication object.<p>
	 * Upon success (return with no exception), this Context is in an "open" state 
	 * and is available to have other methods called (such as 
	 * {@link #addRelationship(URI)}, 
	 * {@link #addSubject(URI, String)}, 
	 * {@link #exportData(String, String)}, 
	 * {@link #getRelationships()}, 
	 * {@link #getSubject(String)}, 
	 * {@link #getSubject(String, Iterator)}, 
	 * {@link #getSubjects(IFilter)},
	 * {@link #getSubjects(IFilter, Iterator)},
	 * {@link #importData(String, String)},
	 * {@link #verifySubjectAttributes(String, Iterator)}}
	 * ). 
	 * Methods requiring the context to be open throw {@link ContextNotOpenException} 
	 * when in an un-open state.<p>
	 * An open state specifies that the specified authentication materials have been used to successfully authenticate to this Context.<p>
	 * If this IContext is already in an open state, this throws {@link ContextOpenException}.<p>
	 * The antithesis of this method is close().
	 *
	 * @param authentication AuthN object used to open the context.  This may be an instance of {@link IAuthNAttributesMaterials} or some other object used to authenticate. 
	 * @return cuid of the DigitalSubject representing the identity used to open the context.  May be null.
	 * @throws IdASException
	 * @throws ContextOpenException, AuthenticationException
	 * @see #close() 
	 */
	public String open(Object authentication) throws IdASException, ContextOpenException;
	
	/**
	 * Closes this Context and returns it to an un-open state.<p>
	 * Methods requiring the context to be open throw ContextNotOpenException 
	 * when in an un-open state.<p>
	 * 
	 * @throws IdASException
	 */
	public void close() throws IdASException;

	/**
	 * Tests whether this context is open and authenticated using the specified identity
	 * @param identity 	May be {@code null}. 
	 * 					When provided, asserts that this context was opened with the specified identity. 
	 * 					When (@code null), no identity check is performed. 
	 * @return {@code true} if this context is open as specified, {@code false} otherwise
	 * @throws IdASException
	 */
	public boolean isOpen(Object identity) throws IdASException;
	
	/**
	 * Returns the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which return the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @return schema is returned as a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public String getSchema() throws IdASException;

	/**
	 * Sets the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which take the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @param schema a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public void setSchema(String schema) throws IdASException;

	/**
	 * Returns a model of the context object as it defined in the context's
	 * schema. Returned model could be used to analyse context's schema and
	 * discover what types of Digital Subjects are defined.
	 * 
	 * @return the model of this Context.
	 * @throws IdASException
	 */
	public IContextModel getContextModel() throws IdASException;

	/**
	 * Returns a model of whatever is named by the passed identifier
	 * 
	 * @param elementType A URI that names a type of element (this could
	 * be an attribute ID, attribute value type, etc.)
	 * @return the model of the specified element type.
	 * @throws IdASException
	 */
	public IModel getModel(URI elementType) throws IdASException;

	/**
	 * Returns a reference to this Context. This value is used in Context Relationships and used to open this Context.
	 * 
	 * @return the reference to this Context.
	 * @throws IdASException
	 */
	public URI getContextRef() throws IdASException;

	/**
	 * Exports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this export only Digital Subjects, or the entire Context (Metadata, Schema, DS, Relationships)?
	 * @param filter May be {@code null}. Used to select a subset of this Context to be exported.
	 * @param representationFormat Indicates the format of the string to be returned. TODO (Arch)(Doc): Specify format type. Give example
	 * @return String in the form of representationFormat containing the exported Context.
	 * @throws IdASException
	 */
	public String exportData(String filter, String representationFormat) throws IdASException;

	/**
	 * Imports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this import only Digital Subjects, or the entire Context (Metadata, Schema, DS, Relationships)?
	 * TODO (Arch)(Doc): Discuss collisions.
	 * @param filter May be {@code null}. Used to select a subset of the external data set to be imported.
	 * @param representationFormat Indicates the format of the external data string being specified in external. TODO (Arch)(Doc): Specify format type. Give example
	 * @throws IdASException
	 */
	public void	importData(String filter, String representationFormat) throws IdASException;

	/*
	 * Factory-like methods for creating various IdAS elements 
	 */
	
	/**
	 * Builds an {@link IAuthNAttributesMaterials} that may be used when calling {@link #open(Object)}
	 */
	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException;
	
	/**
	 * Builds an IAttribute (unassociated with any digital subject).
	 * This is used to build up {@link IAuthNAttributesMaterials}
	 * @param attrID
	 * @return An {@link IAttribute}
	 * @throws IdASException 
	 */
	public IAttribute buildAttribute(URI attrID) throws IdASException;
	
	/**
	 * Builds an IFilter instance that may be used when calling {@link #getSubjects(IFilter)}
	 */
	public IFilter buildFilter() throws IdASException;
	
	/**
	 * Builds an assertion for Digital Subject Attributes 
	 * (see {@link IDigitalSubject#getAttributes()} and {@link IDigitalSubject#getAttribute(URI)}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException;

	/**
	 * Builds an assertion for Digital Subject CUIDs 
	 * (see {@link IDigitalSubject#getSubjectID()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterCUIDAssertion buildCUIDAssertion() throws IdASException;

	/**
	 * Builds an assertion for Digital Subject Metadata 
	 * (see {@link IDigitalSubject#getMetadataSet()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterMetadataAssertion buildMetadataAssertion() throws IdASException;

	/**
	 * Builds an assertion for Digital Subject Types 
	 * (see {@link IDigitalSubject#getType()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterTypeAssertion buildTypeAssertion() throws IdASException;

	/**
	 * Builds a simple attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * IAttributeValue value = buildSimpleValue("http://www.w3.org/2001/XMLSchema/normalizedString", new String("Some Value"));
	 * </code>
	 * @param type The type for the simple value
	 * @param value The value for the simple type. 
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public ISimpleAttrValue buildSimpleAttrValue(URI type, Object value) throws IdASException, InvalidTypeException;
	
	/**
	 * Builds a complex attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * TODO: do an example of building this and then adding some attributes
	 * </code>
	 * @param type The type for the simple value
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public IComplexAttrValue buildComplexAttrValue(URI type) throws IdASException, InvalidTypeException;
	
	/**
	 * Builds a relationship to be passed to {@link #addRelationship(URI)}.<p>
	 * @param type The type of relationship. TODO (Doc): give example.
	 * @return The resulting Relationship.
	 * @throws IdASException
	 */
	public IContextRelationship buildRelationship(URI type) throws IdASException;


	/*
	 * Methods relating to Digital Subjects
	 */

	/**
	 * Creates a new Digital Subject for this Context.<br>
	 * This call is typically followed by one or more calls to 
	 * {@link IDigitalSubject#addAttribute}, {@link IDigitalSubject#addMetadata}
	 * on the returned IDigitalSubject
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @param type The URI specifying the type of Digital Subject being created
	 * @param subjectID May be null.  When specified, uniquely names the subject within
	 * this Context.  When null, indicates that the caller plans to later add an 
	 * attribute or attributes which will be used to construct the unique subject ID
	 * @return The resulting IDigitalSubject.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @throws InvalidTypeException when the type is invalid
	 * @throws InvalidSubjectIDException when the subjectID is invalid
	 * @throws SubjectExistsException when the subjectID is known to already exist
	 * @throws IdASException
	 */
	IDigitalSubject addSubject(URI type, String subjectID) throws IdASException, InvalidTypeException, InvalidSubjectIDException, SubjectExistsException;

	/**
	 * Creates a new Digital Subject for this Context by copying the data from the passed subject<br>
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @param copyFrom an {@link IDigitalSubject} from which data is copied to construct a new 
	 * Digital Subject
	 * @return The resulting IDigitalSubject.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @throws SubjectExistsException when Subject is known to already exist
	 * @throws IdASException
	 */
	IDigitalSubject addSubject(IDigitalSubject copyFrom) throws IdASException, SubjectExistsException;

	/**
	 * This is the same as passing null as the attrSelectionList in {@link #getSubject(String, Iterator)}
	 *	@see #getSubject(String, Iterator)
	 *	 
	 */
	public IDigitalSubject getSubject(String cuid) throws IdASException;
	/**
	 * Returns the Digital Subject matching the specified cuid.
	 * This is effectively the same as calling {@link #getSubjects} with
	 * the filter set to {@literal <appropriate filter for cuid=name>}<p>
	 * TODO (Arch)(Doc): discuss whether modifications to the returned IDigitalSubject affect the stored subject.<p>
	 * 
	 * @param cuid Contextually Unique ID of Digital Subject to return.
	 * @param attrSelectionList Iterator holding the {@link URI}s of IAttribute types 
	 * 							that the caller expects to access.  
	 * 							A {@code null}indicates that all attributes will be read.  
	 * 							An empty set indicates that no attributes will be read.
	 * @return IDigitalSubject.
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchSubjectException when subject specified by cuid is not found.
	 *
	 */
	public IDigitalSubject getSubject(String cuid, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * @see #getSubjects(IFilter, Iterator)
	 * This is the same as passing null as the attrSelectionList in {@link #getSubjects(IFilter, Iterator)}
	 * @return an Iterator of {@link IDigitalSubject}s that match the filter 
	 */
	public Iterator getSubjects(IFilter filter) throws IdASException;
	/**
	 * From this context, return {@link IDigitalSubject}s that match the specified filter <p>
	 * @param filter 	Filter used to constrain the set of {@link IDigitalSubject}s returned.
	 * 					A {@code null} or empty filter causes all {@link IDigitalSubject}s to be returned.
	 * @param attrSelectionList Iterator of {@link URI}s of IAttribute types that the caller expects to access.  A {@code null} indicates that all attributes will be read.  An empty set indicates that no attributes will be read.
	 * @return an Iterator of {@link IDigitalSubject}s that match the filter 
	 * @throws IdASException
	 *
	 */
	public Iterator getSubjects(IFilter filter, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * Simple method for comparing a set of attribute assertions against a Digital Subject.<p>
	 * TODO (Arch): This doesn't leave a way to verify that a DS representing a group contains a specific member in its member value.
	 * 		We may want to consider allowing a filter to be passed here, or at least a partial attribute.
	 * 
	 * @param cuid Contextually Unique ID of Digital Subject being verified.
	 * @param attributes Iterator of {@link IAttribute} to be compared to the Digital Subject specified by cuid. 
	 * @return {@code true} if all attribute assertions in attributes are found to be equal to those attributes in the Digital Subject specified by cuid. {@code false} otherwise. 
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchSubjectException when subject specified by cuid is not found.
	 */
	public boolean verifySubjectAttributes(String cuid, Iterator attributes) throws IdASException;

	/**
	 * Returns the relationships this Object has to others.
	 * @return Iterator of IContextRelationship objects. The set of Relationships this Object has.
	 * @throws IdASException
	 */
	public Iterator getRelationships() throws IdASException;

	/**
	 * Adds a context relationship to this context. 
	 * This call is typically followed by a call to {@link IContextRelationship#addRelatedContext(URI)}
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @param relationshipType The type of relationship to add.
	 * @throws IdASException
	 */
	public IContextRelationship addRelationship(URI relationshipType) throws IdASException;

	/**
	 * Applies all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including digital subjects, metadata, attributes, attribute metadata and it's value(s), and attribute values) 
	 * to a backing data store.  These updates are applied atomically when possible, but may not be 
	 * applied atomically by Context Providers which do not support this.
	 * Update operations which will be applied are any of the following:<br>
	 * {@link IContext#addSubject(URI, String)}<br>
	 * {@link IDigitalSubject#remove()}<br>
	 * {@link IHasAttributes#addAttribute(URI)}<br>
	 * {@link IHasMetadata#addMetadata(URI)}<br>
	 * {@link IAttribute#remove()}<br>
	 * {@link IAttribute#addValue(URI)}<br>
	 * {@link IAttribute#addSimpleValue(URI, Object)}<br>
	 * {@link IAttribute#addComplexValue(URI)}<br>
	 * {@link IAttributeValue#remove()}<br>
	 * {@link ISimpleAttrValue#setData(Object)}<br>
	 * {@link IMetadata#setData(Object)}<br>
	 * On a successful return, all updates which have been performed on this Contet 
	 * or on any contained elements are said to be fully applied to the Context.  
	 * Subsequent update operations must be followed by another call to applyUpdates.
	 * On error (when an exception is thrown), the updates are said to not yet fully applied
	 * to the Context.  In this case, the caller may attempt to remedy the issue which 
	 * caused the call to applyUpdates by performing further update operations followed
	 * by another call to applyUpdates().
	 * @throws IdASException
	 */
	void applyUpdates() throws IdASException;	

	/**
	 * Cancels all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including digital subjects, metadata, attributes, attribute metadata and it's value(s), and attribute values).
	 * Any updates which have already been applied (by calling {@link #applyUpdates()}) are not canceled.   
	 * @throws IdASException
	 * @see #applyUpdates()
	 */
	void cancelUpdates() throws IdASException;	

	
}
