/*******************************************************************************
 * Copyright (c) 2006 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut of interfaces agreed upon 20060707
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.model.IAttributeModel;

/**
 * As defined at (TODO (Doc): add reference), This interface represents an attribute of a DigitalSubject.<p>
 * An attribute is a typed set of values. 
 * When the attribute's model dictates that it is single-valued {@link #isSingleValued()}
 * then this object may be cast to an {@link ISingleValuedAttribute}<p>
 */
public interface IAttribute extends IHasMetadata
{
	/**
	 * Returns the type (AttributeID) of this attribute as a {@link URI}.<p>
	 * From this URI, a consumer should be able to derive some kind of human-readable name (such as "age", "title", "employeeNumber", etc.).<p>
	 * A consumer might also be able to determine from this URI, the type or range of types of object(s) returned from {@link ISingleValuedAttribute#getValue()} and {@link #getValues()}.<p>
	 * These determinations may be made either by examining parts of the URI itself, or dereferencing it (or parts) to other sources of information.<p>
	 * @throws IdASException
	 */
	public URI getAttrID() throws IdASException;

	/**
	 * Returns all values for this attribute.
	 * @return an Iterator of {@link IAttributeValue}s
	 * @throws IdASException
	 */
	public Iterator getValues() throws IdASException;

	/**
	 * Creates a new value for this attribute.  This method is useful
	 * when the caller doesn't know whether the URI specified in the 
	 * type param will build an ITypedValue or IComplexAttrValue.  Otherwise
	 * callers will typically call {@link IAttribute#addSimpleValue(URI, Object)}
	 * or {@link IAttribute#addComplexValue(URI)}
	 * This call is typically followed by a call to {@link IAttributeValue#isSimple()}
	 * on the returned IAttributeValue.
	 * When found to be an ISimpleAttrValue, {@link ISimpleAttrValue#setData(Object)} is typically 
	 * called on the returned IAttributeValue (cast as an ISimpleAttrValue).
	 * When found to be an IComplexAttrValue, one typically calls 
	 * {@link IHasAttributes#addAttribute(URI)} on the returned IAttributeValue 
	 * (cast as an IComplexAttrValue or IHasAttributes).
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param type The URI specifying the type of Value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IAttributeValue addValue(URI type) throws IdASException, InvalidTypeException;

	/**
	 * Creates a new value for this attribute by copying the passed {@link IAttributeValue}
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param copyFrom The {@link IAttributeValue} from which data is copied to create the 
	 * returned {@link IAttributeValue}
	 * @throws {@link IdASException}
	 */
	IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException;

	/**
	 * Creates a new simple value for this attribute. 
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param type The URI specifying the type of Value being created
	 * @param data The data of the attribute value
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	ISimpleAttrValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException;
	
	/**
	 * Creates a new complex value for this attribute. 
	 * This call is typically followed by one or more calls to 
	 * {@link IHasAttributes#addAttribute} on the returned IComplexAttrValue.
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param type The URI specifying the type of complex value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IComplexAttrValue addComplexValue(URI type) throws IdASException, InvalidTypeException;
	
	
	/**
	 * Removes this attribute from its container
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @throws {@link IdASException}
	 */
	void remove() throws IdASException;
	
	/**
	 * Returns a model of this type of attribute as it defined in the context's
	 * schema. The returned model could be used to analyze the attribute's type and
	 * discover what its value model is.
	 * 
	 * @return a model of this type of attribute
	 * @throws IdASException
	 */
	public IAttributeModel getModel() throws IdASException;
	
	/**
	 * Convenience method which returns true when the model for this attribute dictates that only 
	 * a single value my exist.  This is analogous to calling
	 * <pre>
	 * (myAttribute.getModel().getMaxCardinality() == 1) 
	 * </pre>
	 * @return boolean. When true, this object may be cast to an {@link ISingleValuedAttribute} 
	 */
	public boolean isSingleValued() throws IdASException;
	
	/**
	 * Returns true if the passed attribute is equal to this one.
	 * The Attribute id's are compared, the metadata sets are compared,
	 * and all values are compared for equality
	 * @param attr the attribute to be compared to this one
	 * @return true if the passed attr is equal to this attr.
	 * @throws IdASException
	 */
	public boolean equals(IAttribute attr) throws IdASException;
}
