/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut of interfaces agreed upon 20061005
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;

import org.eclipse.higgins.idas.api.IdASException;

/**
 * Represents a filter assertion about a Digital Subject's Attributes 
 * ({@link org.eclipse.higgins.idas.api.IDigitalSubject#getAttribute(URI)} and {@link org.eclipse.higgins.idas.api.IDigitalSubject#getAttributes()}).<p>
 *
 * This assertion is made up of a comparator (operator), attribute type and assertion value.<p>
 *  
 * The particular type and form of the IAttributeValue used as an assertion is dictated by the comparator and the attribute type.
 * In other words, a particular attribute type will specify the comparators valid for that type, as well as the format of assertion value for each valid comparator.
 * TODO (Doc): Refer to example specifications for this
 */
public interface IFilterAttributeAssertion extends IFilterAssertion {

	/**
	 * Compares true when the specified attribute type is present in a tested Digital Subject.<p>
	 * An assertion value is not specified.<P>
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_PRESENT = "urn:org.eclipse.higgins.idas.api.comparator.attribute.present";

	/**
	 * Compares true when a tested Digital Subject has a attribute of the specified type which contains at least one value equal to the specified assertion value.<P> 
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_EQ = "urn:org.eclipse.higgins.idas.api.comparator.attribute.equality";

	/**
	 * Compares true when a tested Digital Subject has a attribute of the specified type which contains at least one value greater or equal to the specified assertion value.<P> 
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_GE = "urn:org.eclipse.higgins.idas.api.comparator.attribute.greaterOrEqual";

	/**
	 * Compares true when a tested Digital Subject has a attribute of the specified type which contains at least one value less or equal to the specified assertion value.<P>
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_LE = "urn:org.eclipse.higgins.idas.api.comparator.attribute.lessOrEqual";

	/**
	 * Compares true when a tested Digital Subject has a attribute of the specified type which contains at least one value containing the substring specified by the assertion value.<P>
	 * TODO: need to handle initial, any, final substrings. 
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_SUBSTR = "urn:org.eclipse.higgins.idas.api.comparator.attribute.substring";

	/**
	 * This Comparator is used for assertions of type SubjectRelationship (and sub-types). 
	 * Compares true when a tested Digital Subject has the asserted relationship (of the asserted type) 
	 * whether that relationship is implied or explicit.<p>
	 * For example, if the assertion specified a type of "urn:parentOf" and reference of 
	 * {"someContext", "urn:Mary"} it could result in a match because either:<br>
	 * A subject has a relationship attribute of type "urn:parentOf" and reference of 
	 * {"someContext", "urn:Mary"), or<br>
	 * A subject is referenced by the subject "urn:Mary" in the context "someContext" 
	 * where the referencing relationship attribute is of type "urn:childOf" 
	 * (assuming the model backing the context provides the proper inferences between 
	 * parentOf and childOf)<p>  
	 * Value: {@value}
	 */
	public static final String COMP_ATTR_INFERRED_REL_EQ = "urn:org.eclipse.higgins.idas.api.comparator.attribute.inferred.relationship.equality";
	
	/**
	 * Specifies the assertion data to be tested against values of a Digital Subject's attribute.
	 * 
	 * To specify an assertion value nested in a complex type (like a postalAddress.city of "Provo"), 
	 * one would pass in an IComplexAttrValue with the attribute(ies) containing 
	 * the assertion(s). i.e. the IFilterAttributeAssertion would have a type like "urn:postalAddress" 
	 * and the assertionValue would be an IComplexAttrValue containing a attribute. That attribute 
	 * would have a type like "urn:city", and would have an ITypedValue containing the assertion "Provo"
	 * 
	 * This allows multiple parts of a complex value to be asserted such that one may specify a filter like
	 * postalAddress.city = "Provo" AND postalAddress.state = "Utah", where both assertions have to 
	 * match the same postalAddress value.
	 * 
	 */
	public void setAssertionValue(IAttributeValue assertion) throws IdASException;
	
	/**
	 * Adds an attribute metadata filter to this attribute assertion.
	 * @param metadataFilter A filter, specifying attribute metadata 
	 * 						which must match for this assertion to match.
	 * 						This allows one to differentiate between two attributes 
	 * 						of the same type where the metadata differs. 
	 */
	public void setMetadataFilter(IFilter metadataFilter) throws IdASException;
}
