/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.model.IMetadataModel;


/**
 * TODO (Doc): Add (or point to) doc which explains what metadata is and how it's used<p>
 * TODO (Doc): Show an example (preferably that of an owner or source for the object to which some metadata is attached)
 * TODO (Doc): Finish javadoc for methods
 */
public interface IMetadata {
	/**
	 * Returns the metadata ID of this metadata as a {@link URI}.<p>
	 * From this URI, a consumer should be able to derive some kind of human-readable identifier
	 * (such as "timeToLive", "owner", "creationDate", etc.).<p>
	 * This metadata ID is used to distinguish this metadata from other metadata elements
	 * within a collection of metadata.
	 * Note that {@link IMetadataModel#getType()} will return the same URI that this method returns.
	 * The reason we use the term "MetaID" here is because from a metadata instance point of view, this is
	 * the identifier which distinguishes this metadata element from others.  The reason it is referred to as a type
	 * from the metadata's model point of view is because it's tied to the capabilities of the metadata.  
	 * @throws IdASException
	 */
	public URI getMetaID() throws IdASException;

	/**
	 * Returns all values for this metadata element.
	 * @return an Iterator of {@link IMetadataValue}s
	 * @throws IdASException
	 */
	public Iterator getValues() throws IdASException;

	/**
	 * Creates a new value for this metadata element. This method is useful
	 * when the caller doesn't know whether the URI specified in the 
	 * dataType param will build an ITypedValue or IComplexMetaValue.  Otherwise
	 * callers will typically call {@link IMetadata#addSimpleValue(URI, Object)}
	 * or {@link IMetadata#addComplexValue(URI)}
	 * This call is typically followed by a call to {@link IMetadataValue#isSimple()}
	 * on the returned IMetadataValue.
	 * When found to be an ISimpleMetaValue, {@link ISimpleMetaValue#setData(Object)} is typically 
	 * called on the returned IMetadataValue (cast as an ISimpleMetaValue).
	 * When found to be an IComplexMetaValue, one typically calls 
	 * {@link IHasMetadata#addMetadata(URI)} on the returned IMetadataValue 
	 * (cast as an IComplexMetaValue or IHasMetadata).
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the data type of Value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IMetadataValue addValue(URI dataType) throws IdASException, InvalidTypeException;

	/**
	 * Creates a new value for this metadata element by copying the passed {@link IMetadataValue}
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param copyFrom The {@link IMetadataValue} from which data is copied to create the 
	 * returned {@link IMetadataValue}
	 * @throws {@link IdASException}
	 */
	IMetadataValue addValue(IMetadataValue copyFrom) throws IdASException;

	/**
	 * Creates a new simple value for this metadata element. 
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the type of Value being created
	 * @param data The data of the metadata value
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	ISimpleMetaValue addSimpleValue(URI dataType, Object data) throws IdASException, InvalidTypeException;
	
	/**
	 * Creates a new complex value for this metadata element. 
	 * This call is typically followed by one or more calls to 
	 * {@link IHasMetadata#addMetadata} on the returned IComplexMetaValue.
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the type of complex value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IComplexMetaValue addComplexValue(URI dataType) throws IdASException, InvalidTypeException;
	
	
	/**
	 * Removes this metadata element from its container
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @throws {@link IdASException}
	 */
	void remove() throws IdASException;
	
	/**
	 * Returns a model of this type of metadata as it defined in the context's
	 * schema. The returned model could be used to analyze the metadata's type and
	 * discover what its value model is.
	 * 
	 * @return a model of this type of metadata
	 * @throws IdASException
	 */
	public IMetadataModel getModel() throws IdASException;
	
	/**
	 * Convenience method which returns true when the model for this metadata dictates that only 
	 * a single value may exist.  This is analogous to calling
	 * <pre>
	 * (myMetadata.getModel().getMaxCardinality() == 1) 
	 * </pre>
	 * @return boolean. When true, this object may be cast to an {@link ISingleValuedMetadata} 
	 */
	public boolean isSingleValued() throws IdASException;
	
	/**
	 * Returns true if the passed metadata is equal to this one.
	 * The Metadata id's are compared, and all values are compared for equality
	 * @param meta the metadata to be compared to this one
	 * @return true if the passed metadata is equal to this metadata.
	 * @throws IdASException
	 */
	public boolean equals(IMetadata meta) throws IdASException;

}