/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.model.IAttributeModel;

/**
 * As defined at (TODO (Doc): add reference), This interface represents an attribute of an Entity.<p>
 * An attribute is a typed set of values. 
 * When the attribute's model dictates that it is single-valued {@link #isSingleValued()}
 * then this object may be cast to an {@link ISingleValuedAttribute}<p>
 * This extends {@link IHasAttributes} so that metadata about the attribute may be presented as 
 * attributes on this attribute.<p>
 */
public interface IAttribute extends IHasAttributes
{
	/**
	 * Returns the attribute ID of this attribute as a {@link URI}.<p>
	 * From this URI, a consumer should be able to derive some kind of human-readable identifier 
	 * (such as "age", "title", "employeeNumber", etc.).<p>
	 * This attribute ID is used to distinguish this attribute from other attributes within a collection of attributes.
	 * Note that {@link IAttributeModel#getType()} will return the same URI that this method returns.
	 * The reason we use the term "AttrID" here is because from an attribute instance point of view, this is
	 * the identifier which distinguishes this attribute from others.  The reason it is referred to as a type
	 * from the attribute's model point of view is because it's tied to the capabilities of the attribute.  
	 * @throws IdASException
	 */
	public URI getAttrID() throws IdASException;

	/**
	 * Returns all values for this attribute.
	 * @return an Iterator of {@link IAttributeValue}s
	 * @throws IdASException
	 */
	public Iterator getValues() throws IdASException;

	/**
	 * Returns all values for this attribute.
	 * @param extensions may be null. One or more extensions valid for this operation. 
	 * @return an Iterator of {@link IAttributeValue}s
	 * @throws IdASException
	 */
	public Iterator getValues(IExtension[] extensions) throws IdASException;

	/**
	 * Creates a new value for this attribute.  This method is useful
	 * when the caller doesn't know whether the URI specified in the 
	 * dataType param will build an ITypedValue or IComplexAttrValue.  Otherwise
	 * callers will typically call {@link IAttribute#addSimpleValue(URI, Object)}
	 * or {@link IAttribute#addComplexValue(URI)}
	 * This call is typically followed by a call to {@link IAttributeValue#isSimple()}
	 * on the returned IAttributeValue.
	 * When found to be an ISimpleAttrValue, {@link ISimpleAttrValue#setData(Object)} is typically 
	 * called on the returned IAttributeValue (cast as an ISimpleAttrValue).
	 * When found to be an IComplexAttrValue, one typically calls 
	 * {@link IHasAttributes#addAttribute(URI)} on the returned IAttributeValue 
	 * (cast as an IComplexAttrValue or IHasAttributes).
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the data type of Value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IAttributeValue addValue(URI dataType) throws IdASException, InvalidTypeException;

	/**
	 * Creates a new value for this attribute by copying the passed {@link IAttributeValue}
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param copyFrom The {@link IAttributeValue} from which data is copied to create the 
	 * returned {@link IAttributeValue}
	 * @throws {@link IdASException}
	 */
	IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException;

	/**
	 * Creates a new simple value for this attribute. 
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the type of Value being created
	 * @param data The data of the attribute value
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	ISimpleAttrValue addSimpleValue(URI dataType, Object data) throws IdASException, InvalidTypeException;
	
	/**
	 * Creates a new complex value for this attribute. 
	 * This call is typically followed by one or more calls to 
	 * {@link IHasAttributes#addAttribute} on the returned IComplexAttrValue.
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param dataType The URI specifying the type of complex value being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the type is invalid
	 */
	IComplexAttrValue addComplexValue(URI dataType) throws IdASException, InvalidTypeException;
	
	
	/**
	 * Removes this attribute from its container
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @throws {@link IdASException}
	 */
	void remove() throws IdASException;
	
	/**
	 * Returns a model of this type of attribute as it defined in the context's
	 * schema. The returned model could be used to analyze the attribute's type and
	 * discover what its value model is.
	 * 
	 * @return a model of this type of attribute
	 * @throws IdASException
	 */
	public IAttributeModel getModel() throws IdASException;
	
	/**
	 * Convenience method which returns true when the model for this attribute dictates that only 
	 * a single value may exist.  This is analogous to calling
	 * <pre>
	 * (myAttribute.getModel().getMaxCardinality() == 1) 
	 * </pre>
	 * @return boolean. When true, this object may be cast to an {@link ISingleValuedAttribute} 
	 */
	public boolean isSingleValued() throws IdASException;
	
	/**
	 * Returns true if the passed attribute is equal to this one.
	 * The Attribute id's are compared, the metadata sets are compared,
	 * and all values are compared for equality
	 * @param attr the attribute to be compared to this one
	 * @return true if the passed attrribute is equal to this attribute.
	 * @throws IdASException
	 */
	public boolean equals(IAttribute attr) throws IdASException;
}
