/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.idas.api.model.IContextModel;

/**
 * As defined at <a href="http://spwiki.editme.com/Context">http://spwiki.editme.com/Context</a> and <a href="http://spwiki.editme.com/ContextProvider">http://spwiki.editme.com/ContextProvider</a>, This interface holds a set of {@link IEntity} objects.<p>
 * All Contexts follow the http://www.eclipse.org/higgins/ontologies/2008/6/higgins ontology
 * as a base. As such, each Context may have the attributes:<p> 
 * http://www.eclipse.org/higgins/ontologies/2008/6/higgins#contextRelation<p>
 * http://www.eclipse.org/higgins/ontologies/2008/6/higgins#contextCorrelation<p>
 * Also, each Context has the attribute 
 * http://www.eclipse.org/higgins/ontologies/2008/6/higgins#contextId. {@link #getContextID()}
 * provides a shortcut to access the value of this attribute.
 */
public interface IContext extends IHasAttributes
{
	/*
	 * Context-wide methods
	 */

	/**
	 * Opens this Context using the provided authentication object.<p>
	 * Upon success (return with no exception), this Context is in an "open" state 
	 * and is available to have other methods called (such as 
	 * {@link #addEntity(URI, String)}, 
	 * {@link #exportData(String, String)}, 
	 * {@link #getEntity(String)}, 
	 * {@link #getEntity(String, Iterator)}, 
	 * {@link #getEntities(IFilter)},
	 * {@link #getEntities(IFilter, Iterator)},
	 * {@link #importData(String, String)},
	 * {@link #verifyEntityAttributes(String, Iterator)}}
	 * ). 
	 * Methods requiring the context to be open throw {@link ContextNotOpenException} 
	 * when in an un-open state.<p>
	 * An open state specifies that the specified authentication materials have been used to successfully authenticate to this Context.<p>
	 * If this IContext is already in an open state, this throws {@link ContextOpenException}.<p>
	 * The antithesis of this method is close().
	 *
	 * @param authentication AuthN object used to open the context.  This may be an instance of {@link IAuthNAttributesMaterials} or some other object used to authenticate.
	 * @param extensions may be null. One or more extensions valid for this operation. 
	 * @return Entity ID of the Entity representing the identity used to open the context.  May be null.
	 * @throws IdASException
	 * @throws ContextOpenException, AuthenticationException
	 * @see #close() 
	 */
	public String open(Object authentication, IExtension[] extensions) throws IdASException, ContextOpenException;
	/**
	 * @see #open(Object, IExtension[])
	 */
	public String open(Object authentication) throws IdASException, ContextOpenException;
	
	
	/**
	 * Closes this Context and returns it to an un-open state.<p>
	 * Methods requiring the context to be open throw ContextNotOpenException 
	 * when in an un-open state.<p>
	 * 
	 * @throws IdASException
	 */
	public void close() throws IdASException;

	/**
	 * Tests whether this context is open and authenticated using the specified identity
	 * @param identity 	May be {@code null}. 
	 * 					When provided, asserts that this context was opened with the specified identity. 
	 * 					When (@code null), no identity check is performed. 
	 * @return {@code true} if this context is open as specified, {@code false} otherwise
	 * @throws IdASException
	 */
	public boolean isOpen(Object identity) throws IdASException;
	
	/**
	 * Returns the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which return the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @return schema is returned as a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public String getSchema() throws IdASException;

	/**
	 * Sets the schema governing this context.<p>
	 * TODO (Doc): Add pointers to OWL.<p>
	 * TODO (Doc): Add (or point to) documentation which includes sample schema.<p>
	 * TODO (Arch)(Impl): Add method(s) which take the schema as a stream, DOM document, or other iterable object.
	 * 
	 * @param schema a {@link String} in OWL-DL format.
	 * @throws IdASException
	 */
	public void setSchema(String schema) throws IdASException;

	/**
	 * Returns a model of the context object as it defined in the context's
	 * schema. Returned model could be used to analyze context's schema and
	 * discover what types of Entities are defined.
	 * 
	 * @return the model of this Context.
	 * @throws IdASException
	 */
	public IContextModel getContextModel() throws IdASException;

	/**
	 * Returns a reference to this Context.
	 * This is a shortcut for reading the http://www.eclipse.org/higgins/ontologies/2008/6/higgins#contextId attribute on this context. 
	 * This value is used in Context Relationships and used to open this Context.
	 * 
	 * @return the reference to this Context.
	 * @throws IdASException
	 */
	public URI getContextID() throws IdASException;

	/**
	 * Exports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this export only Entities, or the entire Context (Attributes, Schema, Entities)?
	 * @param filter May be {@code null}. Used to select a subset of this Context to be exported.
	 * @param representationFormat Indicates the format of the string to be returned. TODO (Arch)(Doc): Specify format type. Give example
	 * @return String in the form of representationFormat containing the exported Context.
	 * @throws IdASException
	 */
	public String exportData(String filter, String representationFormat) throws IdASException;

	/**
	 * Imports this Context.<p>
	 * TODO (Arch): filter format TBD<p>
	 * TODO (Arch): Does this import only Entities, or the entire Context (Attributes, Schema, Entities)?
	 * TODO (Arch)(Doc): Discuss collisions.
	 * @param filter May be {@code null}. Used to select a subset of the external data set to be imported.
	 * @param representationFormat Indicates the format of the external data string being specified in external. TODO (Arch)(Doc): Specify format type. Give example
	 * @throws IdASException
	 */
	public void importData(String filter, String representationFormat) throws IdASException;

	/*
	 * Factory-like methods for creating various IdAS elements 
	 */
	
	/**
	 * Builds an {@link IAuthNAttributesMaterials} that may be used when calling {@link #open(Object)}
	 */
	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException;
	
	/**
	 * Builds an IAttribute (unassociated with any Entity).
	 * This is used to build up {@link IAuthNAttributesMaterials}
	 * @param attrID
	 * @return An {@link IAttribute}
	 * @throws IdASException 
	 */
	public IAttribute buildAttribute(URI attrID) throws IdASException;
	
	/**
	 * Builds an IFilter instance that may be used when calling {@link #getEntities(IFilter)}
	 */
	public IFilter buildFilter() throws IdASException;
	
	/**
	 * Builds an assertion for Entity Attributes 
	 * (see {@link IEntity#getAttributes()} and {@link IEntity#getAttribute(URI)}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException;

	/**
	 * Builds an assertion for Entity IDs 
	 * (see {@link IEntity#getEntityID()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 * Note that this is simply a convenience method equivalent to creating an
	 * IFilterAttributeAssertion for the 
	 * http://www.eclipse.org/higgins/ontologies/2008/6/higgins#entityId attribute
	 */
	public IFilterEntityIDAssertion buildEntityIDAssertion() throws IdASException;

	/**
	 * Builds an assertion for Entity Types 
	 * (see {@link IEntity#getEntityType()}) 
	 * which can be passed to {@link IFilter#setAssertion(IFilterAssertion)}
	 */
	public IFilterEntityTypeAssertion buildEntityTypeAssertion() throws IdASException;

	/**
	 * Builds a simple attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * IAttributeValue value = buildSimpleValue("http://www.w3.org/2001/XMLSchema/normalizedString", new String("Some Value"));
	 * </code>
	 * @param dataType The type for the simple value
	 * @param value The value for the simple type. 
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public ISimpleAttrValue buildSimpleAttrValue(URI dataType, Object value) throws IdASException, InvalidTypeException;
	
	/**
	 * Builds a complex attribute value to be passed to {@link IFilterAttributeAssertion#setAssertionValue(IAttributeValue)}.<p>
	 * Example:<p>
	 * <code>
	 * TODO: do an example of building this and then adding some attributes
	 * </code>
	 * @param dataType The type for the simple value
	 * @return the built attribute value
	 * @throws IdASException
	 * @throws InvalidTypeException
	 */
	public IComplexAttrValue buildComplexAttrValue(URI dataType) throws IdASException, InvalidTypeException;
	
	/**
	 * Creates a new Entity for this Context.<br>
	 * This call is typically followed by one or more calls to 
	 * {@link IEntity#addAttribute} on the returned IEntity
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @param entityType The URI specifying the type of Entity being created
	 * @param entityID May be null.  When specified, uniquely names the Entity within
	 * this Context.  When null, indicates that the caller plans to later add an 
	 * attribute or attributes which will be used to construct the unique Entity ID
	 * @return The resulting IEntity.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()}
	 * @throws InvalidTypeException when the entityType is invalid
	 * @throws InvalidEntityIDException when the entityID is invalid
	 * @throws EntityExistsException when the entityID is known to already exist
	 * @throws IdASException
	 */
	IEntity addEntity(URI entityType, String entityID) throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException;

	/**
	 * Creates a new Entity for this Context by copying the data from the passed entity<br>
	 * Note: This operation is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @param copyFrom an {@link IEntity} from which data is copied to construct a new 
	 * Entity
	 * @return The resulting IEntity.  This is only applied to any backing data 
	 * store after {@link IContext#applyUpdates()} is called.
	 * @throws EntityExistsException when copyFrom is known to already exist
	 * @throws IdASException
	 */
	IEntity addEntity(IEntity copyFrom) throws IdASException, EntityExistsException;

	/**
	 * This is the same as passing null as the attrSelectionList in {@link #getEntity(String, Iterator)}
	 *	@see #getEntity(String, Iterator)
	 *	 
	 */
	public IEntity getEntity(String entityID) throws IdASException;
	/**
	 * Returns the Entity matching the specified entityID.
	 * This is effectively the same as calling {@link #getEntities} with
	 * the filter set to {@literal <appropriate filter for entityID=name>}<p>
	 * 
	 * @param entityID Contextually unique ID of Entity to return.
	 * @param attrSelectionList Iterator holding the {@link URI}s of IAttribute types 
	 * 							that the caller expects to access.  
	 * 							A {@code null}indicates that the caller expects to read all attributes.  
	 * 							An empty set indicates that the caller expects to read no attributes.
	 *                          Note that this is only an indicator of what the caller expects to read. 
	 *                          The caller may eventually end up reading more or less.  The purpose of this 
	 *                          parameter is so that a context provider which gathers data across a network 
	 *                          connection can optimize based on the callers intended usage. 
	 * @return IEntity.
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchEntityException when entity specified by entityID is not found.
	 *
	 */
	public IEntity getEntity(String entityID, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * @see #getEntities(IFilter, Iterator)
	 * This is the same as passing null as the attrSelectionList in {@link #getEntities(IFilter, Iterator)}
	 * @return an Iterator of {@link IEntity}s that match the filter 
	 */
	public Iterator getEntities(IFilter filter) throws IdASException;
	
	/**
	 * From this context, return {@link IEntity}s that match the specified filter <p>
	 * @param filter 	Filter used to constrain the set of {@link IEntity}s returned.
	 * 					A {@code null} or empty filter causes all {@link IEntity}s to be returned.
	 * @param attrSelectionList Iterator of {@link URI}s of IAttribute types that the caller expects to access.  A {@code null} indicates that all attributes will be read.  An empty set indicates that no attributes will be read.
	 * @return an Iterator of {@link IEntity}s that match the filter 
	 * @throws IdASException
	 *
	 */
	public Iterator getEntities(IFilter filter, Iterator attrSelectionList) throws IdASException;
	
	/**
	 * Simple method for comparing a set of attribute assertions against a Entity.<p>
	 * TODO (Arch): This doesn't leave a way to verify that a Entity representing a group contains a specific member in its member value.
	 * 		We may want to consider allowing a filter to be passed here, or at least a partial attribute.
	 * 
	 * @param entityID Contextually unique ID of Entity being verified.
	 * @param attributes Iterator of {@link IAttribute} to be compared to the Entity specified by entityID. 
	 * @return {@code true} if all attribute assertions in attributes are found to be equal to those attributes in the Entity specified by entityID. {@code false} otherwise. 
	 * @throws IdASException (typically a subclass of. See below).
	 * @throws NoSuchEntityException when Entity specified by entityID is not found.
	 */
	public boolean verifyEntityAttributes(String entityID, Iterator attributes) throws IdASException;

	/**
	 * Applies all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including entities, attributes, attribute meta-attributes and it's value(s), and attribute values) 
	 * to a backing data store.  These updates are applied atomically when possible, but may not be 
	 * applied atomically by Context Providers which do not support this.
	 * Update operations which will be applied are any of the following:<br>
	 * {@link IContext#addEntity(URI, String)}<br>
	 * {@link IContext#addEntity(IEntity)}<br>
	 * {@link IEntity#remove()}<br>
	 * {@link IHasAttributes#addAttribute(IAttribute)}<br>
	 * {@link IHasAttributes#addAttribute(URI)}<br>
	 * {@link IHasAttributes#removeAttribute(URI)}<br>
	 * {@link IHasAttributes#removeAttributeValue(IAttribute)}<br>
	 * {@link IHasAttributes#rremoveAttributeValue(URI, Object)}<br>
	 * {@link IAttribute#addComplexValue(URI)}<br>
	 * {@link IAttribute#addSimpleValue(URI, Object)}<br>
	 * {@link IAttribute#addValue(IAttributeValue)}<br>
	 * {@link IAttribute#addValue(URI)}<br>
	 * {@link IAttribute#remove()}<br>
	 * {@link IAttributeValue#remove()}<br>
	 * {@link ITypedValue#setData(Object)}<br>
	 * On a successful return, all updates which have been performed on this Context 
	 * or on any contained elements are said to be fully applied to the Context.  
	 * Subsequent update operations must be followed by another call to applyUpdates.
	 * On error (when an exception is thrown), the updates are said to not yet fully applied
	 * to the Context.  In this case, the caller may attempt to remedy the issue which 
	 * caused the call to applyUpdates by performing further update operations followed
	 * by another call to {@link #applyUpdates()}.  Alternately, the caller may call 
	 * {@link #cancelUpdates()} in order to cancel all updates which have not yet been applied.
	 * @throws IdASException
	 */
	public void applyUpdates() throws IdASException;	

	/**
	 * Cancels all update operations which have been called on this Context 
	 * and called on any element held by this Context 
	 * (including entities, attributes, 
	 * attribute meta-attributes and it's value(s), and attribute values).
	 * Any updates which have already been applied (by calling {@link #applyUpdates()}) are not canceled.   
	 * @throws IdASException
	 * @see #applyUpdates()
	 */
	public void cancelUpdates() throws IdASException;	
	
	/**
	 * Used to update one setting element for this context.  Oftentimes, an instance
	 * of IContext is also an {@link IConfigurableComponent}.  In this case, the new
	 * setting will typically be set (or reset) in the component's settings {@link Map}
	 * @param key setting name
	 * @param value setting value
	 * @param failUnsupported Default false. When true, the operation will fail with 
	 * {@link NotImplementedException) if the context does not support settings of this type.
	 * When true, if the context does not support settings of this type, the setting will be 
	 * ignored by the context.
	 * @throws IdASException 
	 */
	public void setComponentSetting(String key, Object value, boolean failUnsupported) throws IdASException;

	/**
	 * @see setComponentSetting(String, Object, boolean)
	 */
	public void setComponentSetting(String key, Object value) throws IdASException;
	

	
}
