/*******************************************************************************
 * Copyright (c) 2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.MultipleValuesExistException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;

/**
 * Simple username and password authorization materials which may be used in 
 * {@link IContext#open(Object)}  
 *
 */
public class AuthNNamePasswordMaterials implements IAuthNAttributesMaterials {
	private IAuthNAttributesMaterials _impl;
	private IAttribute _managedAttr;
	/**
	 */
	public static final URI ATTR_TYPE_USERNAME = URI.create(IAttributeValue.BASE_OWL_URI + "userName");
	/**
	 * Attribute URI for the attribute used to manage the secrets for this type of materials
	 */
	public static final URI ATTR_MANAGED_URI = URI.create(IAttributeValue.BASE_OWL_URI + "password");
		
	/**
	 * Constructs an AuthNNamePasswordMaterials by taking the username and password as String objects.
	 * @param context needed in order to build attributes from username and password
	 * @param username name of identity
	 * @param password password of identity
	 * @throws IdASException
	 */
	public AuthNNamePasswordMaterials(
		IContext context, 
		String username, 
		String password) throws IdASException
	{
		_impl = context.buildAuthNAttributesMaterials();
		IAttribute attr; 

		attr = addAttribute(ATTR_TYPE_USERNAME);
		attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, username);

		if (password != null)
		{
			_managedAttr = addAttribute(ATTR_MANAGED_URI);
			_managedAttr.addSimpleValue(ITypedValue.STRING_TYPE_URI, password);
		}
	}
	
	/**
	 * Returns the username.  IdAS Context Providers may also wish to access the username in IAttribute form as follows:
	 * <pre> IAttribute usernameAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_USERNAME));</pre>
	 * @return The username is String form
	 * @throws IdASException
	 */
	public String getUsername() throws IdASException {
		IAttribute attr = getAttribute(ATTR_TYPE_USERNAME);
		Iterator iter = attr.getValues();
		String ret = null;
		if (iter.hasNext()) {
			IAttributeValue val = (IAttributeValue) iter.next();
			if (val.isSimple()) {
				ret =((ISimpleAttrValue)val).getLexical();
			} else {
				throw new IdASException("Simple value expected, got complex value: " + val.toString());
			}
			if (iter.hasNext()) {
				throw new MultipleValuesExistException("Single value expected, got multiple");
			}
		}
		if (ret == null) {
			throw new IdASException("No " + ATTR_TYPE_USERNAME + " Attribute present");
		}
		return ret;
	}
	
	/**
	 * Returns the password.  IdAS Context Providers may also wish to access the password in IAttribute form as follows:
	 * <pre> IAttribute passwordAttr = thisMaterials.getAttribute(new URI(thisMaterials.ATTR_TYPE_PASSWORD));</pre>
	 * @return The password is String form
	 * @throws IdASException
	 */
	public String getPassword()  throws IdASException {
		IAttribute attr = getAttribute(ATTR_MANAGED_URI);
		Iterator iter = attr.getValues();
		String ret = null;
		if (iter.hasNext()) {
			IAttributeValue val = (IAttributeValue) iter.next();
			if (val.isSimple()) {
				ret =((ISimpleAttrValue)val).getLexical();
			} else {
				throw new IdASException("Simple value expected, got complex value: " + val.toString());
			}
			if (iter.hasNext()) {
				throw new MultipleValuesExistException("Single value expected, got multiple");
			}
		}
		if (ret == null) {
			throw new IdASException("No " + ATTR_MANAGED_URI + " attribute present");
		}
		return ret;
	}

	/**
	 * This API is used by applications in debug or auditing scenarios and is
	 * not intended to reveal the password.
	 */
	public String toString() {
		try {
			return getUsername();
		} catch (IdASException e) {
			// TODO: log this
//			e.printStackTrace();
			return null;
		}
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		return _impl.addAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		return _impl.addAttribute(copyFrom);
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		return _impl.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		return _impl.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		return _impl.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID)  throws IdASException{
		_impl.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)  throws IdASException{
		_impl.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr)  throws IdASException{
		_impl.removeAttributeValue(attr);
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		return _impl.equals(attributes);
	}

	public IAttribute getManagedAttr() {
		return _managedAttr;
	}

	public URI getManagedAttrID() {
		return ATTR_MANAGED_URI;
	}

}
