/*******************************************************************************
 * Copyright (c) 2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/
package org.eclipse.higgins.idas.common;

import java.net.URI;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IExtension;

/**
 * Specifies a set of attribute IDs and/or attribute value types which,
 * when encountered during certain IdAS operations will be examined for
 * values which are pointers to other resources, and in which case, will 
 * be dereferenced.<br>
 * Operations that this extension is valid for include: {@link IAttribute#getValues(IExtension[])}
 */
public class DereferenceAttrValuesExtension implements IExtension {
	public static final String id = "http://www.eclipse.org/higgins/idas/extension/derefattrvals";
	
	private URI[] attrs = null;
	private URI[] types = null;
	private boolean failUnsupported;
	
	/**
	 * 
	 * @param attrIDs may be null. When null, any attribute that is dereferenceable should be dereferenced.  
	 * When not null, specifies only those attributes which should be dereferenced 
	 * @param types may be null.  When null, any value type that is dereferenceable should be dereferenced.  
	 * When not null, specifies only those value types which should be dereferenced 
	 * @param failIfUnsupported true if the caller wishes the operation being passed this 
	 * extension to fail when it doesn't support the extension. 
	 */
	public DereferenceAttrValuesExtension(URI[] attrIDs, URI[] types, boolean failIfUnsupported) {
		this.attrs = attrIDs;
		this.types = types;
		this.failUnsupported = failIfUnsupported;
	}
	
	public URI getID() {
		return URI.create(id);
	}

	public boolean failIfUnsupported() {
		return failUnsupported;
	}
	
	/**
	 * 
	 * @return true if the passed attribute ID should be dereferenced.
	 */
	public boolean shouldDereference(URI attrID, URI valType) {
		boolean derefAttr = false;
		boolean derefVal = false;
		if (attrs != null) {
			for (int loop = 0; loop < attrs.length; loop++) {
				if (attrs[loop].compareTo(attrID) == 0) {
					// This attrID should be dereferenced
					derefAttr = true;
					break;
				}
			}
		} else {
			derefAttr = true;
		}
		if (types != null) {
			for (int loop = 0; loop < types.length; loop++) {
				if (types[loop].compareTo(valType) == 0) {
					// This val type should be dereferenced
					derefVal = true;
					break;
				}
			}
		} else {
			derefVal = true;
		}
		return derefAttr & derefVal;
	}

}
