/*******************************************************************************
 * Copyright (c) 2006 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.eclipse.higgins.idas.cp.jena2.IFactoryConfiguration;
import org.eclipse.higgins.idas.cp.jena2.util.ConfigUtil;

import com.hp.hpl.jena.ontology.OntDocumentManager;

public abstract class AbstractContextFactory implements IContextFactory {
	private Log log = LogFactory.getLog(AbstractContextFactory.class);
	protected String id = null;

	protected String name = null;

	protected String policy = null;

	protected File configFile = null;

	protected IFactoryConfiguration config = null;

	protected String cacheLocation = null;

	protected OntDocumentManager documentManager = null;

	protected AbstractContextFactory(String id, String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::AbstractContextFactory");
		setID(id);
		setName(name);
/*
  		String cfgDir = System.getProperty("user.home") + File.separator + ".higgins" + File.separator + ".idas";
		File configFile = new File(cfgDir + File.separator + id + ".ini");
		initialize(configFile);
*/
	}

	public String getID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getID");
		return id;
	}

	protected void setID(String id) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setID");
		this.id = id;
	}

	public String getName() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getName");
		if (config != null) {
			return name;
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	protected void setName(String name) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setName");
		this.name = name;
	}

	public void initialize(File configFile) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::initialize");
		System.out.println(this.getClass().getName() + " Loading configuration from: \"" + configFile + "\"");
		this.configFile = configFile;
		config = new FactoryConfiguration(getID());
		config.load(configFile);
		cacheLocation = config.getCachePath();
		documentManager = OntDocumentManager.getInstance();
		documentManager.clearCache();
		documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/higgins", ConfigUtil.getCacheFileName(
				"http://www.eclipse.org/higgins/ontologies/2006/higgins", cacheLocation));
		documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/higgins.owl", ConfigUtil.getCacheFileName(
				"http://www.eclipse.org/higgins/ontologies/2006/higgins.owl", cacheLocation));
		documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/display-data", ConfigUtil.getCacheFileName(
				"http://www.eclipse.org/higgins/ontologies/2006/display-data", cacheLocation));
	}

	public void destroyContext(URI contextRef) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::destroyContext");
		if (config != null) {
			if (config.getContextConfig(contextRef) != null) {
				config.removeContextConfig(contextRef);
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public List getConfigPropNames() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getConfigPropNames");
		if (config != null) {
			List list = new ArrayList();
			for (Enumeration e = config.getDefaultProperties().propertyNames(); e.hasMoreElements();) {
				String s = (String) e.nextElement();
				list.add(s);
			}
			return list;
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public Properties getDefaultConfig() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getDefaultConfig");
		if (config != null) {
			return config.getDefaultProperties();
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public boolean canCreate(URI contextRef, Properties configData) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::canCreate(URI, Properties)");
		if (config != null) {
			if (config.getContextConfig(contextRef) != null) {
				throw new IdASException("Context " + contextRef + " is already configured with different configuration data.");
			}

			return isValidProperties(configData);
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public boolean canCreate(URI contextRef) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::canCreate(URI)");
		if (config != null) {
			if (config.getContextConfig(contextRef) != null) {
				return true;
			} else {
				// TODO check whether it possible to load configuration using
				// contextRef
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
		return false;
	}

	public IContext createContext(URI contextRef, Properties configData) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::createContext(URI, Properties)");
		IContext ctx = null;
		if (config != null) {
			IContextConfiguration cfg = config.getContextConfig(contextRef);
			if (cfg != null) {
				throw new IdASException("Context " + contextRef + " is already configured with different configuration data.");
			}
			if (!isValidProperties(configData)) {
				throw new IdASException("Context " + contextRef + " couldn't be created - invalid configuration data.");
			}
			cfg = config.createContextConfig(contextRef, configData);
			ctx = createContext(cfg);
			if (ctx != null) {
				config.setContextConfig(cfg);
				config.save(configFile);
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
		return ctx;
	}

	public IContext createContext(URI contextRef) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::createContext(URI)");
		IContext ctx = null;
		if (config != null) {
			IContextConfiguration cfg = config.getContextConfig(contextRef);
			if (cfg == null) {
				// TODO try to load configuration data using contextRef
			}
			if (cfg == null) {
				throw new IdASException("Context " + contextRef + "couldn't be created - can't find configuration data.");
			}
			ctx = createContext(cfg);
		} else {
			throw new IdASException("Factory hasn't been initialized properly.");
		}
		return ctx;
	}

	public Iterator getContexts(String filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getContexts");
		if (config != null) {
			return config.getContextRefs().iterator();
		} else {
			throw new IdASException("Factory hasn't been initialized properly.");
		}
	}

	public String getPolicy() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getPolicy");
		return policy;
	}

	public void setPolicy(String policy) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setPolicy");
		this.policy = policy;
	}

	protected boolean isValidProperties(Properties props) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::isValidProperties");
		boolean res = false;
		if (config != null) {
			res = true;
			Properties defaults = config.getDefaultProperties();
			Iterator itr = props.keySet().iterator();
			while (itr.hasNext()) {
				if (!defaults.containsKey(itr.next())) {
					res = false;
					break;
				}
			}
		}
		return res;
	}

	protected abstract IContext createContext(IContextConfiguration c) throws IdASException;
}
