/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.cp.jena2.util.HigginsVocabulary;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.api.model.IAttributeModel;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.NodeIterator;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class Attribute implements IAttribute {
	private Log log = LogFactory.getLog(Attribute.class);

	protected OntProperty property_ = null;

	protected Context context_ = null;

	protected IAttributeModel attModel_ = null;

	protected OntClass ownerClass_ = null;

	protected OntClass valueClass_ = null;

	protected boolean isSimple_ = false;

	// the subject which owns this attribute (directly or through any level of complex value)
	protected DigitalSubject subject_ = null;

	protected Individual owner_ = null;

	protected OntClass metadataHolderClass_ = null;

	protected Individual metadataHolder_ = null;

	protected URI type_ = null;

	protected Attribute(Context context, DigitalSubject subject, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::Attribute(..., DigitalSubject, ...)");
		if (subject == null)
			throw new IdASException("Parameter \"subject\" is null.");
		subject_ = subject;
		init(context, subject.getSubjectIndividual(), property, model);
	}

	protected Attribute(Context context, ComplexValue valueOwner, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::Attribute(..., ComplexValue, ...)");
		if (valueOwner == null)
			throw new IdASException("Parameter \"valueOwner\" is null.");
		subject_ = valueOwner.getSubjectOwner();
		 init(context, valueOwner.getValueIndividual(), property, model);
	}
	
	private void init(Context context, Individual owner, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::init");
		if (context == null)
			throw new IdASException("Parameter \"context\" is null.");
		if (owner == null)
			throw new IdASException("Parameter \"owner\" is null.");
		if (property == null)
			throw new IdASException("Parameter \"property\" is null.");
		context_ = context;
		owner_ = owner;
		property_ = property;
		ownerClass_ = context_.getOntClass(owner_.getRDFType().getURI());
		metadataHolderClass_ = ModelUtils.getAttributeMetadataHolderClass(context, property);
		attModel_ = model;
		checkAttribute();
		initValueClass();
	}

	/**
	 * @throws IdASException
	 */
	private void checkAttribute() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::checkAttribute");
		String uri = property_.getURI();
		OntProperty prop = context_.getOntProperty(uri);
		OntProperty attrProp = context_.getOntProperty(HigginsVocabulary.attribute);
		if (prop == null)
			throw new IdASException("Can not find property with URI " + uri);
		if (attrProp == null)
			throw new IdASException("Can not find property with URI " + HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyRelative(attrProp, prop) == false)
			throw new IdASException("Property with URI " + uri + " is not a subproperty of " + HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyOfClass(context_, ownerClass_, prop) == false)
			throw new IdASException("Can not create attribute with type " + uri + " for DigitalSubject with class " + ownerClass_.getURI());
	}

	/**
	 * @throws IdASException
	 */
	private void initValueClass() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::initValueClass");
		OntResource range = property_.getRange();
		if (range.isClass()) {
			valueClass_ = (OntClass) range.as(OntClass.class);
			OntClass cvClass = context_.getOntClass(HigginsVocabulary.ComplexValue);
			if (cvClass == null)
				throw new IdASException("Can not find property with URI " + HigginsVocabulary.ComplexValue);
			if (ModelUtils.isClassRelative(cvClass, valueClass_)) {
				isSimple_ = false;
				return;
			} else {
				OntClass svClass = context_.getOntClass(HigginsVocabulary.SimpleValue);
				if (svClass == null)
					throw new IdASException("Can not find property with URI " + HigginsVocabulary.SimpleValue);
				if (ModelUtils.isClassRelative(svClass, valueClass_)) {
					isSimple_ = true;
				} else
					throw new IdASException("Class " + valueClass_.getURI() + " is not a subclass of classes " + HigginsVocabulary.SimpleValue + " or "
							+ HigginsVocabulary.ComplexValue);
			}
		} else
			throw new IdASException("Couldn't get range class for attribute property: " + property_.getURI());
	}

	/**
	 * @return <code>Individual</code> of <code>DigitalSubject</code> or <code>ComplexValue</code> which owns this attribure
	 */
	public Individual getOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getOwner");
		return owner_;
	}

	/**
	 * @return Ontology class of Individual of DigitalSubject or ComplexValue which immediately owns this attribure
	 */
	public OntClass getOwnerClass() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getOwnerClass");
		return ownerClass_;
	}

	/**
	 * @return DigitalSubject which owns this attribute (directly or through any level of complex value)
	 */
	public DigitalSubject getSubjectOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getSubjectOwner");
		return subject_;
	}

	/**
	 * @return ObjectProperty of this attribute
	 */
	public DigitalSubject getAttributeProperty() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getSubjectOwner");
		return subject_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addComplexValue(java.net.URI)
	 */
	public IComplexAttrValue addComplexValue(URI type) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addComplexValue");
		if (isSimple_ == false) {
			context_.registerChangedSubject(subject_);
			Individual newValue = context_.getModel().createIndividual(valueClass_);
			owner_.addProperty(property_, newValue);
			return new ComplexValue(context_, this, newValue, property_);
		} else
			throw new IdASException("Can not create complex value for simple attribute with type " + getAttrID().toString());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addSimpleValue(java.net.URI, java.lang.Object)
	 */
	public ISimpleAttrValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addSimpleValue");
		if (isSimple_ == true) {
			context_.registerChangedSubject(subject_);
			Individual newValue = context_.getModel().createIndividual(valueClass_);
			owner_.addProperty(property_, newValue);
			SimpleValue sv = new SimpleValue(context_, this, newValue, property_);
			sv.setData(data);
			return sv;
		} else
			throw new IdASException("Can not create simple value for complex attribute with type " + getAttrID().toString());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(java.net.URI)
	 */
	public IAttributeValue addValue(URI type) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addValue(URI)");
		if (isSimple_)
			return addSimpleValue(type, null);
		else
			return addComplexValue(type);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(org.eclipse.higgins.idas.api.IAttributeValue)
	 */
	public IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addValue");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		URI newType = copyFrom.getType();
		if (newType == null)
			throw new IdASException("Type of passed value \"copyFrom\" is null.");
		if (newType.equals(this.getAttrID()) == false)
			throw new IdASException("Can not create value with type " + newType.toString() + " for attribute with type " + getAttrID().toString());
		if (isSimple_) {
			if (copyFrom instanceof ISimpleAttrValue) {
				ISimpleAttrValue from = (ISimpleAttrValue) copyFrom;
				ISimpleAttrValue newVal = addSimpleValue(newType, from.getData());
				Iterator itrMtd = from.getMetadataSet();
				if (itrMtd != null) {
					while (itrMtd.hasNext()) {
						IMetadata meta = (IMetadata) itrMtd.next();
						newVal.addMetadata(meta);
					}
				}
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of ISimpleValue");
		} else {
			if (copyFrom instanceof IComplexAttrValue) {
				IComplexAttrValue from = (IComplexAttrValue) copyFrom;
				IComplexAttrValue newVal = addComplexValue(newType);
				Iterator itrAttr = from.getAttributes();
				if (itrAttr != null) {
					while (itrAttr.hasNext()) {
						IAttribute attr = (IAttribute) itrAttr.next();
						newVal.addAttribute(attr);
					}
				}
				Iterator itrMtd = from.getMetadataSet();
				if (itrMtd != null) {
					while (itrMtd.hasNext()) {
						IMetadata meta = (IMetadata) itrMtd.next();
						newVal.addMetadata(meta);
					}
				}
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of IComplexValue");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getAttrID()
	 */
	public URI getAttrID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getAttrID");
		if (type_ == null)
			type_ = URI.create(property_.getURI());
		if (type_ == null)
			throw new IdASException("Can not create URI for property " + property_.getURI());
		return type_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getModel()
	 */
	public IAttributeModel getModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getModel");
		return attModel_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#getValues()
	 */
	public Iterator getValues() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getValues");
		ArrayList list = new ArrayList();
		OntClass cvClass = context_.getOntClass(HigginsVocabulary.ComplexValue);
		if (cvClass == null)
			throw new IdASException("Can not find property with URI " + HigginsVocabulary.ComplexValue);
		OntClass svClass = context_.getOntClass(HigginsVocabulary.SimpleValue);
		if (svClass == null)
			throw new IdASException("Can not find property with URI " + HigginsVocabulary.SimpleValue);
		NodeIterator itr = owner_.listPropertyValues(property_);
		while (itr.hasNext()) {
			RDFNode node = itr.nextNode();
			if (node != null) {
				if (node.isLiteral() == false) {
					Individual ind = (Individual) node.as(Individual.class);
					OntClass cls = context_.getOntClass(ind.getRDFType().getURI());
					if (ModelUtils.isClassRelative(cvClass, cls)) {
						list.add(new ComplexValue(context_, this, ind, property_));
					} else if (ModelUtils.isClassRelative(svClass, cls)) {
						list.add(new SimpleValue(context_, this, ind, property_));
					}
				}
			}
		}
		return list.iterator();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#isSingleValued()
	 */
	public boolean isSingleValued() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::isSingleValued");
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttribute#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::remove");
		ArrayList list = new ArrayList();
		NodeIterator itr = owner_.listPropertyValues(property_);
		while (itr.hasNext()) {
			RDFNode node = itr.nextNode();
			list.add(node);
		}
		for (int i = 0; i < list.size(); i++) {
			RDFNode node = (RDFNode) list.get(i);
			owner_.removeProperty(property_, node);
			if (node.isLiteral() == false) {
				Individual ind = (Individual) node.as(Individual.class);
				ModelUtils.removeIndividual(context_, ind);
			}
		}
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	protected Individual findMetadataContainer() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::findMetadataContainer");
		if (metadataHolder_ == null) {
			OntClass attrClass = context_.getOntClass(HigginsVocabulary.Attribute);
			NodeIterator itr = owner_.listPropertyValues(property_);
			while (itr.hasNext()) {
				RDFNode node = itr.nextNode();
				if (node != null) {
					if (node.isLiteral() == false) {
						Individual ind = (Individual) node.as(Individual.class);
						OntClass cls = context_.getOntClass(ind.getRDFType().getURI());
						if (ModelUtils.isClassRelative(attrClass, cls))
							metadataHolder_ = ind;
					}
				}
			}
		}
		return metadataHolder_;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	protected Individual findOrCreateMetadataContainer() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::findOrCreateMetadataContainer");
		if (metadataHolder_ == null) {
			metadataHolder_ = findMetadataContainer();
			if (metadataHolder_ == null) {
				metadataHolder_ = context_.getModel().createIndividual(metadataHolderClass_);
				owner_.addProperty(property_, metadataHolder_);
			}
		}
		return metadataHolder_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(java.net.URI)
	 */
	public IMetadata addMetadata(URI metadataID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addMetadata(URI)");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		IMetadata newMetadata = getMetadata(metadataID);
		return newMetadata;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(org.eclipse.higgins.idas.api.IMetadata)
	 */
	public IMetadata addMetadata(IMetadata copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addMetadata");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IMetadata newMetadata = getMetadata(copyFrom.getID());
		newMetadata.setData(copyFrom.getData());
		return newMetadata;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadata(java.net.URI)
	 */
	public IMetadata getMetadata(URI metadataID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getMetadata");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		String uri = metadataID.toString();
		DatatypeProperty prop = context_.getDatatypeProperty(uri);
		if (prop == null)
			throw new IdASException("Can not find datatype property with URI " + uri);
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyRelative(metadata, prop) == false)
			throw new IdASException("Property with URI " + uri + " is not a subproperty of " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyOfClass(context_, valueClass_, prop) == false)
			throw new IdASException("Can not add metadata with type " + uri + " for class " + valueClass_.getURI());
		Individual metadataHolder = findOrCreateMetadataContainer();
		return new Metadata(context_, metadataHolder, prop, subject_);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadataSet()
	 */
	public Iterator getMetadataSet() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getMetadataSet");
		ArrayList list = new ArrayList();
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		Individual metadataHolder = findMetadataContainer();
		if (metadataHolder != null) {
			StmtIterator props = metadataHolder.listProperties();
			while (props.hasNext()) {
				Statement st = props.nextStatement();
				Property prop = st.getPredicate();
				DatatypeProperty dpProperty = context_.getModel().getDatatypeProperty(prop.getURI());
				if (dpProperty != null) {
					if (ModelUtils.isPropertyRelative(metadata, dpProperty)) {
						IMetadata md = new Metadata(context_, metadataHolder, dpProperty, subject_);
						list.add(md);
					}
				}
			}
		}
		return list.iterator();
	}

	public boolean equals(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

}
