/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.cp.jena2.util.HigginsVocabulary;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.model.impl.AttributeComplexValueModel;
import org.eclipse.higgins.idas.model.impl.AttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class ComplexValue implements IComplexAttrValue {
	private Log log = LogFactory.getLog(ComplexValue.class);
	protected OntProperty property_ = null;

	protected Context context_ = null;

	protected Attribute attribute_ = null;

	// the subject which owns this complex value (through any level of attribute/complex value)
	protected DigitalSubject subject_ = null;

	protected Individual value_ = null;

	protected OntClass valueClass_ = null;

	protected AttributeComplexValueModel valueModel_ = null;

	protected Individual ownerOfAttribute_ = null;

	protected URI type_ = null;

	/**
	 * @param context
	 * @param attribute
	 * @param container
	 * @param attributeProperty
	 * @param subjOwner
	 * @throws IdASException
	 */
	public ComplexValue(Context context, Attribute attribute, Individual container, OntProperty attributeProperty) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::ComplexValue");
		if (context == null)
			throw new IdASException("Parameter \"context\" is null.");
		if (attribute == null)
			throw new IdASException("Parameter \"attribute\" is null.");
		if (container == null)
			throw new IdASException("Parameter \"container\" is null.");
		context_ = context;
		attribute_ = attribute;
		value_ = container;
		ownerOfAttribute_ = attribute.getOwner();
		subject_ = attribute.getSubjectOwner();
		valueClass_ = context_.getOntClass(value_.getRDFType().getURI());
		property_ = attributeProperty;
		OntClass cvClass = context_.getOntClass(HigginsVocabulary.ComplexValue);
		ModelUtils.isClassRelative(cvClass, valueClass_);
		IAttributeModel attrModel = attribute_.getModel();
		if (attrModel != null) {
			IAttributeValueModel valueModel = attrModel.getValueModel();
			if (valueModel instanceof IAttributeComplexValueModel)
				valueModel_ = (AttributeComplexValueModel) valueModel;
		}
	}

	/**
	 * @return <code>DigitalSubject</code> which owns this attribute (directly or through any level of complex value)
	 */
	public DigitalSubject getSubjectOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getSubjectOwner");
		return subject_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#getType()
	 */
	public URI getType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getType");
		if (type_ == null)
			type_ = URI.create(valueClass_.getURI());
		return type_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#isSimple()
	 */
	public boolean isSimple() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::isSimple");
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::remove");
		context_.registerChangedSubject(subject_);
		ownerOfAttribute_.removeProperty(property_, value_);
		ModelUtils.removeIndividual(context_, value_);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(java.net.URI)
	 */
	public IMetadata addMetadata(URI metadataID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addMetadata(URI)");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		IMetadata newMetadata = getMetadata(metadataID);
		return newMetadata;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(org.eclipse.higgins.idas.api.IMetadata)
	 */
	public IMetadata addMetadata(IMetadata copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addMetadata");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IMetadata newMetadata = getMetadata(copyFrom.getID());
		newMetadata.setData(copyFrom.getData());
		context_.registerChangedSubject(subject_);
		return newMetadata;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadata(java.net.URI)
	 */
	public IMetadata getMetadata(URI metadataID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getMetadata");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		String uri = metadataID.toString();
		DatatypeProperty prop = context_.getDatatypeProperty(uri);
		if (prop == null)
			throw new IdASException("Can not find datatype property with URI " + uri);
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyRelative(metadata, prop) == false)
			throw new IdASException("Property with URI " + uri + " is not a subproperty of " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyOfClass(context_, valueClass_, prop) == false)
			throw new IdASException("Can not add metadata with type " + uri + " for class " + valueClass_.getURI());
		return new Metadata(context_, value_, prop, subject_);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadataSet()
	 */
	public Iterator getMetadataSet() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getMetadataSet");
		ArrayList list = new ArrayList();
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		StmtIterator props = value_.listProperties();
		while (props.hasNext()) {
			Statement st = props.nextStatement();
			Property prop = st.getPredicate();
			DatatypeProperty dpProperty = context_.getModel().getDatatypeProperty(prop.getURI());
			if (dpProperty != null) {
				if (ModelUtils.isPropertyRelative(metadata, dpProperty)) {
					IMetadata md = new Metadata(context_, value_, dpProperty, subject_);
					list.add(md);
				}
			}
		}
		return list.iterator();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addAttribute(URI)");
		return getAttribute(attrID);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addAttribute");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IAttribute newAttr = getAttribute(copyFrom.getAttrID());
		Iterator itrVal = copyFrom.getValues();
		context_.registerChangedSubject(subject_);
		while (itrVal.hasNext()) {
			IAttributeValue value = (IAttributeValue) itrVal.next();
			newAttr.addValue(value);
		}
		Iterator itrMd = copyFrom.getMetadataSet();
		while (itrMd.hasNext()) {
			IMetadata metadata = (IMetadata) itrMd.next();
			newAttr.addMetadata(metadata);
		}
		return newAttr;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getAttribute");
		if (attrID == null)
			throw new IdASException("Parameter \"attrID\" is null.");
		String uri = attrID.toString();
		OntProperty op = context_.getOntProperty(uri);
		ObjectProperty attrProp = context_.getObjectProperty(HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyRelative(attrProp, op) == false)
			throw new InvalidTypeException("Property " + uri + " is not subproperty of " + HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyOfClass(context_, valueClass_, op) == false)
			throw new InvalidTypeException("Can not create attribute with type " + uri + " for ComplexValue with type " + valueClass_.getURI());
		AttributeModel attrModel = (AttributeModel)valueModel_.getAttribute(attrID);
		if (attrModel == null)
			throw new IdASException("Can not get model of attribute with type " + uri + " for ComplexValue with type " + valueClass_.getURI());
		int maxCardinality = ModelUtils.getMaxCardinality(context_, valueClass_, op);
		if (maxCardinality == 1)
			return new SingleValuedAttribute(context_, this, op, attrModel);
		else
			return new Attribute(context_, this, op, attrModel);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getAttributes");
		ObjectProperty attrProp = context_.getObjectProperty(HigginsVocabulary.attribute);
		HashSet usedURIs = new HashSet();
		List attrs = new ArrayList();
		StmtIterator props = value_.listProperties();
		while (props.hasNext()) {
			Statement st = props.nextStatement();
			Property prop = st.getPredicate();
			String uri = prop.getURI();
			ObjectProperty property = context_.getModel().getObjectProperty(uri);
			if (property != null) {
				if ((usedURIs.contains(uri) == false) && ModelUtils.isPropertyRelative(attrProp, property)) {
					AttributeModel attrModel = (AttributeModel)valueModel_.getAttribute(URI.create(uri));
					if (attrModel == null)
						throw new IdASException("Can not get model of attribute with type " + uri + " for DigitalSubject with type " + valueClass_.getURI());
					int maxCardinality = ModelUtils.getMaxCardinality(context_, valueClass_, property);
					if (maxCardinality == 1)
						attrs.add(new SingleValuedAttribute(context_, this, property, attrModel ));
					else
						attrs.add(new Attribute(context_, this, property, attrModel));
					usedURIs.add(uri);
				}
			}
		}
		return attrs.iterator();
	}

	/**
	 * @return Individual of this value
	 * @throws IdASException
	 */
	public Individual getValueIndividual() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueIndividual");
		return value_;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public OntClass getValueClass() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueClass");
		return valueClass_;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public AttributeComplexValueModel getValueModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueModel");
		return valueModel_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getSingleValuedAttribute(java.net.URI)
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getSingleValuedAttribute");
		String idURI = attrID.toString();
		IAttribute attr = getAttribute(attrID);
		if (attr == null)
			return null;
		if (attr instanceof ISingleValuedAttribute == false)
			throw new IdASException("Attribute " + idURI + " is not an instance of ISingleValuedAttribute");
		return (ISingleValuedAttribute) attr;
	}

	public boolean equals(IAttributeValue value) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		
	}

}
