/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.authentication;

import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.cp.jena2.IAuthenticationModule;
import org.eclipse.higgins.idas.cp.jena2.IUserAccount;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.RDFNode;

public class AuthenticationModule implements IAuthenticationModule {
	private Log log = LogFactory.getLog(AuthenticationModule.class);

	protected OntModel model_ = null;

	protected OntClass accountClass_ = null;

	protected OntProperty nameProperty_ = null;

	protected OntProperty passwordHashProperty_ = null;

	protected OntProperty tokenProperty_ = null;

	protected Context context_ = null;

	public AuthenticationModule(Context context) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::AuthenticationModule");
		if (context == null)
			throw new IdASException("The parameter \"model\" is null");
		context_ = context;
		model_ = context.getModelNoException();
		accountClass_ = initClass(AuthConstants.USER_ACCOUNT_CLASS);
		nameProperty_ = initProperty(AuthConstants.USER_NAME_PROPERTY);
		passwordHashProperty_ = initProperty(AuthConstants.USER_PASSWORD_HASH_PROPERTY);
		tokenProperty_ = initProperty(AuthConstants.USER_TOKEN_PROPERTY);
	}

	public IUserAccount authenticate(Object credentials) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::authenticate");
		if (credentials instanceof AuthNNamePasswordMaterials) {
			AuthNNamePasswordMaterials am = (AuthNNamePasswordMaterials) credentials;
			String name = am.getUsername();
			if (name == null)
				throw new IdASException("AuthNNamePasswordMaterials contains null username.");
			name = name.trim();
			if (name.length() == 0)
				throw new IdASException("AuthNNamePasswordMaterials contains empty string username.");
			String password = am.getPassword();
			ArrayList list = getSubjectIndividuals(name);
			Individual userInd = null;
			switch (list.size()) {
			case 0: {
				try {
					userInd = createNewAccount(name, password);
					context_.applyUpdates();
				} catch (Exception e) {
					log.error(e);
					context_.cancelUpdates();
					throw new IdASException();
				}
				break;
			}
			case 1: {
				try {
					userInd = (Individual) list.get(0);
				} catch (Exception e) {
					log.error(e);
					throw new IdASException(e.getMessage());
				}
				checkPassword(userInd, password);
				break;
			}
			default:
				throw new IdASException("There are more then one user with name = " + name);
			}
			return new PasswordBasedUserAccount(model_, userInd, am);
		} else
			throw new IdASException("Unsupported credentials type.");
	}

	private Individual createNewAccount(String name, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::createNewAccount");
		Individual userInd = model_.createIndividual(accountClass_);
		Literal nameLtr = model_.createLiteral(name);
		userInd.setPropertyValue(nameProperty_, nameLtr);
		String token = name;
		Literal tokenLtr = model_.createLiteral(token);
		userInd.setPropertyValue(tokenProperty_, tokenLtr);
		setNewPassword(userInd, password);
		return userInd;
	}

	private void setNewPassword(Individual user, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::setNewPassword");
		String newHash = null;
		byte[] bytes = getHash(password);
		if (bytes != null)
			newHash = Base64.encode(bytes);
		if (newHash != null) {
			Literal ltr = model_.createLiteral(newHash);
			user.setPropertyValue(passwordHashProperty_, ltr);
		} else {
			RDFNode node = user.getPropertyValue(passwordHashProperty_);
			if (node != null)
				user.removeProperty(passwordHashProperty_, node);
		}
	}

	private void checkPassword(Individual userInd, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::checkPassword");
		String passHash = null;
		RDFNode rdfNode = userInd.getPropertyValue(passwordHashProperty_);
		if (rdfNode.isLiteral() == false)
			throw new IdASException("Value of property " + passwordHashProperty_.getURI() + " is not literal.");
		Literal litr = (Literal) rdfNode.as(Literal.class);
		Object obj = litr.getValue();
		if (obj != null)
			passHash = obj.toString();
		if (passHash == null || passHash.length() == 0) {
			if (password != null && password.length() > 0)
				throw new IdASException("Wrong password.");
		} else {
			if (password == null || password.length() == 0)
				throw new IdASException("Wrong password.");
			byte[] realPassHash = null;
			byte[] testedHash = null;
			testedHash = getHash(password);
			try {
				realPassHash = Base64.decode(passHash);
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e.getMessage());
			}
			if (Arrays.equals(realPassHash, testedHash) == false)
				throw new IdASException("Wrong password.");
		}
	}

	private OntClass initClass(String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::initClass");
		OntClass ontClass = model_.getOntClass(name);
		if (ontClass == null) {
			try {
				ontClass = model_.createClass(name);
				context_.applyUpdates();
			} catch (Exception e) {
				log.error(e);
				context_.cancelUpdates();
				throw new IdASException();
			}
		}
		return ontClass;
	}

	private OntProperty initProperty(String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::initProperty");
		OntProperty prop = model_.getOntProperty(name);
		if (prop == null) {
			try {
				prop = model_.createDatatypeProperty(name);
				context_.applyUpdates();
			} catch (Exception e) {
				log.error(e);
				context_.cancelUpdates();
				throw new IdASException();
			}
		}
		return prop;
	}

	private byte[] getHash(String str) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::getHash");
		if (str != null && str.length() > 0) {
			try {
				MessageDigest md = MessageDigest.getInstance("SHA-256");
				return md.digest(str.getBytes("UTF-8"));
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e.getMessage());
			}
		} else
			return null;
	}

	public ArrayList getSubjectIndividuals(String userName) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::getSubjectIndividuals");
		String query =
				"SELECT ?ind \n" +
				"WHERE { \n" +
				"  ?ind <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <" + AuthConstants.USER_ACCOUNT_CLASS + ">. \n" +
				"  ?ind <" + AuthConstants.USER_NAME_PROPERTY + "> \"" + userName  + "\" \n" +
				"}";
		log.debug(query);
		ArrayList list = new ArrayList();
		Query q = QueryFactory.create(query);
		QueryExecution qexec = QueryExecutionFactory.create(q, context_.getModelNoException());
		try {
			ResultSet results = qexec.execSelect();
			while (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get("ind");
				Individual digitalSubject = (Individual) a.as(Individual.class);
				list.add(digitalSubject);
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		} finally {
			qexec.close();
		}
		log.debug("Users count by query: " + list.size());
		return list;
	}

}
