/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.filter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;


public class ValueCondition implements ICondition {
	private Log log = LogFactory.getLog(ValueCondition.class);

	private LiteralNode node_;

	private String operator_;

	private ITypedValue value_;
	
	private int level_;

	public ValueCondition(LiteralNode node, String operator, ITypedValue value) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::ValueCondition");
		node_ = node;
		operator_ = operator;
		if (isValueRequired() == false) {
			if (value == null)
				throw new IdASException("Parameter \"value\" is null.");
			if (value.getData() == null)
				throw new IdASException("Data of value is null.");
			if (value.getType() == null)
				throw new IdASException("Type of value is null.");
		}
		value_ = value;
	}

	private boolean isValueRequired() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::isValueRequired");
		if (IFilterAttributeAssertion.COMP_ATTR_PRESENT.equals(operator_))
			return false;
		else
			return true;
	}

	private void populateCondition(StringBuffer buf, String operator) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::populateCondition");
		buf.append(node_.asQueryElement());
		buf.append(operator);
		buf.append("\"");
		buf.append(value_.getLexical());
		buf.append("\"^^");
		buf.append(getXMLTypeSuffix(value_.getType().toString()));
	}

	private String getXMLTypeSuffix(String xmlType) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::getXMLTypeSuffix");
		if (xmlType.startsWith("http://www.w3.org/2001/XMLSchema#"))
			return xmlType.replaceFirst("http://www.w3.org/2001/XMLSchema#", "xsd:");
		else
			return "<" + xmlType + ">";
	}

	public StringBuffer getCondition() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::getCondition");
		StringBuffer buf = new StringBuffer();
		if (level_ == 0)
			buf.append("(");
		if (IFilterAttributeAssertion.COMP_ATTR_SUBSTR.equals(operator_)) {
			buf.append("regex(?");
			buf.append(node_.asQueryElement());
			buf.append(", \"^");
			buf.append(value_.getLexical());
			buf.append("\")");
		} 
		else if (IFilterAttributeAssertion.COMP_ATTR_PRESENT.equals(operator_)) {
			buf.append("bound(");
			buf.append(node_.asQueryElement());
			buf.append(")");
		}
		else if (IFilterAttributeAssertion.COMP_ATTR_EQ.equals(operator_))
			populateCondition(buf, " = ");
		else if (IFilterAttributeAssertion.COMP_ATTR_GE.equals(operator_))
			populateCondition(buf, " >= ");
		else if (IFilterAttributeAssertion.COMP_ATTR_LE.equals(operator_))
			populateCondition(buf, " <= ");
		else if (IFilterAttributeAssertion.COMP_ATTR_EQ.equals(operator_))
			populateCondition(buf, " = ");
		else
			throw new IdASException("Unknown condition : " + operator_);
		if (level_ == 0)
			buf.append(")");
		return buf;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.filter.ICondition#setLevel(int)
	 */
	public void setLevel(int level) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.ValueCondition::setLevel");
		level_ = level;
	}

}
