/**
 * Copyright (c) 2006 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.util;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IDigitalSubjectModel;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.impl.Attribute;
import org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;
import org.eclipse.higgins.idas.cp.jena2.impl.DigitalSubject;
import org.eclipse.higgins.idas.cp.jena2.impl.Metadata;
import org.eclipse.higgins.idas.cp.jena2.impl.SimpleValue;
import org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthConstants;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class ModelUtils {
	private static Log log = LogFactory.getLog(ModelUtils.class);

	/**
	 * Checks if descendant is subclass of ancestor.
	 * 
	 * @param ancestor
	 * @param descendant
	 * @return true if descendant is subclass of ancestor
	 * @throws IdASException
	 */
	public static boolean isClassRelative(OntClass ancestor, OntClass descendant) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isClassRelative");
		if (ancestor.equals(descendant))
			return true;
		try {
			Iterator itr = descendant.listSuperClasses();
			while (itr.hasNext()) {
				OntClass cls = (OntClass) itr.next();
				if (isClassRelative(ancestor, cls))
					return true;
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	/*
	 * Iterator itr = descendant.listSuperClasses(); while (itr.hasNext()) {
	 * OntClass cls = (OntClass)itr.next(); System.out.println(cls.getURI()); if
	 * (cls.equals(ancestor)) return true; } return false; }
	 * 
	 */
	/**
	 * Checks if descendant is subproperty of ancestor.
	 * 
	 * @param ancestor
	 * @param descendant
	 * @return true if descendant is subproperty of ancestor
	 * @throws IdASException
	 */
	public static boolean isPropertyRelative(OntProperty ancestor, OntProperty descendant) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isPropertyRelative");
		OntProperty tmpDescendant = descendant;
		try {
			while (tmpDescendant != null) {
				if (tmpDescendant.hasSuperProperty(ancestor, false))
					return true;
				tmpDescendant = tmpDescendant.getSuperProperty();
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	/**
	 * @param list
	 *            List of <code>OntClass</code>
	 * @param cls
	 */
	private static void fillListSuperClasses(List list, OntClass cls) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::fillListSuperClasses");
		if (cls != null && !cls.isRestriction())
			list.add(cls);
		ExtendedIterator ei = cls.listSuperClasses();
		while (ei.hasNext())
			fillListSuperClasses(list, (OntClass) ei.next());
	}

	/**
	 * @param list
	 *            List of <code>OntClass</code>
	 * @param cls
	 */
	private static void fillListSubClasses(List list, OntClass cls) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::fillListSubClasses");
		if (cls != null && !cls.isRestriction())
			list.add(cls);
		ExtendedIterator ei = cls.listSubClasses();
		while (ei.hasNext())
			fillListSubClasses(list, (OntClass) ei.next());
	}

	/**
	 * @param context
	 * @param ownerClass
	 * @return List of <code>OntClass</code> which are subClasses of
	 *         ownerClass
	 */
	public static List getListSubClasses(Context context, OntClass ownerClass) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getListSubClasses");
		List classesList = new ArrayList();
		fillListSubClasses(classesList, ownerClass);
		return classesList;
	}

	/**
	 * @param context
	 * @param ownerClass
	 * @return List of <code>OntClass</code> which are superClasses of
	 *         ownerClass
	 */
	public static List getListSuperClasses(IJenaContext context, OntClass ownerClass) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getListSuperClasses");
		List classesList = new ArrayList();
		fillListSuperClasses(classesList, ownerClass);
		return classesList;
	}

	/**
	 * Checks if ownerClass can have this property
	 * 
	 * @param model
	 * @param ownerClass
	 * @param property
	 * @return true if Individual of ownerClass can have this property
	 */
	public static boolean isPropertyOfClass(Context context, OntClass ownerClass, OntProperty property) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isPropertyOfClass");
		List classesList = new ArrayList();
		fillListSuperClasses(classesList, ownerClass);
		for (int i = 0; i < classesList.size(); i++) {
			OntClass ontClass = (OntClass) classesList.get(i);
			ExtendedIterator ei = property.listDomain();
			while (ei.hasNext()) {
				Object node = ei.next();
				if (node instanceof OntClass) {
					OntClass domain = (OntClass) node;
					if (ontClass.getURI().equals(domain.getURI()))
						return true;
				}
			}
		}
		return false;
	}

	/**
	 * @param model
	 * @param domainClass
	 *            a subclass of higgins:SimpleAttribute
	 * @return property that should be used for this domainClass
	 * @throws IdASException
	 */
	public static DatatypeProperty getSimpleValuePropertyByDomain(Context context, OntClass domainClass) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getSimpleValuePropertyByDomain");
		if (domainClass != null) {
			DatatypeProperty simpleValueProperty = context.getDatatypeProperty(HigginsVocabulary.simpleValue);
			Property domain = context.getModel().getProperty("http://www.w3.org/2000/01/rdf-schema#domain");
			StmtIterator ei = context.getModel().listStatements(null, domain, domainClass);
			while (ei.hasNext()) {
				DatatypeProperty property = context.getDatatypeProperty(ei.nextStatement().getSubject().getURI());
				if (property != null && isPropertyRelative(simpleValueProperty, property))
					return property;
			}
		}
		return null;
	}

	/**
	 * @param context
	 * @param domainClass
	 * @return
	 * @throws IdASException
	 */
	public static ArrayList getAttributePropertiesByDomain(IJenaContext context, OntClass domainClass) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getAttributePropertiesByDomain");
		ArrayList list = new ArrayList();
		List subClasses = getListSuperClasses(context, domainClass);
		OntProperty attributeProperty = context.getOntProperty(HigginsVocabulary.attribute);
		Property domain = context.getModel().getProperty("http://www.w3.org/2000/01/rdf-schema#domain");
		for (int i = 0; i < subClasses.size(); i++) {
			OntClass cls = (OntClass) subClasses.get(i);
			if (cls != null) {
				StmtIterator ei = context.getModel().listStatements(null, domain, cls);
				while (ei.hasNext()) {
					OntProperty property = context.getModel().getOntProperty(ei.nextStatement().getSubject().getURI());
					if (property != null && isPropertyRelative(attributeProperty, property))
						list.add(property);
				}
			}
		}
		return list;
	}

	/**
	 * 
	 * @param model
	 * @param cuid
	 * @return Individual of DigitalSubject by uniqueIdentifier
	 * @throws IdASException
	 */
	public static Individual getSubjectByID(Context context, String cuid) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getSubjectByID");
		Individual digitalSubject = null;
		String queryString = "PREFIX  higgins: <http://www.eclipse.org/higgins/ontologies/2006/higgins.owl#> \n"
				+ "PREFIX  xsd: <http://www.w3.org/2001/XMLSchema#> \n" + "SELECT ?ds \n WHERE { \n" + " ?ds higgins:uniqueIdentifier ?a . \n"
				+ " ?a higgins:string  \"" + cuid + "\"^^xsd:string}";
		Query query = QueryFactory.create(queryString);
		QueryExecution qexec = QueryExecutionFactory.create(query, context.getModel());
		try {
			ResultSet results = qexec.execSelect();
			if (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get("ds");
				digitalSubject = (Individual) a.as(Individual.class);
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		} finally {
			qexec.close();
		}
		return digitalSubject;
	}

	/**
	 * @param context
	 * @param attrProp
	 * @return
	 * @throws IdASException
	 */
	public static OntClass getAttributeMetadataHolderClass(Context context, OntProperty attrProp) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getAttributeMetadataHolderClass");
		OntClass attributeClass = context.getOntClass(HigginsVocabulary.Attribute);
		OntProperty curProp = attrProp;
		while (curProp != null) {
			ExtendedIterator itr = curProp.listRange();
			while (itr.hasNext()) {
				Object node = itr.next();
				if (node instanceof OntClass) {
					OntClass dClass = (OntClass) node;
					if (isClassRelative(attributeClass, dClass))
						return dClass;
				}
			}
			curProp = curProp.getSuperProperty();
		}
		throw new IdASException("Can not find attribute metadata holder class (subclass of " + HigginsVocabulary.Attribute + ") for attribute property "
				+ attrProp.getURI());
	}

	/**
	 * Remove Individual and all its property values
	 * 
	 * @param model
	 * @param ind
	 * @throws IdASException
	 */
	public static void removeIndividual(Context context, Individual ind) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::removeIndividual");
		String type = (ind.getRDFType() != null) ? ind.getRDFType().getURI() : null;
		if (type == null)
			throw new IdASException("Indiividual doesn't have RDF type");
		StmtIterator si = ind.listProperties();
		List stmts = new ArrayList(); // <Statement>
		List individuals = new ArrayList(); // <Individual>(
		while (si.hasNext()) {
			Statement st = si.nextStatement();
			RDFNode node = st.getObject();
			if (node.isLiteral())
				stmts.add(st);
			else if (node.isResource()) {
				Individual child = null;
				try {
					child = (Individual) node.as(Individual.class);
				} catch (ClassCastException e) {
					throw new IdASException(e.getMessage());
				}
				if (child != null && !type.equals(child.getURI()))
					individuals.add(child);
			}
		}
		Iterator itrStmt = stmts.iterator();
		while (itrStmt.hasNext()) {
			Statement st = (Statement) itrStmt.next();
			context.getModel().remove(st);
		}
		Iterator itrIndv = individuals.iterator();
		while (itrIndv.hasNext()) {
			Individual child = (Individual) itrIndv.next();
			removeIndividual(context, child);
		}
		ind.remove();
	}

	/**
	 * @param context
	 * @param cls
	 * @param attr
	 * @return
	 * @throws IdASException
	 */
	public static int getMaxCardinality(Context context, OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMaxCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						return cr.getMaxCardinality();
					}
				}
			} else {
				int res = getMaxCardinality(context, c, attr);
				if (res != -1)
					return res;
			}
		}
		return -1;
	}

	/**
	 * @param context
	 * @param cls
	 * @param attr
	 * @return
	 * @throws IdASException
	 */
	public static int getMinCardinality(Context context, OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMinCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(true); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						return cr.getMinCardinality();
					}
				}
			}
		}
		return -1;
	}

	/**
	 * @param context
	 * @param subjectInd
	 * @param userToken
	 * @throws IdASException
	 */
	public static void setUserToken(Context context, Individual subjectInd, String userToken) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::setUserToken");
		OntProperty tokenProp = context.getOntProperty(AuthConstants.USER_TOKEN_PROPERTY);
		if (userToken == null || userToken.length() == 0) {
			RDFNode node = subjectInd.getPropertyValue(tokenProp);
			subjectInd.removeProperty(tokenProp, node);
		} else {
			Literal ltr = context.getModel().createLiteral(userToken);
			subjectInd.setPropertyValue(tokenProp, ltr);
		}
	}

	/**
	 * @param context
	 * @param subjectInd
	 * @return
	 * @throws IdASException
	 */
	public static String getUserToken(Context context, Individual subjectInd) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getUserToken");
		OntProperty tokenProp = context.getOntProperty(AuthConstants.USER_TOKEN_PROPERTY);
		RDFNode tokenNode = subjectInd.getPropertyValue(tokenProp);
		if (tokenNode != null) {
			if (tokenNode.isLiteral()) {
				Literal ltr = (Literal) tokenNode.as(Literal.class);
				Object obj = ltr.getValue();
				if (obj != null)
					return obj.toString();
			} else
				throw new IdASException("Property " + AuthConstants.USER_TOKEN_PROPERTY + " contains non-literal value.");
		}
		return null;
	}

	/**
	 * @param context
	 * @return
	 */
	public static String generateUniqueID(Context context) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::generateUniqueID");
		String rnd = String.valueOf(Math.random()).substring(0);
		return String.valueOf(System.currentTimeMillis()) + rnd;
	}

	/**
	 * @param context
	 * @param xmlDatatype
	 * @return
	 * @throws IdASException
	 */
	public static String getSimpleValuePropertyByXMLType(Context context, URI xmlDatatype) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getSimpleValuePropertyByXMLType");
		if (xmlDatatype == null)
			return null;
		String xmlType = xmlDatatype.toString();
		if (xmlType.startsWith(HigginsVocabulary.XSD_NS)) {
			String higginsType = xmlType.replaceAll(HigginsVocabulary.XSD_NS, HigginsVocabulary.NAME_SPACE);
			DatatypeProperty simpleValue = context.getDatatypeProperty(HigginsVocabulary.simpleValue);
			DatatypeProperty dp = context.getDatatypeProperty(higginsType);
			if (dp == null)
				throw new IdASException("Can not find datatype property " + higginsType);
			if (isPropertyRelative(simpleValue, dp))
				return higginsType;
			else
				throw new IdASException("Property \"" + higginsType + "\" is not subtype of property " + HigginsVocabulary.simpleValue);
		} else
			throw new IdASException("Value \"" + xmlType + "\" is not an xml datatype property.");
	}

	/**
	 * @param context
	 * @param subj
	 * @throws IdASException
	 */
	public static void validateSubject(Context context, DigitalSubject subj) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateSubject");
		log.debug("ContextURI = " + context.getNS() + ", subject type = " + subj.getType());
		if (subj == null)
			throw new IdASException("Parameter \"subj\" is null.");
		Iterator attributeModels = null;
		IDigitalSubjectModel subjModel = subj.getModel();
		if (subjModel != null)
			attributeModels = subjModel.getAttributeModels();
		Iterator itrAttr = subj.getAttributes();
		validateAttributes(context, itrAttr, attributeModels);
		validateMetadataSet(context, subj.getMetadataSet());
	}

	/**
	 * @param context
	 * @param attributes
	 * @param attributeModels
	 * @throws IdASException
	 */
	public static void validateAttributes(Context context, Iterator attributes, Iterator attributeModels) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateAttributes");
		ArrayList attrTypes = new ArrayList();
		while (attributes.hasNext()) {
			Object obj = attributes.next();
			if (obj instanceof Attribute == false)
				throw new IdASException("Instance of " + Attribute.class.getName() + " expected, " + obj.getClass().getName() + " found.");
			Attribute attr = (Attribute) obj;
			validateAttribute(context, attr);
			attrTypes.add(attr.getAttrID().toString());
		}
	}

	/**
	 * @param context
	 * @param attributeType
	 * @return
	 * @throws IdASException
	 */
	public static boolean isFunctionalProperty(Context context, String attributeType) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isFunctionalProperty");
		ObjectProperty op = context.getObjectProperty(attributeType);
		if (op != null && op.isFunctionalProperty() == true)
			return true;
		else
			return false;
	}

	/**
	 * Checks whether the number of values satisfy with cardinality restrictions
	 * of attribute
	 * 
	 * @param context
	 * @param attr
	 * @param realValuesCount
	 * @throws IdASException
	 */
	public static void checkCardinality(Context context, Attribute attr, int realValuesCount) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::checkCardinality");
		URI attrTypeURI = attr.getAttrID();
		if (attrTypeURI == null)
			throw new IdASException("Attribute type is null.");
		String attrType = attrTypeURI.toString();
		OntClass ontClass = attr.getOwnerClass();
		if (ontClass == null)
			throw new IdASException("Can not get ownerClass for attribute with type " + attrType);
		Iterator ei = ontClass.listSuperClasses(false);
		while (ei.hasNext()) {
			OntClass superClass = (OntClass) ei.next();
			if (superClass.isRestriction()) {
				Restriction rs = superClass.asRestriction();
				String rsProperty = rs.getOnProperty().getURI();
				if (attrType.equals(rsProperty)) {
					try {
						if (rs.isCardinalityRestriction()) {
							CardinalityRestriction cr = rs.asCardinalityRestriction();
							int expectedCount = cr.getCardinality();
							if (expectedCount != realValuesCount)
								throw new IdASException("Wrong count (" + realValuesCount + ") of values for attribute with type " + attrType
										+ ", expected count is " + expectedCount + ".");
						} else if (rs.isMinCardinalityRestriction()) {
							MinCardinalityRestriction mcr = rs.asMinCardinalityRestriction();
							int expectedCount = mcr.getMinCardinality();
							if (expectedCount > realValuesCount)
								throw new IdASException("Wrong count (" + realValuesCount + ") of values for attribute with type " + attrType
										+ ", expected count is more or equals " + expectedCount + ".");
						} else if (rs.isMaxCardinalityRestriction()) {
							MaxCardinalityRestriction mcr = rs.asMaxCardinalityRestriction();
							int expectedCount = mcr.getMaxCardinality();
							if (expectedCount < realValuesCount)
								throw new IdASException("Wrong count (" + realValuesCount + ") of values for attribute with type " + attrType
										+ ", expected count is less or equals " + expectedCount + ".");
						}
					} catch (Exception e) {
						log.error(e);
						throw new IdASException(e.getMessage());
					}
				}
			}
		}
	}

	/**
	 * @param context
	 * @param attr
	 * @throws IdASException
	 */
	public static void validateAttribute(Context context, Attribute attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateAttribute");
		URI type = attr.getAttrID();
		if (type == null)
			throw new IdASException("Type(ID) of attribute is null.");
		String typeStr = type.toString();
		int valuesCount = 0;
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			valuesCount++;
			Object obj = itr.next();
			if (obj instanceof SimpleValue) {
				boolean isFunctionalProperty = isFunctionalProperty(context, typeStr);
				validateSimpleValue(context, (SimpleValue) obj, isFunctionalProperty);
			} else if (obj instanceof ComplexValue)
				validateComplexValue(context, (ComplexValue) obj);
			else {
				if (obj == null)
					throw new IdASException("Value of attribute is null.");
				else
					throw new IdASException("Unexpected class of value of attribute : " + obj.getClass().getName());
			}
		}
		checkCardinality(context, attr, valuesCount);
		validateMetadataSet(context, attr.getMetadataSet());
	}

	/**
	 * @param context
	 * @param val
	 * @throws IdASException
	 */
	public static void validateComplexValue(Context context, ComplexValue val) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateComplexValue");
		if (val != null) {
			Iterator attributeModels = null;
			IAttributeComplexValueModel valueModel = val.getValueModel();
			if (valueModel != null)
				attributeModels = valueModel.getAttributes();
			Iterator itrAttr = val.getAttributes();
			validateAttributes(context, itrAttr, attributeModels);
			validateMetadataSet(context, val.getMetadataSet());
		}
	}

	/**
	 * @param context
	 * @param val
	 * @param isFunctionalProperty
	 * @throws IdASException
	 */
	public static void validateSimpleValue(Context context, SimpleValue val, boolean isFunctionalProperty) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateSimpleValue");
		validateMetadataSet(context, val.getMetadataSet());
	}

	/**
	 * @param context
	 * @param metadataSet
	 * @throws IdASException
	 */
	public static void validateMetadataSet(Context context, Iterator metadataSet) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateMetadataSet");
		while (metadataSet.hasNext()) {
			Object obj = metadataSet.next();
			if (obj instanceof Metadata == false)
				throw new IdASException("Instance of " + Metadata.class.getName() + " expected, " + obj.getClass().getName() + " found.");
			Metadata metadata = (Metadata) obj;
			validateMetadata(context, metadata);
		}

	}

	/**
	 * @param context
	 * @param metadata
	 * @throws IdASException
	 */
	public static void validateMetadata(Context context, Metadata metadata) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateMetadata");
	}

}
