/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.util.HigginsVocabulary;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.model.impl.AttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class Entity implements IEntity {
	private Log log = LogFactory.getLog(Entity.class);

	protected Context context_ = null;

	protected IEntityModel model_ = null;

	protected Individual subject_ = null;

	protected OntClass ontClass_ = null;

	protected URI type_ = null;

	protected String userToken_ = null;

	/**
	 * Constructs Entity for already stored in ontology Individual
	 * 
	 * @param context
	 * @param subject
	 * @throws IdASException
	 */
	public Entity(Context context, Individual subject) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::Entity");
		if (context == null)
			throw new IdASException("Couldn't create Entity with null IContext");
		if (subject == null)
			throw new IdASException("Couldn't create Entity for null subject individual");
		context_ = context;
		subject_ = subject;
		ontClass_ = context.getOntClass(subject.getRDFType().getURI());
		URI type = getEntityType();
		model_ = context_.getContextModel().getEntityModel(type);
		if (model_ == null)
			throw new IdASException("Can not get entity model for Entity URI " + type.toString());
		userToken_ = ModelUtils.getUserToken(context_, subject_);
	}

	public IContext getContext() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getContext");
		return context_;
	}

	public IJenaContext getJenaContext() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getJenaContext");
		return context_;
	}

	public Individual getSubjectIndividual() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getSubjectIndividual");
		return subject_;
	}

	public IEntityModel getModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getModel");
		return model_;
	}

	public String getEntityID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getEntityID");
		IAttribute sAttr = getAttribute(HigginsVocabulary.uniqueIdentifierURI);
		if (sAttr == null)
			throw new IdASException("Attribute" + HigginsVocabulary.uniqueIdentifier + " was not found");
		IAttributeValue val = null;
		Iterator itr = sAttr.getValues();
		if (itr != null && itr.hasNext())
			val = (IAttributeValue)itr.next();
		if (val == null)
			throw new IdASException("Value of attribute " + HigginsVocabulary.uniqueIdentifier + " is null");
		if (val.isSimple() == false)
			throw new IdASException("Value of attribute " + HigginsVocabulary.uniqueIdentifier + " is not SimpleValue");
		Object data = ((ISimpleAttrValue) val).getData();
		if (data == null)
			throw new IdASException("Value of attribute " + HigginsVocabulary.uniqueIdentifier + " contains null data");
		return data.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IDigitalSubject#getEntityType()
	 */
	public URI getEntityType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getEntityType");
		if (type_ == null)
			type_ = URI.create(ontClass_.getURI());
		return type_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IDigitalSubject#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::remove");
		context_.unregisterChangedSubject(this);
		ModelUtils.removeIndividual(context_, subject_);
	}

	public IMetadata addMetadata(URI metadataID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addMetadata");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		IMetadata newMetadata = getMetadata(metadataID);
		return newMetadata;
	}

	public IMetadata addMetadata(IMetadata copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addMetadata");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IMetadata newMetadata = getMetadata(copyFrom.getMetaID());

		//TODO implement add metadata
//		newMetadata.setData(copyFrom.getData());

		context_.registerChangedSubject(this);
		return newMetadata;
	}

	public IMetadata getMetadata(URI metadataID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getMetadata");
		if (metadataID == null)
			throw new IdASException("Parameter \"metadataID\" is null.");
		String uri = metadataID.toString();
		DatatypeProperty prop = context_.getDatatypeProperty(uri);
		if (prop == null)
			throw new IdASException("Can not find datatype property with URI " + uri);
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyRelative(metadata, prop) == false)
			throw new IdASException("Property with URI " + uri + " is not a subproperty of " + HigginsVocabulary.metadata);
		if (ModelUtils.isPropertyOfClass(context_, ontClass_, prop) == false)
			throw new IdASException("Can not add metadata with type " + uri + " for class " + ontClass_.getURI());
		return new Metadata(context_, subject_, prop, this);
	}

	public Iterator getMetadataSet() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getMetadataSet");
		ArrayList list = new ArrayList();
		DatatypeProperty metadata = context_.getDatatypeProperty(HigginsVocabulary.metadata);
		if (metadata == null)
			throw new IdASException("Can not find datatype property with URI " + HigginsVocabulary.metadata);
		StmtIterator props = subject_.listProperties();
		while (props.hasNext()) {
			Statement st = props.nextStatement();
			Property prop = st.getPredicate();
			DatatypeProperty dpProperty = context_.getModel().getDatatypeProperty(prop.getURI());
			if (dpProperty != null) {
				if (ModelUtils.isPropertyRelative(metadata, dpProperty)) {
					IMetadata md = new Metadata(context_, subject_, dpProperty, this);
					list.add(md);
				}
			}
		}
		return list.iterator();
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addAttribute");
		return getAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addAttribute");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IAttribute newAttr = getAttribute(copyFrom.getAttrID());
		context_.registerChangedSubject(this);
		Iterator itrVal = copyFrom.getValues();
		while (itrVal.hasNext()) {
			IAttributeValue value = (IAttributeValue) itrVal.next();
			newAttr.addValue(value);
		}
		Iterator itrMd = copyFrom.getMetadataSet();
		while (itrMd.hasNext()) {
			IMetadata metadata = (IMetadata) itrMd.next();
			newAttr.addMetadata(metadata);
		}
		return newAttr;
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttribute");
		if (attrID == null)
			throw new IdASException("Parameter \"attrID\" is null.");
		String uri = attrID.toString();
		OntProperty op = context_.getOntProperty(uri);
		ObjectProperty attrProp = context_.getObjectProperty(HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyRelative(attrProp, op) == false)
			throw new InvalidTypeException("Property " + uri + " is not subproperty of " + HigginsVocabulary.attribute);
		if (ModelUtils.isPropertyOfClass(context_, ontClass_, op) == false)
			throw new InvalidTypeException("Can not create attribute with type " + uri + " for Entity with type " + ontClass_.getURI());
		AttributeModel attrModel = (AttributeModel) model_.getAttributeModel(attrID);
		if (attrModel == null)
			throw new IdASException("Can not get model of attribute with type " + uri + " for Entity with type " + ontClass_.getURI());
		int maxCardinality = ModelUtils.getMaxCardinality(context_, ontClass_, op);
		if (maxCardinality == 1)
			return new SingleValuedAttribute(context_, this, op, attrModel);
		else
			return new Attribute(context_, this, op, attrModel);
	}

/*	public Iterator getAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttributes");
		ObjectProperty attrProp = context_.getObjectProperty(HigginsVocabulary.attribute);
		HashSet usedURIs = new HashSet();
		List attrs = new ArrayList();
		StmtIterator props = subject_.listProperties();
		while (props.hasNext()) {
			Statement st = props.nextStatement();
			Property prop = st.getPredicate();
			String uri = prop.getURI();
			ObjectProperty property = context_.getModel().getObjectProperty(uri);
			if (property != null) {
				if ((usedURIs.contains(uri) == false) && ModelUtils.isPropertyRelative(attrProp, property)) {
					AttributeModel attrModel = (AttributeModel) model_.getAttributeModel(URI.create(uri));
					if (attrModel == null)
						throw new IdASException("Can not get model of attribute with type " + uri + " for Entity with type " + ontClass_.getURI());
					int maxCardinality = ModelUtils.getMaxCardinality(context_, ontClass_, property);
					if (maxCardinality == 1)
						attrs.add(new SingleValuedAttribute(context_, this, property, attrModel));
					else
						attrs.add(new Attribute(context_, this, property, attrModel));
					usedURIs.add(uri);
				}
			}
		}
		return attrs.iterator();
	}
*/
	public Iterator getAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttributes");
		List attrs = new ArrayList();
		Iterator itr = model_.getAttributeModels();
		while(itr.hasNext()) {
			AttributeModel attrModel = (AttributeModel)itr.next();
			URI attrType = attrModel.getType();
			if (attrType == null)
				throw new IdASException("Can not get the type of attribute model for Entity with type " + ontClass_.getURI());
			ObjectProperty property = context_.getModel().getObjectProperty(attrType.toString());
			if (property == null)
				throw new IdASException("Can not get object property with type " + attrType);
			int maxCardinality = ModelUtils.getMaxCardinality(context_, ontClass_, property);
			if (maxCardinality == 1)
				attrs.add(new SingleValuedAttribute(context_, this, property, attrModel));
			else
				attrs.add(new Attribute(context_, this, property, attrModel));
		}
		return attrs.iterator();
	}

	// TODO need we to add this method to interface?
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getSingleValuedAttribute");
		String idURI = attrID.toString();
		IAttribute attr = getAttribute(attrID);
		if (attr == null)
			return null;
		if (attr instanceof ISingleValuedAttribute == false)
			throw new IdASException("Attribute " + idURI + " is not an instance of ISingleValuedAttribute");
		return (ISingleValuedAttribute) attr;
	}

	public String getUserToken() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getUserToken");
		return userToken_;
	}

	public boolean equals(Object obj) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::equals");
		if (obj == null)
			return false;
		if (obj instanceof Entity == false)
			return false;
		Entity ds = (Entity)obj;
		return subject_.equals(ds.getSubjectIndividual());
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		
	}

}
