package org.eclipse.higgins.idas.model.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.AnnotationProperty;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntDocumentManager;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.ontology.Ontology;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.reasoner.ValidityReport;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class ContextModel implements IContextModel {
	private Log log = LogFactory.getLog(ContextModel.class);

	private static boolean debug = true;

	private static String localBase = "c:\\Documents and Settings\\Val\\.higgins\\.ontology\\";

	public static final String IDAS_BASE_URI = "http://www.eclipse.org/higgins/ontologies/2006/higgins#";

	public static final String C_DS_URI = IDAS_BASE_URI + "Entity";

	public static final String C_ATT_URI = IDAS_BASE_URI + "Value";

	public static final String C_SIMP_ATT_URI = IDAS_BASE_URI + "SimpleValue";

	public static final String C_COMP_ATT_URI = IDAS_BASE_URI + "ComplexValue";

	public static final String P_ATT_URI = IDAS_BASE_URI + "attribute";

	public static final String DD_BASE_URI = "http://www.eclipse.org/higgins/ontologies/2006/display-data#";

	public static final String C_DD_URI = DD_BASE_URI + "DisplayData";

	public static final String P_DD_URI = DD_BASE_URI + "displayData";

	public static final String P_DD_LABEL_URI = DD_BASE_URI + "label";

	public static final String P_DD_DESC_URI = DD_BASE_URI + "description";

	public static final String P_DD_ORDER_URI = DD_BASE_URI + "order";

	public static final String P_DD_IMAGE_URI = DD_BASE_URI + "image";
	
	protected OntModel model = null;

	protected OntClass c_ds = null;
	protected OntClass c_val = null;
	protected OntClass c_simp_val = null;
	protected OntClass c_comp_val = null;

	protected ObjectProperty p_att = null;
	
	protected OntClass c_dd = null;

	protected AnnotationProperty p_dd = null;
	
	protected DatatypeProperty p_dd_label = null;
	protected DatatypeProperty p_dd_desc = null;
	protected DatatypeProperty p_dd_order = null;
	protected DatatypeProperty p_dd_image = null;
	
	protected OntClass c_cs = null;

	protected List attributes = null;

	protected List subjects = null;
	protected Hashtable subjectMap = null;

	protected URI type = null;

	protected IDisplayData displayData = null;
	protected boolean dd_init = false;
	
	protected IJenaContext context_ = null;

	public static void main(String[] args) {
		if (args.length > 0) {
			try {
				IContextModel model = create(args[0]);
				System.out.println(model.getType());
				Iterator subjects = model.getEntityModels();
				if (subjects != null) {
					while(subjects.hasNext()) {
						showSubject((IEntityModel)subjects.next(), "");
					}
				}
			} catch (IdASException e) {
				e.printStackTrace();
			}
		}
	}

	private static void showSubject(IEntityModel sm, String prefix) {
		System.out.println(prefix + "DS: " + sm.getType());
		showDisplayData(sm.getDisplayData(), prefix + "\t");
		Iterator attributes = sm.getAttributeModels();
		if (attributes != null) {
			while (attributes.hasNext()) {
				showAttribute((IAttributeModel)attributes.next(), prefix + "\t", true);
			}
		}
	}

	private static void showAttribute(IAttributeModel am, String prefix, boolean isAttributeOfSubjerct) {
		if (isAttributeOfSubjerct)
			System.out.println(prefix + "ATTRIBUTE_OF_SUBJECT: " + am.getType());
		else
			System.out.println(prefix + "ATT: " + am.getType());
		showDisplayData(am.getDisplayData(), prefix + "\t");
		IAttributeValueModel v = am.getValueModel();
		//TODO uncomment two lines below		
		//		System.out.println(prefix + "\tMinCardinality: " + v.getMinCardinality());
		//		System.out.println(prefix + "\tMaxCardinality: " + v.getMaxCardinality());
		if (v.isSimple()) {
			AttributeSimpleValueModel sv = (AttributeSimpleValueModel) v;
			System.out.println(prefix + "VAL: " + sv.getType() + " isOneOf=" + sv.isOneOf());
			if (sv.isOneOf()) {
				List values = sv.getOneOf();
				for (int i = 0; i < values.size(); i++) {
					System.out.println(prefix + "\t" + i + ": " + values.get(i));
				}
			}
		} else {
			IAttributeComplexValueModel cv = (IAttributeComplexValueModel) v;
			Iterator attrs = cv.getAttributes();
			if (attrs != null) {
				while(attrs.hasNext()) {
					showAttribute((IAttributeModel)attrs.next(), prefix + "\t", false);
				}
			}
		}
	}

	private static void showDisplayData(IDisplayData dd, String prefix) {
		if (dd != null) {
			System.out.println(prefix + dd);
		}
	}

	public static boolean isDebug() {
		return debug;
	}

	public static void setDebug(boolean debug) {
		ContextModel.debug = debug;
	}

	public static String getLocalBase() {
		return localBase;
	}

	public static void setLocalBase(String localBase) {
		ContextModel.localBase = localBase;
	}
	
	public static IContextModel create(String schema) throws IdASException {
		URL url = null;
		try {
			File f = new File(schema);
			if (f.exists()) {
				URI uri = f.toURI();
				url = uri.toURL();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		if (url == null) {
			try {
				url = new URL(schema);
			} catch (Exception e) {
				throw new IdASModelException("Can't find context's ontology model: "
						+ schema, e);
			}
		}

		return create(url);
	}

	/*
	public static IContextModel create(String schema) throws IdASException {
		if (schema == null) {
			throw new IdASModelException("The schema parameter can't be null.");
		}
		ByteArrayInputStream bio = new ByteArrayInputStream(schema.getBytes());
		return create(bio);
	}
	*/

	
	public static IContextModel create(IJenaContext context, OntModel model) throws IdASException {
		return new ContextModel(context, model);
	}

	public static IContextModel create(URL schema) throws IdASException {
		return new ContextModel(schema);
	}

	public static IContextModel create(InputStream schema) throws IdASException {
		return new ContextModel(schema);
	}

	private ContextModel(IJenaContext context, OntModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::ContextModel(IJenaContext, OntModel)");
		context_ = context;
		this.model = model;
		init();
	}

	private ContextModel(URL schema) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::ContextModel(URL)");
		try {
			OntDocumentManager mgr = OntDocumentManager.getInstance();
			File f = new File(mgr.doAltURLMapping(schema.toString()));
			if (f.exists()) {
				open(new FileInputStream(f));
			} else {
				open(schema.openStream());
			}
		} catch (IdASException e) {
			throw e;
		} catch (Exception e) {
			log.error(e);
			throw new IdASModelException("Can't open context's ontology model: " + schema, e);
		}
	}

	private ContextModel(InputStream schema) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::ContextModel(InputStream)");
		open(schema);
	}

	private void open(InputStream is) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::open");
		OntModelSpec spec = new OntModelSpec(OntModelSpec.OWL_DL_MEM_TRANS_INF);
		model = ModelFactory.createOntologyModel(spec);

		// For debug purposes only
		/*
		OntDocumentManager mg = null;
		if (isDebug()) {
			mg = model.getDocumentManager();
			mg.addAltEntry(
					"http://www.eclipse.org/higgins/ontologies/2006/higgins",
					localBase + "higgins.owl");
			mg.addAltEntry(
					"http://www.eclipse.org/higgins/ontologies/2006/higgins",
					"http://www.eclipse.org/higgins/ontologies/2006/higgins.owl");
			
			mg.addAltEntry(
					"http://www.eclipse.org/higgins/ontologies/2006/display-data",
					localBase + "display-data.owl");
		}
		*/
		model.read(is, null);
		init();
	}
	

	private void init() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::init");
 		ValidityReport report = model.validate();
		if (report != null && !report.isValid()) {
			throw new IdASModelException("Context schema is not valid OWL DL ontology.");
		}
		// IdAS
		c_ds = model.getOntClass(C_DS_URI);
		if (c_ds == null)
			throw new IdASModelException("Invalid schema. Couldn't find class by URI " + C_DS_URI);
		p_att = model.getObjectProperty(P_ATT_URI);
		if (p_att == null)
			throw new IdASModelException("Invalid schema. Couldn't find object property by URI " + P_ATT_URI);
		c_val = model.getOntClass(C_ATT_URI);
		if (c_val == null)
			throw new IdASModelException("Invalid schema. Couldn't find class by URI " + C_ATT_URI);
		c_simp_val = model.getOntClass(C_SIMP_ATT_URI);
		if (c_simp_val == null)
			throw new IdASModelException("Invalid schema. Couldn't find class by URI " + C_SIMP_ATT_URI);
		c_comp_val = model.getOntClass(C_COMP_ATT_URI);
		if (c_comp_val == null)
			throw new IdASModelException("Invalid schema. Couldn't find class by URI " + C_COMP_ATT_URI);
		/*
		c_att = model.getOntClass(C_ATT_URI);
		c_simp_att = model.getOntClass(C_SIMP_ATT_URI);
		c_comp_att = model.getOntClass(C_COMP_ATT_URI);
*/		
		// Display data
/*
		c_dd = model.getOntClass(C_DD_URI);
		if (c_dd == null)
			throw new IdASModelException("Invalid schema. Couldn't find class by URI " + C_DD_URI);
		p_dd = model.getAnnotationProperty(P_DD_URI);
		if (p_dd == null)
			throw new IdASModelException("Invalid schema. Couldn't find annotation property with URI " + P_DD_URI);
		p_dd_label = model.getDatatypeProperty(P_DD_LABEL_URI);
		if (p_dd_label == null)
			throw new IdASModelException("Invalid schema. Couldn't find datatype property by URI " + P_DD_LABEL_URI);
		p_dd_desc = model.getDatatypeProperty(P_DD_DESC_URI);
		if (p_dd_desc == null)
			throw new IdASModelException("Invalid schema. Couldn't find datatype property by URI " + P_DD_DESC_URI);
		p_dd_order = model.getDatatypeProperty(P_DD_ORDER_URI);
		if (p_dd_order == null)
			throw new IdASModelException("Invalid schema. Couldn't find datatype property by URI " + P_DD_ORDER_URI);
		p_dd_image = model.getDatatypeProperty(P_DD_IMAGE_URI);
		if (p_dd_image == null)
			throw new IdASModelException("Invalid schema. Couldn't find datatype property by URI " + P_DD_IMAGE_URI);
*/
		initType();
		initSubjects();
		initAttributes();
	}
	
	
	protected void initType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::initType");
		if (type == null) {
			Set impSet = model.listImportedOntologyURIs(true);
			for (ExtendedIterator itr = model.listOntologies(); itr.hasNext();) {
				Ontology oo = (Ontology) itr.next();
				if (!impSet.contains(oo.getURI())) {
					try {
						type = new URI(oo.getURI());
						itr.close();
						break;
					} catch (Exception e) {
						log.error(e);
						type = null;
					}
				}
			}
		}
		if (type == null) {
			throw new IdASModelException("Invalid context schema - can't initialize schema's URL");
		}
	}
	
	protected void initSubjects() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::initSubjects");
		if (subjects == null) {
			subjects = new ArrayList();
			subjectMap =  new Hashtable();
			for (Iterator itr = model.listNamedClasses(); itr.hasNext(); ) {
				OntClass c = (OntClass) itr.next();
				log.debug("Try to create SubjectModel for class " + c.getURI());
				String ns = c.getNameSpace();
				if (ns.startsWith(getType().toString())) {
					if (ModelUtils.isClassRelative(c_ds, c)) {
						IEntityModel dsm = new EntityModel(this, c);
						log.debug("Created SubjectModel for class " + c.getURI());
						subjects.add(dsm);
						subjectMap.put(dsm.getType(), dsm);
					}
				}
			}
		}
	}
	
	//TODO Do we need this method for context model? (SergeyL)
	protected void initAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::initAttributes");
		if (attributes == null) {
			attributes = new ArrayList();
			// TODO Add attribute models
		}
	}

	protected void initDisplayData() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::initDisplayData");
		if (displayData == null) {
			if (c_cs != null && getP_DD() != null && c_cs.hasProperty(getP_DD())) {
				RDFNode n = c_cs.getPropertyValue(getP_DD());
				System.out.println(n);
				if (n.canAs(Individual.class)) {
					Individual in = (Individual) n.as(Individual.class);
					Resource r = in.getRDFType();
					if (r.canAs(OntClass.class)) {
						OntClass t = (OntClass) r.as(OntClass.class);
						if (t.equals(getC_DD())) {
							displayData = new DisplayData(this, in);
						}
					}
				}
			}
		}
	}
	
	public IModel getModel(URI type) {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getModel");
		return null;
	}

	public Iterator getAttributes() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getAttributes");
		if (attributes != null)
			return attributes.iterator();
		else
			return null;
	}

	public Iterator getEntityModels() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getSubjectModels");
		if (subjects != null)
			return subjects.iterator();
		else
			return null;
	}

	public IEntityModel getEntityModel(URI type) {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getSubjectModel");
		if (type != null)
			return (IEntityModel)subjectMap.get(type);
		else
			return null;
	}

	public URI getType() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getType");
		return type;
	}

	public IDisplayData getDisplayData() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getDisplayData");
		return displayData;
	}


	protected OntClass getC_VAL() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getC_VAL");
		return c_val;
	}

	protected OntClass getC_COMP_VAL() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getC_COMP_VAL");
		return c_comp_val;
	}

	protected OntClass getC_SIMP_VAL() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getC_SIMP_VAL");
		return c_simp_val;
	}

	protected ObjectProperty getP_ATT() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_ATT");
		return p_att;
	}

	protected OntClass getC_DS() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getC_DS");
		return c_ds;
	}

	protected OntModel getModel() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getModel");
		return model;
	}

	protected OntClass getC_DD() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getC_DD");
		return c_dd;
	}

	protected AnnotationProperty getP_DD() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_DD");
		return p_dd;
	}

	protected DatatypeProperty getP_DD_Desc() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_DD_Desc");
		return p_dd_desc;
	}

	protected DatatypeProperty getP_DD_Label() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_DD_Label");
		return p_dd_label;
	}

	protected DatatypeProperty getP_DD_Order() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_DD_Order");
		return p_dd_order;
	}

	protected DatatypeProperty getP_DD_Image() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getP_DD_Image");
		return p_dd_image;
	}

	protected IJenaContext getContext() {
		log.trace("org.eclipse.higgins.idas.model.impl.ContextModel::getContext");
		return context_;
	}
}
