/*******************************************************************************
 * Copyright (c) 2006 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.util.XMLDateConvertor;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.datatypes.TypeMapper;
import com.hp.hpl.jena.datatypes.xsd.XSDDatatype;
import com.hp.hpl.jena.ontology.DataRange;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.Literal;

public class XMLValueModel {
	private static Log staticLog = LogFactory.getLog(XMLValueModel.class);

	private Log log = LogFactory.getLog(XMLValueModel.class);

	protected URI type = null;

	protected boolean isOneOf = false;

	protected List oneOf = null;

	protected OntProperty prop = null;

	protected XSDDatatype xsdType = null;

	protected static Hashtable typesMap = new Hashtable();

	public static synchronized XMLValueModel getInstance(DatatypeProperty prop) throws IdASException {
		if (prop == null)
			return null;
		String uri = prop.getURI();
		if (typesMap.contains(uri))
			return (XMLValueModel) typesMap.get(uri);
		else {
			XMLValueModel valueModel = new XMLValueModel(prop);
			typesMap.put(uri, valueModel);
			return valueModel;
		}
	}

	protected XMLValueModel(OntProperty prop) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::XMLValueModel");
		this.prop = prop;
		initialize();
	}

	protected void initialize() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::initialize");
		oneOf = new ArrayList();
		OntResource r = prop.getRange();
		if (r != null) {
			if (r.isDataRange()) {
				DataRange dr = r.asDataRange();
				isOneOf = true;
				for (Iterator itr = dr.listOneOf(); itr.hasNext();) {
					Literal l = (Literal) itr.next();
					RDFDatatype t = l.getDatatype();
					initType(t.getURI());
					oneOf.add(l.getValue());
				}
			} else {
				initType(r.getURI());
			}
		}
	}

	protected void initType(String typeUri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::initType");
		if (type == null) {
			try {
				xsdType = findXSDDatatype(typeUri);
				if (xsdType == null) {
					throw new IdASModelException("Unsupported datatype of property's value: Property=" + prop.getURI() + " Datatype=" + typeUri);
				}
				type = new URI(typeUri);
			} catch (IdASException e) {
				throw e;
			} catch (Exception e) {
				throw new IdASModelException("Invalid datatype of property's value: Property=" + prop.getURI() + " Datatype=" + typeUri, e);
			}
		} else if (typeUri.equals(type) == false) {
			throw new IdASModelException("Datatype mismatch for enumerated values: Property=" + prop.getURI() + " Datatype1=" + type + " Datatype2=" + typeUri);
		}
	}

	public static XSDDatatype findXSDDatatype(String typeUri) {
		if (typeUri == null)
			return null;
		RDFDatatype rdfType = TypeMapper.getInstance().getTypeByName(typeUri);
		if (rdfType == null) {
			staticLog.warn("Couldn't find RDFDatatype by URI = " + typeUri);
			return null;
		}
		if (rdfType instanceof XSDDatatype == false) {
			staticLog.warn("URI " + typeUri + " is not xsd datatype");
			return null;
		}
		return (XSDDatatype) rdfType;
	}

	public static boolean isSupported(String typeUri) {
		return (findXSDDatatype(typeUri) != null);
	}

	public boolean isOneOf() {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::isOneOf");
		return isOneOf;
	}

	public List getOneOf() {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::getOneOf");
		return oneOf;
	}

	public URI getType() {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::getType");
		return type;
	}

	public boolean isValidValue(Object valueForm) {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::isValidValue");
		return xsdType.isValidValue(valueForm);
	}

	public boolean isValid(String lexicalForm) {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::isValid");
		return xsdType.isValid(lexicalForm);
	}

	public Object toValueForm(String lexicalForm) throws IdASModelException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::toValueForm");
		if (lexicalForm == null)
			return "";
		try {
			return xsdType.parse(lexicalForm);
		} catch (Exception e) {
			throw new IdASModelException(e);
		}
	}

	public Object toValueForm(Literal literal) throws IdASModelException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::toValueForm");
		if (literal != null) {
			Object obj = parse(literal);
			return obj;
		}
		else
			return null;
	}

	public String toLexicalForm(Object value) throws IdASModelException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::toLexicalForm");
		if (value == null)
			return null;
		return unparseObj(value);
	}

	private String unparseObj(Object value) throws IdASModelException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::unparseObj");
		try {
			if (value instanceof Date) {
				return XMLDateConvertor.format((Date) value);
			}
			if (value instanceof byte[]) {
				return Base64.encode((byte[]) value);
			}
			if (value instanceof ByteBuffer) {
				return Base64.encode(((ByteBuffer)value).array());
			}
			return xsdType.unparse(value);
		} catch (Exception e) {
			throw new IdASModelException("Can not get lexical form for xml type " + type.toString() + ", object value " + value);
		}
	}

	private Object parse(Literal lt) throws IdASModelException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::parse");
		try {
			//
			if (type.equals(ISimpleAttrValue.DATE_TYPE_URI) || type.equals(ISimpleAttrValue.DATETIME_TYPE_URI) || type.equals(ISimpleAttrValue.DATETIME_TYPE_URI)
					|| type.equals(ISimpleAttrValue.TIME_TYPE_URI))
				return XMLDateConvertor.parse(lt.getString());
			//
			if (type.equals(ISimpleAttrValue.ANYURI_TYPE_URI)) {
				try {
					return new URI(lt.getString());
				}
				catch(Exception e) {
					log.error(e);
					throw new IdASModelException(e);
				}
			}
			if (type.equals(ISimpleAttrValue.BASE64BINARY_TYPE_URI)) {
				byte[] bytes = null;
				try {
					bytes = (byte[])lt.getValue();
					ByteBuffer bb = ByteBuffer.allocate(bytes.length);
					bb.put(bytes);
					return bb;
				}
				catch(ClassCastException e) {
					log.error(e);
					throw new IdASException(e);
				}
			}
			return lt.getValue();
		} catch (Exception e) {
			throw new IdASModelException(e);
		}
	}

}
