/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.AuthenticationException;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContextRelation;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityIDAssertion;
import org.eclipse.higgins.idas.api.IFilterMetadataAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidEntityIDException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.IUserAccount;
import org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterEntityIDAssertion;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterMetadataAssertion;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterTypeAssertion;
import org.eclipse.higgins.idas.cp.jena2.util.HigginsVocabulary;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.spi.BasicAttribute;
import org.eclipse.higgins.idas.spi.BasicAuthNAttributesMaterials;
import org.eclipse.higgins.idas.spi.BasicComplexValue;
import org.eclipse.higgins.idas.spi.BasicSimpleValue;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.model.impl.ContextModel;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntDocumentManager;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.rdf.model.RDFWriter;

public abstract class Context implements IJenaContext {
	private Log log = LogFactory.getLog(Context.class);

	private Hashtable ontClasses_ = new Hashtable();

	private Hashtable ontProperties_ = new Hashtable();

	private Hashtable datatypeProperties_ = new Hashtable();

	private Hashtable objectProperties_ = new Hashtable();

	protected String ns_ = null;

	protected OntModel model_ = null;

	protected Individual contextObject_ = null;

	protected String schemaURL_ = null;

	protected String schemaFileName_ = null;

	protected IContextModel contextModel_ = null;

	protected Vector changedSubjects_ = new Vector();

	protected IUserAccount userAccount_ = null;

	protected AuthenticationModule authModule_ = null;

	/*	From now default is no any validation. If need to validate, create configuration 
		property "validateSubjects" and set it to true
	*/
	protected boolean validateChangedSubjects_ = false; 
	/*
	 * protected abstract void init() throws IdASException;
	 * 
	 * protected abstract void save() throws IdASException;
	 */
	protected void init() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::init");

	}

	protected void save() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::save");
	}

	public void addRelationship(IContextRelation relationship) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::addRelationship");
		throw new NotImplementedException("");
	}

	public IEntity addEntity(URI type, String entityID) throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::addSubject");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		if (type == null)
			throw new IdASException("Parameter \"type\" is null");
		if (entityID != null && ModelUtils.getSubjectByID(this, entityID) != null)
			throw new EntityExistsException("Subject with ID = " + entityID + " already exists");
		String uri = type.toString();
		OntClass ontClass = getOntClass(uri);
		String uriDS = HigginsVocabulary.DigitalSubject;
		OntClass higginsDigitalSubject_ = getOntClass(uriDS);
		if (ModelUtils.isClassRelative(higginsDigitalSubject_, ontClass) == false)
			throw new IdASException("Class " + uri + " is not a subclass of " + uriDS);
		Individual ind = model_.createIndividual(ontClass);
		String userToken = userAccount_.getUserToken();
		if (userToken != null && userToken.length() > 0)
			ModelUtils.setUserToken(this, ind, userToken);
		Entity subj = new Entity(this, ind);
		String newSubjID = (entityID != null) ? entityID : ModelUtils.generateUniqueID(this);
		subj.getAttribute(HigginsVocabulary.uniqueIdentifierURI).addSimpleValue(null, newSubjID);
		
		return subj;
	}

	public IEntity addEntity(IEntity copyFrom) throws IdASException, EntityExistsException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::addSubject");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null");
		String uri = copyFrom.getEntityType().toString();
		OntClass ontClass = getOntClass(uri);
		String uriDS = HigginsVocabulary.DigitalSubject;
		OntClass higginsDigitalSubject_ = getOntClass(uriDS);
		if (ModelUtils.isClassRelative(higginsDigitalSubject_, ontClass) == false)
			throw new IdASException("Class " + uri + " is not a subclass of " + uriDS);
		Individual ind = model_.createIndividual(ontClass);
		String userToken = userAccount_.getUserToken();
		if (userToken != null && userToken.length() > 0)
			ModelUtils.setUserToken(this, ind, userToken);
		Entity subj = new Entity(this, ind);
		subj.getAttribute(HigginsVocabulary.uniqueIdentifierURI).addSimpleValue(null, ModelUtils.generateUniqueID(this));
		Iterator mtd = copyFrom.getMetadataSet();
		while (mtd.hasNext()) {
			IMetadata metadata = (IMetadata) mtd.next();
			subj.addMetadata(metadata);
		}
		Iterator attr = copyFrom.getAttributes();
		while (attr.hasNext()) {
			IAttribute attribute = (IAttribute) attr.next();
			// TODO should we copy uniqueIdentifier
			if (HigginsVocabulary.uniqueIdentifierURI.equals(attribute.getAttrID()) == false)
				subj.addAttribute(attribute);
		}
		registerChangedSubject(subj);
		return subj;
	}

	public IAttribute buildAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildAttribute");
		return new BasicAttribute(attrID, (Iterator) null, null, this);
	}

	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildAttributeAssertion");
		return new FilterAttributeAssertion(this);
	}

	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildAuthNAttributesMaterials");
		return new BasicAuthNAttributesMaterials(null, this);
	}

	public IFilterEntityIDAssertion buildEntityIDAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildEntityIDAssertion");
		return new FilterEntityIDAssertion(this);
	}

	public IComplexAttrValue buildComplexAttrValue(URI type) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildComplexAttrValue");
		return new BasicComplexValue(type, null, null);
	}

	public IFilter buildFilter() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildFilter");
		return new Filter(this);
	}

	public IFilterMetadataAssertion buildMetadataAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildMetadataAssertion");
		return new FilterMetadataAssertion(this);
	}

	public IContextRelation buildRelationship(URI type) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildRelationship");
		throw new NotImplementedException("");
	}

	public ISimpleAttrValue buildSimpleAttrValue(URI type, Object value) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildSimpleAttrValue");
		return new BasicSimpleValue(type, value);
	}

	public IFilterEntityTypeAssertion buildEntityTypeAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::buildTypeAssertion");
		return new FilterTypeAssertion(this);
	}

	/** ******************************************************* */

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IContext#close()
	 */
	public void close() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::close");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		save();
		userAccount_ = null;
		ontClasses_.clear();
		ontProperties_.clear();
		datatypeProperties_.clear();
		objectProperties_.clear();
	}

	public String exportData(String filter, String representationFormat) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::exportData");
		throw new NotImplementedException("");
	}

	public IContextModel getContextModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getContextModel");
		if (contextModel_ == null) {
			/*
			 * if (schemaFileName_ != null) { File f = new
			 * File(schemaFileName_); if (f.exists()) { try { contextModel_ =
			 * ContextModel.create(new FileInputStream(f)); } catch (Exception
			 * e) { Logger.logError(e); } } } if (contextModel_ == null) {
			 * contextModel_ = ContextModel.create(schemaURL_); }
			 */
			Iterator it = model_.listImportedModels();
			if (it.hasNext()) {
				OntModel m = (OntModel) it.next();
				contextModel_ = ContextModel.create(this, m);
			} else {
				contextModel_ = ContextModel.create(this, model_);
				log.info("Can not get imported model. ContextModel was created using working model.");
			}
		}
		if (contextModel_ == null)
			throw new IdASException("Can not initialize context model");
		return contextModel_;
	}

	public URI getContextID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getContextID");
		try {
			return new URI(ns_);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
	}

	public Iterator getRelationships() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getRelationships");
		throw new NotImplementedException("");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IContext#getSchema()
	 */
	public String getSchema() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSchema");
		String res = null;
		OntModel schema = getModel().getImportedModel(schemaURL_);
		if (schema != null) {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			RDFWriter writer = schema.getBaseModel().getWriter("RDF/XML-ABBREV");
			// RDFWriter writer = schema.getWriter("RDF/XML-ABBREV");
			writer.setProperty("xmlbase", ns_);
			writer.setProperty("allowBadURIs", "false");
			writer.write(schema, baos, ns_);
			res = baos.toString();
		}
		return res;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getSubject(java.lang.String)
	 */
	public IEntity getEntity(String cuid) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSubject");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		if (cuid == null)
			throw new NoSuchEntityException("Couldn't get Entity with null cuid");
		Individual subj = ModelUtils.getSubjectByID(this, cuid);
		if (subj == null)
			throw new NoSuchEntityException("Couldn't find Entity with cuid : " + cuid);
		else {
			if (hasUserPermissions(subj))
				return new Entity(this, subj);
			else
				throw new AuthenticationException("The user opened the context does not have permissions to the subject with ID = " + cuid + ".");
		}
	}

	/**
	 * @return true if user opened the context has permissions to operate with this subject
	 */
	private boolean hasUserPermissions(Individual subj) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::hasUserPermissions");
		String userToken = userAccount_.getUserToken();
		if (userToken != null && userToken.length() > 0) {
			String realToken = ModelUtils.getUserToken(this, subj);
			return userToken.equals(realToken);
		}
		return true;
	}

	public IEntity getEntity(String cuid, Iterator attrSelectionList) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSubject");
		return getEntity(cuid);
	}

	public Iterator getEntities(IFilter filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSubjects");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		Filter f = null;
		if (filter == null) {
			f = (Filter)buildFilter();
			FilterTypeAssertion fass = (FilterTypeAssertion)buildEntityTypeAssertion();
			fass.setID(HigginsVocabulary.DigitalSubjectURI);
			fass.includeSubtypes(true);
			f.addFilter(fass);
		}
		else {
			if (filter instanceof Filter == false)
				throw new IdASException("IFIlter instance should be an instance of " + Filter.class.getName());
			else
				f = (Filter)filter;
		}
		FilterContext fc = new FilterContext(this, false);
		ArrayList result = new ArrayList();
		List lst = fc.getSubjectIndividuals(f);
		int size = lst.size();
		for (int i = 0; i < size; i ++) {
			Individual ind = (Individual)lst.get(i);
			Entity ds = new Entity(this, ind);
			result.add(ds);
		}
		return result.iterator();
	}

	public ArrayList getNodeIndividuals(IFilter filter, boolean ignoreUserToken) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSubjects");
		if (filter == null)
			throw new IdASException("Parameter \"filter\" is null.");
		if (filter instanceof Filter == false)
			throw new IdASException("IFIlter instance should be an instance of " + Filter.class.getName());
		Filter f = (Filter)filter;
		FilterContext fc = new FilterContext(this, ignoreUserToken);
		ArrayList result = new ArrayList();
		List lst = fc.getSubjectIndividuals(f);
		int size = lst.size();
		for (int i = 0; i < size; i ++) {
			Individual ind = (Individual)lst.get(i);
			result.add(ind);
		}
		return result;
	}

	public Iterator getEntities(IFilter filter, Iterator attrSelectionList) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSubjects");
		return getEntities(filter);
	}

	public void importData(String filter, String representationFormat) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::importData");
		throw new NotImplementedException("");
	}

	public boolean isOpen(Object identity) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::isOpen");
		return userAccount_ != null;
	}

	public String open(Object authentication) throws IdASException, ContextOpenException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::open");
//		if (userAccount_ != null) 	throw new ContextOpenException("");
		init();
		if (authModule_ == null)
			authModule_ = new AuthenticationModule(this);
		userAccount_ = authModule_.authenticate(authentication);
		return null;
	}

	public void removeRelationship(IContextRelation relationship) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::removeRelationship");
		throw new NotImplementedException("");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IContext#removeSubject(java.lang.String)
	 */
	public void removeSubject(String cuid) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::removeSubject");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		Individual subjInd = ModelUtils.getSubjectByID(this, cuid);
		if (subjInd != null)
			ModelUtils.removeIndividual(this, subjInd);
		else
			throw new NoSuchEntityException("Couldn't find Entity with cuid : " + cuid);
	}

	public void setSchema(String schema) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::setSchema");
		schemaURL_ = schema;
	}

	public boolean verifyEntityAttributes(String subjectID, Iterator attributes) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::verifySubjectAttributes");
		throw new NotImplementedException("");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#getModel()
	 */
	public OntModel getModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getModel");
		if (userAccount_ == null)
			throw new ContextNotOpenException("");
		if (model_ == null)
			throw new IdASException("The context is open but its ontology model is not initialized");
		return model_;
	}

	/**
	 * @return
	 */
	public OntModel getModelNoException() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getModelNoException");
		return model_;
	}

	public String getNS() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getNS");
		return ns_;
	}

	/**
	 * @return 
	 */
	public IUserAccount getUserAccount() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getUserAccount");
		return userAccount_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#getDatatypeProperty(java.lang.String)
	 */
	public DatatypeProperty getDatatypeProperty(String uri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getDatatypeProperty");
		if (uri == null)
			throw new IdASException("Couldn't get DatatypeProperty for null URI");
		if (datatypeProperties_.containsKey(uri))
			return (DatatypeProperty) datatypeProperties_.get(uri);
		else {
			DatatypeProperty prop = getModel().getDatatypeProperty(uri);
			if (prop != null) {
				datatypeProperties_.put(uri, prop);
				return prop;
			} else
				throw new IdASException("Couldn't find DatatypeProperty with URI : " + uri);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#getObjectProperty(java.lang.String)
	 */
	public ObjectProperty getObjectProperty(String uri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getObjectProperty");
		if (uri == null)
			throw new IdASException("Couldn't get ObjectProperty for null URI");
		if (objectProperties_.containsKey(uri))
			return (ObjectProperty) objectProperties_.get(uri);
		else {
			ObjectProperty prop = getModel().getObjectProperty(uri);
			if (prop != null) {
				objectProperties_.put(uri, prop);
				return prop;
			} else
				throw new IdASException("Couldn't find ObjectProperty with URI : " + uri);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#getOntProperty(java.lang.String)
	 */
	public OntProperty getOntProperty(String uri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getOntProperty");
		if (uri == null)
			throw new IdASException("Couldn't get OntProperty for null URI");
		if (ontProperties_.containsKey(uri))
			return (OntProperty) ontProperties_.get(uri);
		else {
			OntProperty prop = getModel().getOntProperty(uri);
			if (prop != null) {
				ontProperties_.put(uri, prop);
				return prop;
			} else
				throw new IdASException("Couldn't find OntProperty with URI : " + uri);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#getOntClass(java.lang.String)
	 */
	public OntClass getOntClass(String uri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getOntClass");
		if (uri == null)
			throw new IdASException("Couldn't get OntClass for null URI");
		if (ontClasses_.containsKey(uri))
			return (OntClass) ontClasses_.get(uri);
		else {
			OntClass cls = getModel().getOntClass(uri);
			if (cls != null) {
				ontClasses_.put(uri, cls);
				return cls;
			} else
				throw new IdASException("Couldn't find OntClass with URI : " + uri);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#getSchemaURI()
	 */
	public String getSchemaURI() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::getSchemaURI");
		/*
		 * if (schemaFileName_ != null && schemaFileName_.length() > 0) { try {
		 * Bundle pluginBundle =
		 * Platform.getBundle("org.eclipse.higgins.idas.cp.jena2"); URL
		 * schemaURL = Platform.find(pluginBundle, new Path(schemaFileName_));
		 * URL url = Platform.resolve(schemaURL); if (url != null) return
		 * url.toString(); } catch (Exception e) { Logger.logError(e); } }
		 */
		return schemaURL_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#init(java.util.Properties)
	 */
	public void init(Properties props) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::init");
		// TODO Auto-generated method stub
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#applyUpdates()
	 */
	public void applyUpdates() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::applyUpdates");
		validateSubjects();
		commit();
	}

	/**
	 * @throws IdASException
	 */
	private void validateSubjects() throws IdASException {
		if (validateChangedSubjects_ == false)
			return;
		int size = changedSubjects_.size();
		for (int i = 0; i < size; i++) {
			Entity subj = (Entity)changedSubjects_.get(i);
			ModelUtils.validateSubject(this, subj);
		}
	}

	/**
	 * Register the subject which was changed and should be validated (optionally) before saving
	 * 
	 * @param subj
	 * @throws IdASException
	 */
	public void registerChangedSubject(Entity subj) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::registerChangedSubject");
		if (validateChangedSubjects_ == false)
			return;
		if (changedSubjects_.contains(subj) == false)
			changedSubjects_.add(subj);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#unregisterChangedSubject(org.eclipse.higgins.idas.cp.jena2.impl.Entity)
	 */
	public void unregisterChangedSubject(Entity subj) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::unregisterChangedSubject");
		if (changedSubjects_.contains(subj))
			changedSubjects_.remove(subj);
	}

	/**
	 * @param publicURI
	 * @param localSchemaFile
	 * @throws IdASException
	 */
	protected void setSchemaURI(String publicURI, String localSchemaFile) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::setSchemaURI");
		System.out.println("Context URI: \"" + getContextID() + "\", schema URI: public URI \"" + publicURI + "\", localFile \"" + localSchemaFile + "\"");
		schemaURL_ = publicURI;
		if (schemaURL_ == null)
			throw new IdASException("Invalid context schema: url=" + schemaURL_);
		// schemaFileName_ = resolveFileName(localSchemaFile);
		schemaFileName_ = localSchemaFile;
		if (schemaFileName_ != null) {
			try {
				File f = new File(schemaFileName_);
				if (!f.exists()) {
					URL u = new URL(schemaURL_);
					BufferedReader r = new BufferedReader(new InputStreamReader(u.openStream()));
					PrintWriter p = new PrintWriter(new FileOutputStream(f));
					String s = null;
					while ((s = r.readLine()) != null) {
						p.println(s);
					}
					p.close();
				}
				if (f.exists()) {
					OntDocumentManager dm = OntDocumentManager.getInstance();
					dm.addAltEntry(schemaURL_, schemaFileName_);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#addRelationship(java.net.URI)
	 */
	public IContextRelation addRelationship(URI relationshipType) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Context::addRelationship");
		throw new NotImplementedException("");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#setValidationMode(boolean)
	 */
	public void setValidationMode(boolean performValidation) {
		validateChangedSubjects_ = performValidation;
	}

}
