/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class ContextConfiguration implements IContextConfiguration {
	private Log log = LogFactory.getLog(ContextConfiguration.class);

	private String cachePath = null;

	private URI contextID = null;

	private Properties properties = null;

	public ContextConfiguration(String cachePath, URI contextID, Properties defaults, Properties properties) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::ContextConfiguration(String, URI, Properties, Properties)");
		this.cachePath = cachePath;
		this.contextID = contextID;
		setProperties(defaults, properties);
	}

	public ContextConfiguration(Properties properties) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::ContextConfiguration(Properties)");
		this.cachePath = "";
		this.contextID = URI.create("http://test");
		setProperties(new Properties(), properties);
	}

	public ContextConfiguration(String cachePath, Properties defaults, Element e) throws Exception {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::ContextConfiguration(String, Properties, Element)");
		this.cachePath = cachePath;
		load(defaults, e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IContextConfiguration#getCachePath()
	 */
	public String getCachePath() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getCachePath");
		return cachePath;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#getContextID()
	 */
	public URI getContextID() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getContextID");
		return contextID;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#getProperties()
	 */
	public Properties getProperties() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getProperties");
		return properties;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#load(java.util.Properties,
	 *      org.w3c.dom.Element)
	 */
	public void load(Properties defaults, Element e) throws Exception {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::load");
		String uri = e.getAttribute("uri");
		if (uri == null || uri.length() == 0)
			throw new Exception("Element \"context\" doesn't have \"uri\" attribute");
		contextID = new URI(uri);
		setProperties(defaults, loadProperties(e));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#store(org.w3c.dom.Element)
	 */
	public void store(Element e) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::store");
		e.setAttribute("uri", getContextID().toString());
		saveProperties(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#getProperty(java.lang.String)
	 */
	public String getProperty(String key) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getProperty");
		String res = null;
		if (properties != null) {
			res = properties.getProperty(key);
		}
		return res;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#getPropertyNames()
	 */
	public List getPropertyNames() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getPropertyNames");
		ArrayList list = new ArrayList();
		for (Iterator itr = properties.keySet().iterator(); itr.hasNext();) {
			list.add((String) itr.next());
		}
		return list;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IContextConfiguration#getProperty(java.lang.String,
	 *      java.lang.String)
	 */
	public String getProperty(String key, String defaultValue) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::getProperty");
		String res = defaultValue;
		if (properties != null) {
			res = properties.getProperty(key, defaultValue);
		}
		return res;
	}

	private void setProperties(Properties defaults, Properties properties) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::setProperties");
		this.properties = new Properties(defaults);
		Iterator itr = properties.keySet().iterator();
		while (itr.hasNext()) {
			String k = (String) itr.next();
			this.properties.setProperty(k, properties.getProperty(k));
		}
	}

	private Properties loadProperties(Element e) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::loadProperties");
		Properties props = new Properties();
		NodeList nl = e.getChildNodes();
		for (int i = 0; i < nl.getLength(); i++) {
			Node node = nl.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && node.getNodeName().equals("parameter")) {
				Element el = (Element) node;
				String name = el.getAttribute("name");
				if (name.length() > 0) {
					String value = el.getAttribute("value");
					props.put(name, value);
				}
			}
		}
		return props;
	}

	private void saveProperties(Element e) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ContextConfiguration::saveProperties");
		Document doc = e.getOwnerDocument();
		List itrbl = getPropertyNames();
		if (itrbl != null) {
			Iterator itr = itrbl.iterator();
			while (itr.hasNext()) {
				String n = (String) itr.next();
				String v = getProperty(n);
				Element p = doc.createElement("parameter");
				p.setAttribute("name", n);
				p.setAttribute("value", v);
				e.appendChild(p);
			}
		}
	}
}
