/*******************************************************************************
 * Copyright (c) 2006 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;
import org.eclipse.higgins.idas.api.model.IMetadataModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class AttributeModel implements IAttributeModel {
	private Log log = LogFactory.getLog(AttributeModel.class);

	protected URI type = null;
	
	protected boolean isDirect = false;

	protected IAttributeValueModel value = null;

	protected IDisplayData displayData = null;

	protected ContextModel ctx = null;

	protected OntProperty att = null;

	protected OntClass owner = null;

	protected int maxCardinality = 0;

	protected int minCardinality = 0;

	protected AttributeModel(ContextModel ctx, OntProperty op, OntClass owner)
			throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::AttributeModel");
		this.ctx = ctx;
		this.att = op;
		this.owner = owner;
		initialize();
	}

	protected void initialize() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::initialize");
		initType();
		initDisplayData();
		initValue();
	}

	protected void initType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::initType");
		if (type == null) {
			try {
				type = new URI(att.getURI());
				String s = ctx.getType().toString() + "#";
				isDirect = att.getURI().startsWith(s); 
				//System.out.println("Attribute " + att.getURI() + " is " + (att.getURI().startsWith(s) ? "direct" : "not direct"));
			} catch (URISyntaxException e) {
				e.printStackTrace();
				type = null;
			}
		}
	}

	protected void initDisplayData() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::initDisplayData");
		if (displayData == null) {
			if (ctx.getP_DD() != null && att.hasProperty(ctx.getP_DD())) {
				RDFNode n = att.getPropertyValue(ctx.getP_DD());
				if (n.canAs(Individual.class)) {
					Individual in = (Individual) n.as(Individual.class);
					Resource r = in.getRDFType();
					if (r.canAs(OntClass.class)) {
						OntClass t = (OntClass) r.as(OntClass.class);
						if (t.equals(ctx.getC_DD())) {
							displayData = new DisplayData(ctx, in);
						}
					}
				}
			}
		}
	}

	protected void initValue() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::initValue");
		if (value == null) {
			OntResource r = att.getRange();
			if (r != null && r.isClass()) {
				OntClass c = r.asClass();
				if (c.hasSuperClass(ctx.getC_SIMP_VAL())) {
					value = new AttributeSimpleValueModel(ctx, c);
				} else if (c.hasSuperClass(ctx.getC_COMP_VAL())) {
					value = new AttributeComplexValueModel(ctx, c);
				} else {
					throw new IdASModelException(
							"Unknown type of attribute's value: " + c.getURI());
				}
			} else {
				throw new IdASModelException(
						"Can't determine type of attribute's value: "
								+ att.getURI());
			}
		}
	}

	public URI getType() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getType");
		return type;
	}

	public boolean isDirect() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::isDirect");
		return isDirect;
	}

	public IDisplayData getDisplayData() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getDisplayData");
		return displayData;
	}

	public int getMaxCardinality() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getMaxCardinality");
		return maxCardinality;
	}

	public int getMinCardinality() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getMinCardinality");
		return minCardinality;
	}

	protected void initCardinality() throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::initCardinality");
		for (ExtendedIterator itr = owner.listSuperClasses(true); itr.hasNext(); ) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(att)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						minCardinality = cr.getCardinality();
						maxCardinality = cr.getCardinality();
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						maxCardinality = cr.getMaxCardinality();
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						minCardinality = cr.getMinCardinality();
					}
				}
			}
		}
	}

	public IMetadataModel getMetadataModel(URI metadataID) {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getMetadataModel");
		return null;
	}

	public Iterator getMetadataModels() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getMetadataModels");
		return null;
	}

	public IAttributeValueModel getValueModel() {
		log.trace("org.eclipse.higgins.idas.model.impl.AttributeModel::getValueModel");
		return value;
	}
	

}
