/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.model.impl.AttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;

public class ComplexValue implements IComplexAttrValue {
	private Log log = LogFactory.getLog(ComplexValue.class);
	protected OntProperty property_ = null;

	protected Context context_ = null;

	protected Attribute attribute_ = null;

	// the subject which owns this complex value (through any level of attribute/complex value)
	protected Entity subject_ = null;

	protected Individual value_ = null;

	protected OntClass valueClass_ = null;

	protected IAttributeComplexValueModel valueModel_ = null;

	protected Individual ownerOfAttribute_ = null;

	protected URI type_ = null;

	/**
	 * @param context
	 * @param attribute
	 * @param container
	 * @param attributeProperty
	 * @param subjOwner
	 * @throws IdASException
	 */
	public ComplexValue(Context context, Attribute attribute, Individual container, OntProperty attributeProperty) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::ComplexValue");
		if (context == null)
			throw new IdASException("Parameter \"context\" is null.");
		if (attribute == null)
			throw new IdASException("Parameter \"attribute\" is null.");
		if (container == null)
			throw new IdASException("Parameter \"container\" is null.");
		context_ = context;
		attribute_ = attribute;
		value_ = container;
		ownerOfAttribute_ = attribute.getOwner();
		subject_ = attribute.getSubjectOwner();
		property_ = attributeProperty;
		valueModel_ = (IAttributeComplexValueModel)attribute_.getModel().getValueModel();
		String valueModelClass = valueModel_.getType().toString();
		valueClass_ = context_.getOntClass(valueModelClass);
		if (valueClass_ == null)
			throw new IdASException("Could not find ontology class with type = " + valueModelClass);
	}

	/**
	 * @return <code>Entity</code> which owns this attribute (directly or through any level of complex value)
	 */
	public Entity getSubjectOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getSubjectOwner");
		return subject_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#getType()
	 */
	public URI getDataType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getType");
		return getValueType();
	}

	public URI getValueType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueType");
		if (type_ == null)
			type_ = URI.create(valueClass_.getURI());
		return type_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#isSimple()
	 */
	public boolean isSimple() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::isSimple");
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IAttributeValue#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::remove");
		context_.registerChangedSubject(subject_);
		ownerOfAttribute_.removeProperty(property_, value_);
		ModelUtils.removeIndividual(context_, value_);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addAttribute(URI)");
		return getAttribute(attrID);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::addAttribute");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IAttribute newAttr = getAttribute(copyFrom.getAttrID());
		Iterator itrVal = copyFrom.getValues();
		context_.registerChangedSubject(subject_);
		while (itrVal.hasNext()) {
			IAttributeValue value = (IAttributeValue) itrVal.next();
			newAttr.addValue(value);
		}
		return newAttr;
	}

	
	public IAttribute getAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttribute");
		if (attrID == null)
			throw new IdASException("Parameter \"attrID\" is null.");
		String uri = attrID.toString();
		IAttributeModel am = valueModel_.getAttributeModel(attrID);
		if (am == null)
			throw new IdASException("Unsupported attribute " + uri + "for entity with type " + type_.toString());
		OntProperty op = context_.getOntProperty(uri);
		int maxCardinality = ModelUtils.getMaxCardinality(context_, valueClass_, op);
		if (maxCardinality == 1)
			return new SingleValuedAttribute(context_, this, op, am);
		else
			return new Attribute(context_, this, op, am);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttributes");
		List attrs = new ArrayList();
		Iterator itr = valueModel_.getAttributeModels();
		while(itr.hasNext()) {
			AttributeModel attrModel = (AttributeModel)itr.next();
			URI attrType = attrModel.getType();
			if (attrType == null)
				throw new IdASException("Can not get the type of attribute model for Entity with type " + valueClass_.getURI());
			OntProperty property = context_.getModel().getOntProperty(attrType.toString());
			if (property == null)
				throw new IdASException("Can not get object property with type " + attrType);
			int maxCardinality = ModelUtils.getMaxCardinality(context_, valueClass_, property);
			if (maxCardinality == 1)
				attrs.add(new SingleValuedAttribute(context_, this, property, attrModel));
			else
				attrs.add(new Attribute(context_, this, property, attrModel));
		}
		return attrs.iterator();
	}
	
	
	/**
	 * @return Individual of this value
	 * @throws IdASException
	 */
	public Individual getValueIndividual() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueIndividual");
		return value_;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public OntClass getValueClass() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueClass");
		return valueClass_;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public IAttributeComplexValueModel getValueModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getValueModel");
		return valueModel_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getSingleValuedAttribute(java.net.URI)
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue::getSingleValuedAttribute");
		String idURI = attrID.toString();
		IAttribute attr = getAttribute(attrID);
		if (attr == null)
			return null;
		if (attr instanceof ISingleValuedAttribute == false)
			throw new IdASException("Attribute " + idURI + " is not an instance of ISingleValuedAttribute");
		return (ISingleValuedAttribute) attr;
	}

	public boolean equals(IAttributeValue value) throws IdASException {
		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		throw new NotImplementedException();
	}

	public IEntityModel getModel() throws IdASException {
		return valueModel_;
	}


}
