/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class EntityModel implements IEntityModel {
	private Log log = LogFactory.getLog(EntityModel.class);

	private ContextModel ctx_ = null;

	private OntClass entityClass_ = null;

	private URI type_ = null;

	private String parentType_ = null;

	private List allAttrList_ = null;

	private List ownAttrList_ = null;

	private List inheritedAttrList_ = null;

	private List subEntities_ = null;

	private Hashtable attrMap_ = null;

	private String modelID_ = null;

	private HashMap attrToMinCardinality_ = null;

	private HashMap attrToMaxCardinality_ = null;

	private EntityModel parent_ = null;

	public EntityModel(OntClass entityClass, EntityModel parent, ContextModel ctx) throws Exception {
		entityClass_ = entityClass;
		ctx_ = ctx;
		parent_ = parent;
		subEntities_ = new ArrayList();
		initialize();
	}

	private void initialize() throws Exception {
		modelID_ = ModelUtils.getModelShortName(ctx_.getOntModel(), entityClass_);
		initType();
		initSubType();
		initAttributes();
		initDisplayData();
		initCardinality();
	}

	private void initSubType() throws IdASException {
		OntClass cls = ModelUtils.getSuperClass(entityClass_);
		if (cls != null)
			parentType_ = cls.getURI();
	}

	private void initType() throws IdASException {
		try {
			type_ = new URI(entityClass_.getURI());
		} catch (URISyntaxException e) {
			log.error(e);
			throw new IdASModelException("The entity class " + entityClass_.getURI() + " is not valid URI", e);
		}
	}

	private void initAttributes() throws IdASException {
		attrMap_ = new Hashtable();
		allAttrList_ = new ArrayList();
		ArrayList own = ModelUtils.getOwnPropertiesByDomain(ctx_.getOntModel(), entityClass_);
		ownAttrList_ = initAttributes(own);
		ArrayList inherited = ModelUtils.getInheritedPropertiesByDomain(ctx_.getOntModel(), entityClass_);
		inheritedAttrList_ = initAttributes(inherited);
		allAttrList_.addAll(ownAttrList_);
		allAttrList_.addAll(inheritedAttrList_);
	}

	private ArrayList initAttributes(ArrayList properties) throws IdASException {
		ArrayList attributeModels = new ArrayList();
		OntModel mdl = ctx_.getOntModel();
		for (int i = 0, size = properties.size(); i < size; i++) {
			OntProperty p = (OntProperty) properties.get(i);
			IAttributeModel m = null;
			if (p.isDatatypeProperty()) {
				DatatypeProperty dp = p.asDatatypeProperty();
				m = new AttributeModel(ctx_, dp, this);
				attributeModels.add(m);
				attrMap_.put(m.getType(), m);
			} else if (p.isObjectProperty()) {
				if (ModelUtils.isPropertyRelative(ctx_.getHigginsVocabulary().getEntityRelation(), p)) {
					initRelation();
				} else if (ModelUtils.isPropertyRelative(ctx_.getHigginsVocabulary().getEntityCorrelation(), p)) {
					initCorelation();
				} else {
					ObjectProperty op = p.asObjectProperty();
					m = new AttributeModel(ctx_, op, this);
					attributeModels.add(m);
					attrMap_.put(m.getType(), m);
				}
			} else
				continue;
		}
		return attributeModels;
	}

	private void initRelation() throws IdASException {
	}

	private void initCorelation() throws IdASException {
	}

	private void initDisplayData() throws IdASException {
	}

	public Iterator getAttributeModels() {
		return allAttrList_.iterator();
	}

	public Iterator getOwnAttributeModels() {
		return ownAttrList_.iterator();
	}

	public Iterator getInheritedAttributeModels() {
		return inheritedAttrList_.iterator();
	}

	public URI getType() {
		return type_;
	}

	public IAttributeModel getAttributeModel(URI type) {
		if (type == null)
			return null;
		if (attrMap_.containsKey(type))
			return (IAttributeModel) attrMap_.get(type);
		else
			return null;
	}

	public IModel getModel(URI modelURI) {
		return null;
	}

	public OntClass getOntClass() {
		return entityClass_;
	}

	public String getId() {
		return modelID_;
	}

	public String getParentType() {
		return parentType_;
	}

	public int getMinCardinality(String attrType) {
		if (attrToMinCardinality_.containsKey(attrType)) {
			Integer i = (Integer) attrToMinCardinality_.get(attrType);
			return i.intValue();
		} else
			return 0;
	}

	public int getMaxCardinality(String attrType) {
		if (attrToMaxCardinality_.containsKey(attrType)) {
			Integer i = (Integer) attrToMaxCardinality_.get(attrType);
			return i.intValue();
		} else
			return 0;
	}

	private void initCardinality() throws IdASException {
		attrToMinCardinality_ = new HashMap();
		attrToMaxCardinality_ = new HashMap();
		OntClass cls = entityClass_;
		while (cls != null) {
			for (ExtendedIterator itr = cls.listSuperClasses(true); itr.hasNext();) {
				OntClass c = (OntClass) itr.next();
				if (c.isRestriction()) {
					Restriction r = c.asRestriction();
					Property prop = r.getOnProperty();
					if (prop != null) {
						String propType = prop.getURI();
						if (r.isCardinalityRestriction()) {
							CardinalityRestriction cr = r.asCardinalityRestriction();
							Integer cardinality = new Integer(cr.getCardinality());
							attrToMinCardinality_.put(propType, cardinality);
							attrToMaxCardinality_.put(propType, cardinality);
						} else if (r.isMaxCardinalityRestriction()) {
							MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
							Integer cardinality = new Integer(cr.getMaxCardinality());
							attrToMaxCardinality_.put(propType, cardinality);
						} else if (r.isMinCardinalityRestriction()) {
							MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
							Integer cardinality = new Integer(cr.getMinCardinality());
							attrToMinCardinality_.put(propType, cardinality);
						}

					}
				}
			}
			cls = ModelUtils.getSuperClass(cls);
		}
	}

	public boolean isBlankNode() {
		return false;
	}

	public boolean isSimple() {
		return false;
	}

	public IEntityModel getParentEntity() {
		// TODO Auto-generated method stub
		return parent_;
	}

	public void addSubEntity(EntityModel child) {
		subEntities_.add(child);
	}

	public List getSubEntities() {
		// TODO Auto-generated method stub
		return subEntities_;
	}

	public IDisplayData getDisplayData() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}
}
