/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.authentication;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.AuthenticationException;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.common.AuthNSelfIssuedMaterials;
import org.eclipse.higgins.idas.cp.jena2.IAuthenticationModule;
import org.eclipse.higgins.idas.cp.jena2.IUserAccount;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;
import org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.RDFNode;

public class AuthenticationModule implements IAuthenticationModule {
	private Log log = LogFactory.getLog(AuthenticationModule.class);

	protected OntModel model_ = null;

	protected OntClass accountClass_ = null;

	protected OntProperty nameProperty_ = null;

	protected OntProperty passwordHashProperty_ = null;

	protected OntProperty tokenProperty_ = null;

	protected Context context_ = null;

	/**
	 * @param context
	 * @throws IdASException
	 */
	public AuthenticationModule(Context context) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::AuthenticationModule");
		if (context == null)
			throw new IdASException("The parameter \"model\" is null");
		context_ = context;
		model_ = context.getModelNoException();
		accountClass_ = initClass(AuthConstants.USER_ACCOUNT_CLASS);
		nameProperty_ = initProperty(AuthConstants.USER_NAME_PROPERTY);
		passwordHashProperty_ = initProperty(AuthConstants.USER_PASSWORD_HASH_PROPERTY);
		tokenProperty_ = initProperty(AuthConstants.USER_TOKEN_PROPERTY);
	}

	/**
	 * @param credentials
	 * @return
	 * @throws IdASException
	 */
	public IUserAccount authenticateNamePasswordMaterials(AuthNNamePasswordMaterials credentials) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::authenticateNamePasswordMaterials");
		String name = credentials.getUsername();
		if (name == null)
			throw new IdASException("AuthNNamePasswordMaterials contains null username.");
		name = name.trim();
		if (name.length() == 0)
			throw new IdASException("AuthNNamePasswordMaterials contains empty string username.");
		String password = credentials.getPassword();
		ArrayList list = getAccountIndividuals(name);
		Individual userInd = null;
		switch (list.size()) {
		case 0: {
			try {
				userInd = createNewAccount(name, password);
				context_.applyUpdates();
			} catch (Exception e) {
				log.error(e);
				context_.cancelUpdates();
				throw new IdASException();
			}
			break;
		}
		case 1: {
			try {
				userInd = (Individual) list.get(0);
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e);
			}
			checkPassword(userInd, password);
			break;
		}
		default:
			throw new IdASException("There are more then one user with name = " + name);
		}
		return new PasswordBasedUserAccount(model_, userInd, credentials);
	}

	/**
	 * Find Nodes with simple attribute <code>AuthConstants.USER_PPID_PROPERTY</code>
	 * @param ppid
	 * @return
	 * @throws IdASException
	 */
	private Individual getNodeByPPIDAttribute(String ppid) throws IdASException {
		if (ppid == null)
			return null;
		ISimpleAttrValue sv = context_.buildSimpleAttrValue(ISimpleAttrValue.STRING_TYPE_URI, ppid);
		IFilter f = context_.buildFilter();
		f.setOperator(IFilter.OP_AND);
		IFilterAttributeAssertion fas = context_.buildAttributeAssertion();
		fas.includeSubtypes(true);
		fas.setComparator(FilterAttributeAssertion.COMP_ATTR_EQ);
		fas.setAssertionValue(sv);
		fas.setID(URI.create(AuthConstants.USER_PPID_PROPERTY));
		f.addFilter(fas);
		ArrayList lst = context_.getNodeIndividuals(f, true);
		if (lst.size() == 0)
			return null;
		if (lst.size() > 1)
			throw new AuthenticationException("There are more than one Nodes with the same value of " + AuthConstants.USER_PPID_PROPERTY + " attribute" );
		return (Individual)lst.get(0);
	}

	/**
	 * @param ctxNode
	 * @return
	 * @throws IdASException
	 */
	private String getUserTokenByNode(Individual ctxNode) throws IdASException {
		if (ctxNode == null)
			throw new IdASException("Parameter \"node\" is null.");
		DatatypeProperty dp = context_.getModelNoException().getDatatypeProperty(AuthConstants.USER_TOKEN_PROPERTY);
		if (dp == null)
			throw new IdASException("Can not get datatype property by uri = " + AuthConstants.USER_TOKEN_PROPERTY);
		RDFNode node = ctxNode.getPropertyValue(dp);
		if (node == null)
			throw new IdASException("Individual of Node does not have user token property value.");
		if (node.isLiteral()) {
			Literal lt = (Literal)node.as(Literal.class);
			return lt.getString();
		}
		else
			throw new IdASException("User token property expected to contain literal value.");
	}

	/**
	 * @param token
	 * @return
	 * @throws IdASException
	 */
	private Individual getUserAccountByToken(String token) throws IdASException {
		if (token == null)
			throw new IdASException("Parameter \"token\" is null.");
		String query =
			"SELECT  ?ind \n" +
			"WHERE { \n" +
			"\t ?ind <" + AuthConstants.USER_TOKEN_PROPERTY + "> \"" + token +  "\". \n" +
			"\t ?ind <http://www.w3.org/1999/02/22-rdf-syntax-ns#type>  <" + AuthConstants.USER_ACCOUNT_CLASS + ">. \n" +
			"\n}";
		log.debug(query);
		ArrayList list = new ArrayList();
		Query q = QueryFactory.create(query);
		QueryExecution qexec = QueryExecutionFactory.create(q, context_.getModelNoException());
		try {
			ResultSet results = qexec.execSelect();
			while (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get("ind");
				Individual ind = (Individual) a.as(Individual.class);
				list.add(ind);
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		} finally {
			qexec.close();
		}
		log.debug("Number of selected individuals by query : " + list.size());
		if (list.size() == 1)
			return (Individual)list.get(0);
		else
			throw new IdASException("Number of user accounts with required user token = " + list.size());
	}

	/**
	 * @param credentials
	 * @return
	 * @throws IdASException
	 */
	public IUserAccount authenticateSelfIssuedMaterials(AuthNSelfIssuedMaterials credentials) throws IdASException {
		if (credentials == null)
			return null;
		String ppid = null;
		ByteArrayOutputStream cardKey = new ByteArrayOutputStream();
		try {
			cardKey.write(credentials.getPPIDBytes());
			cardKey.write(credentials.getPublicKeyModBytes());
			cardKey.write(credentials.getPublicKeyExpBytes());
			MessageDigest md = MessageDigest.getInstance("SHA-1");
			byte[] hash = md.digest(cardKey.toByteArray());
			ppid = Base64.encode(hash).trim();
		}
		catch (IOException e) {
			log.error(e);
			throw new IdASException(e);
		}
		catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new IdASException(e);
		}
		Individual ds = getNodeByPPIDAttribute(ppid);
		String userToken = getUserTokenByNode(ds);
		Individual ind = getUserAccountByToken(userToken);
		if (ind == null)
			return null;
		else
			return new PPIDBasedUserAccount(model_, ind, credentials);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IAuthenticationModule#authenticate(java.lang.Object)
	 */
	public IUserAccount authenticate(Object credentials) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::authenticate");
		if (credentials instanceof AuthNNamePasswordMaterials) {
			AuthNNamePasswordMaterials am = (AuthNNamePasswordMaterials) credentials;
			return authenticateNamePasswordMaterials(am);
		} 
		else if (credentials instanceof AuthNSelfIssuedMaterials) {
			AuthNSelfIssuedMaterials am = (AuthNSelfIssuedMaterials) credentials;
			return authenticateSelfIssuedMaterials(am);
			
		} else
			throw new IdASException("Unsupported credentials type.");
	}

	/**
	 * @param name
	 * @param password
	 * @return
	 * @throws IdASException
	 */
	private Individual createNewAccount(String name, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::createNewAccount");
		Individual userInd = model_.createIndividual(accountClass_);
		Literal nameLtr = model_.createLiteral(name);
		userInd.setPropertyValue(nameProperty_, nameLtr);
		String token = name;
		Literal tokenLtr = model_.createLiteral(token);
		userInd.setPropertyValue(tokenProperty_, tokenLtr);
		setNewPassword(userInd, password);
		return userInd;
	}

	/**
	 * @param user
	 * @param password
	 * @throws IdASException
	 */
	private void setNewPassword(Individual user, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::setNewPassword");
		String newHash = null;
		byte[] bytes = getHash(password);
		if (bytes != null)
			newHash = Base64.encode(bytes);
		if (newHash != null) {
			Literal ltr = model_.createLiteral(newHash);
			user.setPropertyValue(passwordHashProperty_, ltr);
		} else {
			RDFNode node = user.getPropertyValue(passwordHashProperty_);
			if (node != null)
				user.removeProperty(passwordHashProperty_, node);
		}
	}

	/**
	 * @param userInd
	 * @param password
	 * @throws IdASException
	 */
	private void checkPassword(Individual userInd, String password) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::checkPassword");
		String passHash = null;
		RDFNode rdfNode = userInd.getPropertyValue(passwordHashProperty_);
		if (rdfNode.isLiteral() == false)
			throw new IdASException("Value of property " + passwordHashProperty_.getURI() + " is not literal.");
		Literal litr = (Literal) rdfNode.as(Literal.class);
		Object obj = litr.getValue();
		if (obj != null)
			passHash = obj.toString();
		if (passHash == null || passHash.length() == 0) {
			if (password != null && password.length() > 0)
				throw new IdASException("Wrong password.");
		} else {
			if (password == null || password.length() == 0)
				throw new IdASException("Wrong password.");
			byte[] realPassHash = null;
			byte[] testedHash = null;
			testedHash = getHash(password);
			try {
				realPassHash = Base64.decode(passHash);
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e);
			}
			if (Arrays.equals(realPassHash, testedHash) == false)
				throw new IdASException("Wrong password.");
		}
	}

	/**
	 * @param name
	 * @return
	 * @throws IdASException
	 */
	private OntClass initClass(String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::initClass");
		OntClass ontClass = model_.getOntClass(name);
		if (ontClass == null) {
			try {
				ontClass = model_.createClass(name);
				context_.applyUpdates();
			} catch (Exception e) {
				log.error(e);
				context_.cancelUpdates();
				throw new IdASException();
			}
		}
		return ontClass;
	}

	/**
	 * @param name
	 * @return
	 * @throws IdASException
	 */
	private OntProperty initProperty(String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::initProperty");
		OntProperty prop = model_.getOntProperty(name);
		if (prop == null) {
			try {
				prop = model_.createDatatypeProperty(name);
				context_.applyUpdates();
			} catch (Exception e) {
				log.error(e);
				context_.cancelUpdates();
				throw new IdASException();
			}
		}
		return prop;
	}

	/**
	 * @param str
	 * @return
	 * @throws IdASException
	 */
	private byte[] getHash(String str) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::getHash");
		if (str != null && str.length() > 0) {
			try {
				MessageDigest md = MessageDigest.getInstance("SHA-256");
				return md.digest(str.getBytes("UTF-8"));
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e);
			}
		} else
			return null;
	}

	/**
	 * @param userName
	 * @return
	 * @throws IdASException
	 */
	public ArrayList getAccountIndividuals(String userName) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthenticationModule::getAccountIndividuals");
		String query =
				"SELECT ?ind \n" +
				"WHERE { \n" +
				"  ?ind <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <" + AuthConstants.USER_ACCOUNT_CLASS + ">. \n" +
				"  ?ind <" + AuthConstants.USER_NAME_PROPERTY + "> \"" + userName  + "\" \n" +
				"}";
		log.debug(query);
		ArrayList list = new ArrayList();
		QueryExecution qexec = null;
		try {
			Query q = QueryFactory.create(query);
			qexec = QueryExecutionFactory.create(q, context_.getModelNoException());
			ResultSet results = qexec.execSelect();
			while (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get("ind");
				Individual node = (Individual) a.as(Individual.class);
				list.add(node);
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e);
			throw new IdASException(e);
		} finally {
			try {
				if (qexec != null)
					qexec.close();
			}
			catch(Exception e1) {
				log.error(e1);
				throw new IdASException(e1);
			}
		}
		log.debug("Users count by query: " + list.size());
		return list;
	}

}
