/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.serialize.LineSeparator;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.eclipse.higgins.idas.cp.jena2.IFactoryConfiguration;
import org.eclipse.higgins.idas.cp.jena2.util.ConfigUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

public class FactoryConfiguration implements IFactoryConfiguration {
	private Log log = LogFactory.getLog(FactoryConfiguration.class);

	private Properties defaults = null;

	private String id = null;
	
	private String cachePath = null;

	public Hashtable configTable = new Hashtable();

	public FactoryConfiguration(String id) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::FactoryConfiguration");
		this.id = id;
		loadDefaults();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#getDefaultProperties()
	 */
	public Properties getDefaultProperties() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::getDefaultProperties");
		return (Properties) defaults.clone();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#getContextIDs()
	 */
	public List getContextIDs() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::getContextIDs");
		return new ArrayList(configTable.keySet());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#getContextConfig(java.net.URI)
	 */
	public IContextConfiguration getContextConfig(URI key) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::getContextConfig");
		if (key == null)
			return null;
		else
			return (IContextConfiguration)configTable.get(key);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#setContextConfig(org.eclipse.higgins.idas.cp.jena2.impl.ContextConfig)
	 */
	public void setContextConfig(IContextConfiguration config) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::setContextConfig");
		configTable.put(config.getContextID(), config);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#addContextConfig(java.net.URI,
	 *      java.util.Properties)
	 */
	public void addContextConfig(URI contextID, Properties p) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::addContextConfig");
		if (!configTable.containsKey(contextID)) {
			setContextConfig(createContextConfig(contextID, p));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#removeContextConfig(java.net.URI)
	 */
	public void removeContextConfig(URI contextID) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::removeContextConfig");
		if (configTable.containsKey(contextID)) {
			configTable.remove(contextID);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#createContextConfig(java.net.URI,
	 *      java.util.Properties)
	 */
	public IContextConfiguration createContextConfig(URI contextID,
			Properties p) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::createContextConfig");
		return new ContextConfiguration(cachePath, contextID, defaults, p);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#load(java.io.File)
	 */
	public void load(File config) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::load");
		FileInputStream in = null;
		try {
			in = new FileInputStream(config);
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			Document doc = null;
			DocumentBuilder db = dbf.newDocumentBuilder();
			doc = db.parse(new InputSource(in));
			Element root = doc.getDocumentElement();
			String id = root.getAttribute("id");
			if (id != null) {
				if (!id.equals(this.id)) {
					System.out.println("Warning: factory id doesn't match - " + this.id + "!=" + id);
				}
			}
			cachePath = ConfigUtil.resolvePath(root.getAttribute("cacheLocation")); 
			NodeList nl = root.getChildNodes();
			for (int i = 0; i < nl.getLength(); i++) {
				Node node = nl.item(i);
				if (node.getNodeType() == Node.ELEMENT_NODE) {
					Element el = (Element) node;
					if (el.getNodeName().equals("context")) {
						try {
							String uri = el.getAttribute("uri");
							if (uri.length() > 0) {
								setContextConfig(new ContextConfiguration(
										cachePath, defaults, el));
							}
						} catch (Exception e) {
							e.printStackTrace();
						}
					} else if (el.getNodeName().equals("default")) {
						updateDefaults(loadProperties(el));
					}
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.IFactoryConfiguration#save(java.io.File)
	 */
	public void save(File config) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::save");
		OutputStreamWriter writer = null;
		try {
			//writer = new OutputStreamWriter(new FileOutputStream(config));

			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document doc = builder.newDocument();
			Element root = doc.createElement("factory");
			root.setAttribute("id", id);
			root.setAttribute("cacheLocation", cachePath);
			doc.appendChild(root);
			
			Element e = doc.createElement("default");
			root.appendChild(e);
			saveProperties(e, defaults);

			Iterator itr = configTable.keySet().iterator();
			while (itr.hasNext()) {
				Object key = itr.next();
				IContextConfiguration cfg = (IContextConfiguration)configTable.get(key);
				e = doc.createElement("context");
				root.appendChild(e);
				cfg.store(e);
			}
			OutputFormat format = new OutputFormat(doc);
			format.setLineSeparator(LineSeparator.Windows);
			format.setIndenting(true);
			XMLSerializer serializer = new XMLSerializer (
			    new FileOutputStream(config), format);
			serializer.asDOMSerializer();
			serializer.serialize(doc);
			
			/*
			DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();

			DOMImplementationLS impl = 
			    (DOMImplementationLS)registry.getDOMImplementation("LS");

			LSSerializer wr = impl.createLSSerializer();
			LSOutput out = impl.createLSOutput();
			out.setByteStream(new FileOutputStream(config));
			wr.write(doc, out);
			*/
			
			/*
			DOMSource domSource = new DOMSource(doc);
			//StreamResult streamResult = new StreamResult(writer);
			StreamResult streamResult = new StreamResult(config);
			Transformer transformer = TransformerFactory.newInstance()
					.newTransformer();
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.transform(domSource, streamResult);
			*/
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	public String getCachePath() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::getCachePath");
		return cachePath;
	}

	private void loadDefaults() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::loadDefaults");
		defaults = new Properties();
		InputStream is = null;
		try {
			is = Thread.currentThread().getContextClassLoader()
					.getResourceAsStream(id + ".properties");
			defaults.load(is);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private Properties loadProperties(Element e) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::loadProperties");
		Properties props = new Properties();
		NodeList nl = e.getChildNodes();
		for (int i = 0; i < nl.getLength(); i++) {
			Node node = nl.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE
					&& node.getNodeName().equals("parameter")) {
				Element el = (Element) node;
				String name = el.getAttribute("name");
				if (name.length() > 0) {
					String value = el.getAttribute("value");
					props.put(name, value);
				}
			}
		}
		return props;
	}
	
	private void saveProperties(Element e, Properties properties) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::saveProperties");
		Document doc = e.getOwnerDocument();
		for (Enumeration en = properties.propertyNames(); en.hasMoreElements(); ) {
			String n = (String) en.nextElement();
			String v = properties.getProperty(n);
			Element p = doc.createElement("parameter");
			p.setAttribute("name", n);
			p.setAttribute("value", v);
			e.appendChild(p);
		}
	}
	
	private void updateDefaults(Properties properties) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FactoryConfiguration::updateDefaults");
		for (Enumeration en = defaults.propertyNames(); en.hasMoreElements(); ) {
			String n = (String) en.nextElement();
			String v = properties.getProperty(n, defaults.getProperty(n));
			defaults.setProperty(n, v);
		}
	}
}
