/**
 * Copyright (c) 2006 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.eclipse.higgins.idas.cp.jena2.util.ConfigUtil;

import com.hp.hpl.jena.db.DBConnection;
import com.hp.hpl.jena.db.IDBConnection;
import com.hp.hpl.jena.ontology.OntDocumentManager;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.ontology.Ontology;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.ModelMaker;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class JDBCContext extends Context {
	private Log log = LogFactory.getLog(JDBCContext.class);

	protected String dbURL_ = null;

	protected String dbUser_ = null;

	protected String dbPassword_ = null;

	protected String dbEngineName_ = null;

	protected String dbClass_ = null;

	protected IDBConnection conn_ = null;

	protected ModelMaker maker_ = null;

	public JDBCContext() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::JDBCContext");
	}

	public JDBCContext(OntDocumentManager documentManager, IContextConfiguration config) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::JDBCContext(OntDocumentManager, IContextConfiguration)");
		initialize(config);
	}

	public JDBCContext(OntDocumentManager documentManager, IContextId contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::JDBCContext(OntDocumentManager, IContextId)");
		initialize(contextID);
	}

	public void initialize(IContextId contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::initialize(IContextId)");
		if (contextID == null)
			throw new IdASException("Parameter \"contextID\" is null.");
		Map config = contextID.getConfiguration();
		try {
			dbURL_ = (String)config.get("db.url");
			dbUser_ = (String)config.get("db.user");
			dbPassword_ = (String)config.get("db.password");
			dbEngineName_ = (String)config.get("db.engine");
			dbClass_ = (String)config.get("db.driver");
//			schemaURL_ = (String)config.get("schema.url");
//			schemaFileName_ = (String)config.get("schema.file");
			ns_ = (String)config.get("context.uri");
			String schemaCache = (String)config.get("schema.cache");
			String publicURI = (String) config.get("schema.url"); 
			String localSchemaFile = ConfigUtil.getFilePath((String) config.get("schema.file"), schemaCache);
			setSchemaURI(publicURI, localSchemaFile);
			if (config.containsKey("validateSubjects")) {
				validateChangedSubjects_ = Boolean.valueOf((String)config.get("validateSubjects")).booleanValue();	
			}
			contextModel_ = JDBCContextFactory.getContextModel(localSchemaFile, publicURI);
		}
		catch(ClassCastException e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
	}

	private String getProp(Properties props, String propertyName) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::getProp");
		if (props == null)
			throw new IdASException("Parameter props is null");
		if (props.containsKey(propertyName))
			return props.get(propertyName).toString();
		else
			throw new IdASException("Property " + propertyName + " was not found");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#init(java.util.Properties)
	 */
	public void init(Properties props) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::init");
		dbURL_ = getProp(props, "db.url");
		dbUser_ = getProp(props, "db.user");
		dbPassword_ = getProp(props, "db.password");
		dbEngineName_ = getProp(props, "db.engine");
		dbClass_ = getProp(props, "db.driver");
		schemaURL_ = getProp(props, "schema.url");
		schemaFileName_ = getProp(props, "schema.file");
		ns_ = getProp(props, "uri");
	}

	public void initialize(IContextConfiguration config) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::initialize");
		dbURL_ = config.getProperty("db.url");
		if (dbURL_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " db.url=" + dbURL_);
		}
		dbUser_ = config.getProperty("db.user");
		if (dbUser_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " db.user=" + dbUser_);
		}
		dbPassword_ = config.getProperty("db.password");
		if (dbPassword_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " db.password=" + dbPassword_);
		}
		dbEngineName_ = config.getProperty("db.engine");
		if (dbEngineName_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " db.engine=" + dbEngineName_);
		}
		dbClass_ = config.getProperty("db.driver");
		if (dbClass_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " db.driver=" + dbClass_);
		}

		ns_ = config.getContextID().toString();
		if (ns_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID());
		}
		String schemaCache = (String)config.getProperty("schema.cache");
		String publicURI = (String) config.getProperty("schema.url"); 
		String localSchemaFile = ConfigUtil.getFilePath((String) config.getProperty("schema.file"), schemaCache);
		setSchemaURI(publicURI, localSchemaFile);
		contextModel_ = JDBCContextFactory.getContextModel(localSchemaFile, publicURI);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.AbstractContext#init()
	 */

	/**
	 * @return
	 * @throws IdASException
	 */
	private OntModel getSchemaModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::getSchemaModel");
		InputStream schemaIS = null;
		File schemaFile = null;
		if (schemaFileName_ != null)
			schemaFile = new File(schemaFileName_);
		try {
			if (schemaFile != null && schemaFile.exists() && schemaFile.isFile()) {
				schemaIS = new FileInputStream(schemaFile);
			} else {
				URL schemaURL = new URL(schemaURL_);
				schemaIS = schemaURL.openStream();
			}
			OntModel model = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
			model.setDynamicImports(true);
			model.read(schemaIS, schemaURL_);
			return model;
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		}
	}
	
	
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.Context#init()
	 */
	protected void init() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::init");
		try {
			Class.forName(dbClass_);
			conn_ = new DBConnection(dbURL_, dbUser_, dbPassword_, dbEngineName_);
			maker_ = ModelFactory.createModelRDBMaker(conn_);
			OntModelSpec spec = new OntModelSpec(OntModelSpec.OWL_DL_MEM);
			spec.getDocumentManager().addAltEntry(schemaURL_, schemaFileName_);

			//we need to rid of imported models from the DB schema, they should read in from files or discovered from url and stay in memory
			//spec.setImportModelMaker(maker_);
			spec.setImportModelMaker(ModelFactory.createFileModelMaker((new File(schemaFileName_)).getAbsolutePath()));
			
			Model base = null;
			if (maker_.hasModel(ns_)) {
				base = maker_.openModel(ns_);
				model_ = ModelFactory.createOntologyModel(spec, base);
			} else {
				base = maker_.createModel(ns_);
				model_ = ModelFactory.createOntologyModel(spec, base);
				model_.setDynamicImports(true);
				
				String schema = getSchemaURI();
				if (schema == null)
					throw new IdASException("Couldn't get schema URI");
				model_.createOntology(schema);
				Resource imp = model_.createResource(schema);
				Property importProp = model_.getProperty("http://www.w3.org/2002/07/owl#imports");
				Ontology ont = model_.createOntology("");
				ont.addProperty(importProp, imp);
				Model baseModel = model_.getBaseModel();
				// add namespaces from schema
				baseModel.setNsPrefix("schema", schemaURL_);
				ExtendedIterator ei = model_.listImportedModels();
				while (ei.hasNext()) {
					Model m = (Model) ei.next();
					Map nsMap = m.getNsPrefixMap();
					Iterator keys = nsMap.keySet().iterator();
					while (keys.hasNext()) {
						String key = (String) keys.next();
						String value = (String) nsMap.get(key);
						if (baseModel.getNsURIPrefix(value) == null)
							baseModel.setNsPrefix(key, value);
					}
				}
			}

		} catch (Exception e) {
			try {
				if (conn_ != null)
					conn_.close();
			} catch (Exception ex) {
				log.error(ex);
				throw new IdASException(ex);
			}
			log.error(e);
			throw new IdASException(e);
		}
	}
	
	
	
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.AbstractContext#save()
	 */
	protected void save() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::save");
		try {
			maker_.close();
			conn_.close();
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#begin()
	 */
	public void begin() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::begin");
		// TODO Auto-generated method stub
		if (model_.supportsTransactions()) {
			try {
				model_.begin();
			}
			catch(Exception e) {
				log.error(e);
				throw new IdASException(e.getMessage());
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#commit()
	 */
	public void commit() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::commit");
		if (model_.supportsTransactions()) {
			try {
				model_.commit();
			}
			catch(Exception e) {
				log.error(e);
				throw new IdASException(e.getMessage());
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#isTransactionSupported()
	 */
	public boolean isTransactionSupported() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::isTransactionSupported");
		return model_.supportsTransactions();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.cp.jena2.IJenaContext#rollback()
	 */
	public void rollback() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::rollback");
		if (model_.supportsTransactions()) {
			try {
				model_.abort();
			}
			catch(Exception e) {
				log.error(e);
				throw new IdASException(e.getMessage());
			}
		}
		else {
			log.error("Can not rollback changes. Database does not support transactions.");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#cancelUpdates()
	 */
	public void cancelUpdates() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.JDBCContext::cancelUpdates");
		rollback();
	}

}
