/**
 * Copyright (c) 2006 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.eclipse.higgins.idas.cp.jena2.util.ConfigUtil;

import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.ontology.Ontology;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFWriter;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class FileContext extends Context {
	private Log log = LogFactory.getLog(FileContext.class);

	protected File contextFile_ = null;

	public FileContext() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::FileContext");
	}

	public FileContext(IContextConfiguration config) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::FileContext(IContextConfiguration)");
		initialize(config);
	}

	public FileContext(IContextId contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::FileContext(IContextId)");
		initialize(contextID);
	}

	public void initialize(IContextId contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::initialize(IContextId)");
		if (contextID == null)
			throw new IdASException("Parameter \"contextID\" is null.");
		Map config = contextID.getConfiguration();
		try {
			contextFile_ = new File((String) config.get("context.file"));
			ns_ = (String) config.get("context.uri");
			String schemaCache = (String)config.get("schema.cache");
			String publicURI = (String) config.get("schema.url"); 
			String localSchemaFile = ConfigUtil.getFilePath((String) config.get("schema.file"), schemaCache);
			setSchemaURI(publicURI, localSchemaFile);
			if (config.containsKey("validateSubjects")) {
				validateChangedSubjects_ = Boolean.valueOf((String)config.get("validateSubjects")).booleanValue();	
			}
			contextModel_ = FileContextFactory.getContextModel(localSchemaFile, publicURI);
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		}
	}

	/**
	 * @param props
	 * @param propertyName
	 * @return
	 * @throws IdASException
	 */
	private String getProp(Properties props, String propertyName) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::getProp");
		if (props == null)
			throw new IdASException("Parameter props is null");
		if (props.containsKey(propertyName))
			return props.get(propertyName).toString();
		else
			throw new IdASException("Property " + propertyName + " was not found");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.IModelContext#init(java.util.Properties)
	 */
	public void init(Properties props) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::init");
		contextFile_ = new File(getProp(props, "context.file"));
		schemaURL_ = getProp(props, "schema.url");
		schemaFileName_ = getProp(props, "schema.file");
		ns_ = getProp(props, "uri");
	}

	public void initialize(IContextConfiguration config) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::initialize(IContextConfiguration)");
		contextFile_ = new File(config.getProperty("context.file"));
		if (contextFile_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID() + " context.file=" + contextFile_);
		}

		ns_ = config.getContextID().toString();
		if (ns_ == null) {
			throw new IdASException("Invalid context configuration: contextID=" + config.getContextID());
		}
		String schemaCache = (String)config.getProperty("schema.cache");
		String publicURI = (String) config.getProperty("schema.url"); 
		String localSchemaFile = ConfigUtil.getFilePath((String) config.getProperty("schema.file"), schemaCache);
		setSchemaURI(publicURI, localSchemaFile);
		contextModel_ = FileContextFactory.getContextModel(localSchemaFile, publicURI);
	}

	/**
	 * @throws IdASException
	 */
	private void loadOntology() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::loadOntology");
		FileInputStream fis = null;
		try {
			model_ = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
			fis = new FileInputStream(contextFile_);
			model_.read(fis, ns_);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new IdASException(e);
		} finally {
			try {
				fis.close();
			} catch (IOException e) {
				log.error(e);
			}
		}
	}

	private void createOntology() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::createOntology");
		model_ = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
		if (schemaFileName_ != null && schemaFileName_.trim().length() > 0) {
			File fileSchema = new File(schemaFileName_);
			if (fileSchema.exists() && fileSchema.isFile() == true)
				model_.getDocumentManager().addAltEntry(schemaURL_, schemaFileName_);
		}
		model_.setDynamicImports(true);
		String schema = getSchemaURI();
		if (schema == null)
			throw new IdASException("Couldn't get schema URI");
		model_.createOntology(schema);
		Resource imp = model_.createResource(schema);
		Property importProp = model_.getProperty("http://www.w3.org/2002/07/owl#imports");
		Ontology ont = model_.createOntology("");
		ont.addProperty(importProp, imp);
		Model baseModel = model_.getBaseModel();
		// add namespaces from schema
		baseModel.setNsPrefix("schema", schemaURL_);
		ExtendedIterator ei = model_.listImportedModels();
		while (ei.hasNext()) {
			Model m = (Model) ei.next();
			Map nsMap = m.getNsPrefixMap();
			Iterator keys = nsMap.keySet().iterator();
			while (keys.hasNext()) {
				String key = (String) keys.next();
				String value = (String) nsMap.get(key);
				if (baseModel.getNsURIPrefix(value) == null)
					baseModel.setNsPrefix(key, value);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.Context#init()
	 */
	protected void init() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::init");
		try {
			if (contextFile_.exists())
				loadOntology();
			else {
				if (schemaURL_ != null)
					createOntology();
				else
					throw new IdASException("Couldn't create context without schema");
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.AbstractContext#save()
	 */
	protected void save() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::save");
		FileOutputStream fos = null;
		RDFWriter w = model_.getWriter("RDF/XML-ABBREV");
		w.setProperty("showXmlDeclaration", "true");
		w.setProperty("xmlbase", ns_);
		w.setProperty("allowBadURIs", "true");
		w.setProperty("relativeURIs", "same-document,relative");
		try {
			File dir = contextFile_.getParentFile();
			if (dir != null && !dir.exists())
				dir.mkdirs();
			File tmpFile = File.createTempFile("tmp", ".$");
			fos = new FileOutputStream(tmpFile);
			w.write(model_.getBaseModel(), fos, ns_);
			fos.flush();
			fos.close();
			if (!contextFile_.exists() || contextFile_.delete())
				tmpFile.renameTo(contextFile_);
			else
				throw new IdASException("Couldn't save context");
		} catch (Exception e) {
			try {
				if (fos != null)
					fos.close();
			} catch (Exception ex) {
				log.error(ex);
			}
			log.error(e);
			throw new IdASException(e);
		}
	}

	public boolean isTransactionSupported() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::isTransactionSupported");
		return true;
	}

	public void begin() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::begin");
	}

	public void commit() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::commit");
		save();
	}

	public void rollback() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::rollback");
		init();
	}

	public void cancelUpdates() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.FileContext::cancelUpdates");
		rollback();
	}

}
