/**
 * Copyright (c) 2006 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.util;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.impl.Attribute;
import org.eclipse.higgins.idas.cp.jena2.impl.ComplexValue;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;
import org.eclipse.higgins.idas.cp.jena2.impl.Entity;
import org.eclipse.higgins.idas.cp.jena2.impl.SimpleValue;
import org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthConstants;
import org.eclipse.higgins.idas.model.impl.HigginsVocabulary;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class ModelUtils {
	private static Log log = LogFactory.getLog(ModelUtils.class);

	/**
	 * Checks if descendant is subclass of ancestor.
	 * 
	 * @param ancestor
	 * @param descendant
	 * @return true if descendant is subclass of ancestor
	 * @throws IdASException
	 */
	public static boolean isClassRelative(OntClass ancestor, OntClass descendant) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isClassRelative");
		if (ancestor.equals(descendant))
			return true;
		try {
			Iterator itr = descendant.listSuperClasses();
			while (itr.hasNext()) {
				OntClass cls = (OntClass) itr.next();
				if (isClassRelative(ancestor, cls))
					return true;
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	/*
	 * Iterator itr = descendant.listSuperClasses(); while (itr.hasNext()) {
	 * OntClass cls = (OntClass)itr.next(); System.out.println(cls.getURI()); if
	 * (cls.equals(ancestor)) return true; } return false; }
	 * 
	 */
	/**
	 * Checks if descendant is subproperty of ancestor.
	 * 
	 * @param ancestor
	 * @param descendant
	 * @return true if descendant is subproperty of ancestor
	 * @throws IdASException
	 */
	public static boolean isPropertyRelative(OntProperty ancestor, OntProperty descendant) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isPropertyRelative");
		OntProperty tmpDescendant = descendant;
		try {
			while (tmpDescendant != null) {
				if (tmpDescendant.hasSuperProperty(ancestor, false))
					return true;
				tmpDescendant = tmpDescendant.getSuperProperty();
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		}
		return false;
	}

	/**
	 * @param list
	 *            List of <code>OntClass</code>
	 * @param cls
	 */
	private static void fillListSuperClasses(List list, OntClass cls) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::fillListSuperClasses");
		if (cls != null && !cls.isRestriction())
			list.add(cls);
		ExtendedIterator ei = cls.listSuperClasses();
		while (ei.hasNext())
			fillListSuperClasses(list, (OntClass) ei.next());
	}

	/**
	 * @param list
	 *            List of <code>OntClass</code>
	 * @param cls
	 */
	private static void fillListSubClasses(List list, OntClass cls) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::fillListSubClasses");
		if (cls != null && !cls.isRestriction())
			list.add(cls);
		ExtendedIterator ei = cls.listSubClasses();
		while (ei.hasNext())
			fillListSubClasses(list, (OntClass) ei.next());
	}

	/**
	 * @param context
	 * @param ownerClass
	 * @return List of <code>OntClass</code> which are subClasses of
	 *         ownerClass
	 */
	public static List getListSubClasses(Context context, OntClass ownerClass) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getListSubClasses");
		List classesList = new ArrayList();
		fillListSubClasses(classesList, ownerClass);
		return classesList;
	}

	/**
	 * @param context
	 * @param ownerClass
	 * @return List of <code>OntClass</code> which are superClasses of
	 *         ownerClass
	 */
	public static List getListSuperClasses(IJenaContext context, OntClass ownerClass) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getListSuperClasses");
		List classesList = new ArrayList();
		fillListSuperClasses(classesList, ownerClass);
		return classesList;
	}

	public static boolean isAttributeOfEntity(Context context, OntProperty property) throws IdASException {
		OntClass entity = context.getModelNoException().getOntClass(HigginsVocabulary.Entity);
		ExtendedIterator ei = property.listDomain();
		while (ei.hasNext()) {
			Object node = ei.next();
			if (node instanceof OntClass) {
				OntClass domain = (OntClass) node;
				if (isClassRelative(entity, domain))
					return true;
			}
		}
		return false;
	}

	/**
	 * Checks if ownerClass can have this property
	 * 
	 * @param model
	 * @param ownerClass
	 * @param property
	 * @return true if Individual of ownerClass can have this property
	 */
	public static boolean isPropertyOfClass(Context context, OntClass ownerClass, OntProperty property) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isPropertyOfClass");
		List classesList = new ArrayList();
		fillListSuperClasses(classesList, ownerClass);
		for (int i = 0; i < classesList.size(); i++) {
			OntClass ontClass = (OntClass) classesList.get(i);
			ExtendedIterator ei = property.listDomain();
			while (ei.hasNext()) {
				Object node = ei.next();
				if (node instanceof OntClass) {
					OntClass domain = (OntClass) node;
					if (ontClass.getURI().equals(domain.getURI()))
						return true;
				}
			}
		}
		return false;
	}


	/**
	 * 
	 * @param model
	 * @param cuid
	 * @return Individual of Entity by uniqueIdentifier
	 * @throws IdASException
	 */
	public static Individual getSubjectByID(Context context, String cuid) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getSubjectByID");
		Individual digitalSubject = null;
		String queryString = "PREFIX  higgins: <" + HigginsVocabulary.NAME_SPACE + "> \n"
				+ "PREFIX  xsd: <http://www.w3.org/2001/XMLSchema#> \n" + "SELECT ?ds \n WHERE { \n" + " ?ds higgins:entityId   \"" + cuid + "\"^^xsd:string}";
		Query query = QueryFactory.create(queryString);
		QueryExecution qexec = QueryExecutionFactory.create(query, context.getModel());
		try {
			ResultSet results = qexec.execSelect();
			if (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get("ds");
				digitalSubject = (Individual) a.as(Individual.class);
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		} finally {
			qexec.close();
		}
		return digitalSubject;
	}

	/**
	 * Remove Individual and all its property values
	 * 
	 * @param model
	 * @param ind
	 * @throws IdASException
	 */
	public static void removeIndividual(Context context, Individual ind) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::removeIndividual");
		String type = (ind.getRDFType() != null) ? ind.getRDFType().getURI() : null;
		if (type == null)
			throw new IdASException("Indiividual doesn't have RDF type");
		StmtIterator si = ind.listProperties();
		List stmts = new ArrayList(); // <Statement>
		List individuals = new ArrayList(); // <Individual>(
		while (si.hasNext()) {
			Statement st = si.nextStatement();
			RDFNode node = st.getObject();
			if (node.isLiteral())
				stmts.add(st);
			else if (node.isResource()) {
				Individual child = null;
				try {
					child = (Individual) node.as(Individual.class);
				} catch (ClassCastException e) {
					throw new IdASException(e.getMessage());
				}
				if (child != null && !type.equals(child.getURI()))
					individuals.add(child);
			}
		}
		Iterator itrStmt = stmts.iterator();
		while (itrStmt.hasNext()) {
			Statement st = (Statement) itrStmt.next();
			context.getModel().remove(st);
		}
		Iterator itrIndv = individuals.iterator();
		while (itrIndv.hasNext()) {
			Individual child = (Individual) itrIndv.next();
			removeIndividual(context, child);
		}
		ind.remove();
	}

	/**
	 * @param context
	 * @param cls
	 * @param attr
	 * @return
	 * @throws IdASException
	 */
	public static int getMaxCardinality(Context context, OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMaxCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						return cr.getMaxCardinality();
					}
				}
			} else {
				int res = getMaxCardinality(context, c, attr);
				if (res != -1)
					return res;
			}
		}
		return -1;
	}

	/**
	 * @param context
	 * @param cls
	 * @param attr
	 * @return
	 * @throws IdASException
	 */
	public static int getMinCardinality(Context context, OntClass cls, OntProperty attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getMinCardinality");
		for (ExtendedIterator itr = cls.listSuperClasses(true); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						return cr.getCardinality();
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						return cr.getMinCardinality();
					}
				}
			}
		}
		return -1;
	}

	/**
	 * @param context
	 * @param subjectInd
	 * @param userToken
	 * @throws IdASException
	 */
	public static void setUserToken(Context context, Individual subjectInd, String userToken) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::setUserToken");
		OntProperty tokenProp = context.getOntProperty(AuthConstants.USER_TOKEN_PROPERTY);
		if (userToken == null || userToken.length() == 0) {
			RDFNode node = subjectInd.getPropertyValue(tokenProp);
			subjectInd.removeProperty(tokenProp, node);
		} else {
			Literal ltr = context.getModel().createLiteral(userToken);
			subjectInd.setPropertyValue(tokenProp, ltr);
		}
	}

	/**
	 * @param context
	 * @param subjectInd
	 * @return
	 * @throws IdASException
	 */
	public static String getUserToken(Context context, Individual subjectInd) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::getUserToken");
		OntProperty tokenProp = context.getOntProperty(AuthConstants.USER_TOKEN_PROPERTY);
		RDFNode tokenNode = subjectInd.getPropertyValue(tokenProp);
		if (tokenNode != null) {
			if (tokenNode.isLiteral()) {
				Literal ltr = (Literal) tokenNode.as(Literal.class);
				Object obj = ltr.getValue();
				if (obj != null)
					return obj.toString();
			} else
				throw new IdASException("Property " + AuthConstants.USER_TOKEN_PROPERTY + " contains non-literal value.");
		}
		return null;
	}

	/**
	 * @param context
	 * @return
	 */
	public static String generateUniqueID(Context context) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::generateUniqueID");
		String rnd = String.valueOf(Math.random()).substring(0);
		return String.valueOf(System.currentTimeMillis()) + rnd;
	}

	/**
	 * @param context
	 * @param subj
	 * @throws IdASException
	 */
	public static void validateSubject(Context context, Entity subj) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateSubject");
		log.debug("ContextURI = " + context.getNS() + ", subject type = " + subj.getEntityType());
		if (subj == null)
			throw new IdASException("Parameter \"subj\" is null.");
		Iterator attributeModels = null;
		IEntityModel subjModel = subj.getModel();
		if (subjModel != null)
			attributeModels = subjModel.getAttributeModels();
		Iterator itrAttr = subj.getAttributes();
		validateAttributes(context, itrAttr, attributeModels);
	}

	/**
	 * @param context
	 * @param attributes
	 * @param attributeModels
	 * @throws IdASException
	 */
	public static void validateAttributes(Context context, Iterator attributes, Iterator attributeModels) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateAttributes");
		ArrayList attrIDs = new ArrayList();
		while (attributes.hasNext()) {
			Object obj = attributes.next();
			if (obj instanceof Attribute == false)
				throw new IdASException("Instance of " + Attribute.class.getName() + " expected, " + obj.getClass().getName() + " found.");
			Attribute attr = (Attribute) obj;
			validateAttribute(context, attr);
			attrIDs.add(attr.getAttrID().toString());
		}
	}

	/**
	 * @param context
	 * @param attributeType
	 * @return
	 * @throws IdASException
	 */
	public static boolean isFunctionalProperty(Context context, String attributeType) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::isFunctionalProperty");
		OntProperty op = context.getOntProperty(attributeType);
		if (op != null && op.isFunctionalProperty() == true)
			return true;
		else
			return false;
	}

	/**
	 * Checks whether the number of values satisfy with cardinality restrictions
	 * of attribute
	 * 
	 * @param context
	 * @param attr
	 * @param realValuesCount
	 * @throws IdASException
	 */
	public static void checkCardinality(Context context, Attribute attr, int realValuesCount) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::checkCardinality");
		IAttributeModel am = attr.getModel();
		int min = am.getMinCardinality();
		int max = am.getMaxCardinality();
		if (min != 0 && realValuesCount < min)
			throw new IdASException("Wrong cardinality. Real values count = " + realValuesCount + " min cardinality = " + min);
		if (max != 0 &&  realValuesCount > max)
			throw new IdASException("Wrong cardinality. Real values count = " + realValuesCount + " max cardinality = " + max);
	}

	/**
	 * @param context
	 * @param attr
	 * @throws IdASException
	 */
	public static void validateAttribute(Context context, Attribute attr) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateAttribute");
		URI type = attr.getAttrID();
		if (type == null)
			throw new IdASException("Type(ID) of attribute is null.");
		String typeStr = type.toString();
		int valuesCount = 0;
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			valuesCount++;
			Object obj = itr.next();
			if (obj instanceof SimpleValue) {
				boolean isFunctionalProperty = isFunctionalProperty(context, typeStr);
				validateSimpleValue(context, (SimpleValue) obj, isFunctionalProperty);
			} else if (obj instanceof ComplexValue)
				validateComplexValue(context, (ComplexValue) obj);
			else {
				if (obj == null)
					throw new IdASException("Value of attribute is null.");
				else
					throw new IdASException("Unexpected class of value of attribute : " + obj.getClass().getName());
			}
		}
		checkCardinality(context, attr, valuesCount);
	}

	/**
	 * @param context
	 * @param val
	 * @throws IdASException
	 */
	public static void validateComplexValue(Context context, ComplexValue val) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateComplexValue");
		if (val != null) {
			Iterator attributeModels = null;
			IAttributeComplexValueModel valueModel = (IAttributeComplexValueModel)val.getModel();
			if (valueModel != null)
				attributeModels = valueModel.getAttributeModels();
			Iterator itrAttr = val.getAttributes();
			validateAttributes(context, itrAttr, attributeModels);
		}
	}

	/**
	 * @param context
	 * @param val
	 * @param isFunctionalProperty
	 * @throws IdASException
	 */
	public static void validateSimpleValue(Context context, SimpleValue val, boolean isFunctionalProperty) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.util.ModelUtils::validateSimpleValue");
	}


}
