/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeComplexValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class AttributeComplexValueModel implements  IAttributeComplexValueModel {
	private Log log = LogFactory.getLog(AttributeComplexValueModel.class);

	private URI type_ = null;

	private String parentType_ = null;

	private List allAttrList_ = null;

	private List ownAttrList_ = null;

	private List inheritedAttrList_ = null;

	private Hashtable attrMap_ = null;

	private ContextModel ctx_ = null;

	private OntClass valueClass_ = null;

	private ObjectProperty attr_ = null;

	private String modelID_ = null;

	private AttributeModel parent_ = null;

	private HashMap attrToMinCardinality_ = null;

	private HashMap attrToMaxCardinality_ = null;

	public AttributeComplexValueModel(ContextModel ctx, AttributeModel parent, ObjectProperty attr) throws IdASException {
		ctx_ = ctx;
		parent_ = parent;
		attr_ = attr;
		initialize();
		ctx_.registerBlankEntityModel(this);
	}

	private void initialize() throws IdASException {
		initType();
		initAttributes();
		initSubType();
		modelID_ = ModelUtils.getModelShortName(ctx_.getOntModel(), valueClass_);
		initCardinality();
	}

	private void initSubType() throws IdASException {
		OntClass cls = valueClass_.getSuperClass();
		if (cls != null)
			parentType_ = cls.getURI();
		else
			parentType_ = null;
	}

	private void initType() throws IdASException {
		try {
			OntResource range = attr_.getRange();
			if (range.isClass()) {
				valueClass_ = (OntClass) range.as(OntClass.class);
			} else
				throw new IdASException("Couldn't get range class for attribute property: " + attr_.getURI());
			type_ = new URI(valueClass_.getURI());
		} catch (URISyntaxException e) {
			log.error(e);
			throw new IdASModelException("The attributeValue class " + valueClass_.getURI() + " is not valid URI", e);
		}
	}

	private void initAttributes() throws IdASException {
		attrMap_ = new Hashtable();
		allAttrList_ = new ArrayList();
		ArrayList own = ModelUtils.getOwnPropertiesByDomain(ctx_.getOntModel(), valueClass_);
		ownAttrList_ = initAttributes(own);
		ArrayList inherited = ModelUtils.getInheritedPropertiesByDomain(ctx_.getOntModel(), valueClass_);
		inheritedAttrList_ = initAttributes(inherited);
		allAttrList_.addAll(ownAttrList_);
		allAttrList_.addAll(inheritedAttrList_);
	}

	private ArrayList initAttributes(ArrayList properties) throws IdASException {
		ArrayList attributeModels = new ArrayList();
		OntModel mdl = ctx_.getOntModel();
		for (int i = 0, size = properties.size(); i < size; i++) {
			OntProperty p = (OntProperty) properties.get(i);
			IAttributeModel m = null;
			if (p.isDatatypeProperty()) {
				DatatypeProperty dp = p.asDatatypeProperty();
				m = new AttributeModel(ctx_, dp, this);
			} else if (p.isObjectProperty()) {
				ObjectProperty op = p.asObjectProperty();
				m = new AttributeModel(ctx_, op, this);
			} else
				continue;
			attributeModels.add(m);
			attrMap_.put(m.getType(), m);
		}
		return attributeModels;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.model.IAttributeComplexValueModel#getAttributes()
	 */
	public Iterator getAttributes() {
		return allAttrList_.iterator();
	}

	public Iterator getAttributeModels() {
		return getAttributes();
	}

	public Iterator getOwnAttributeModels() {
		return ownAttrList_.iterator();
	}

	public Iterator getInheritedAttributeModels() {
		return inheritedAttrList_.iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.model.IAttributeValueModel#isSimple()
	 */
	public boolean isSimple() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.model.IModel#getType()
	 */
	public URI getType() {
		return type_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.model.IAttributeComplexValueModel#getAttribute(URI
	 *      type)
	 */
	public IAttributeModel getAttributeModel(URI type) {
		if (type != null)
			return (IAttributeModel) attrMap_.get(type);
		else
			return null;
	}

	public int getMaxCardinality() {
		return -1;
	}

	public Iterator getMetadataModels() {
		return null;
	}

	public int getMinCardinality() {
		return -1;
	}

	public OntClass getOntClass() {
		return valueClass_;
	}

	public String getParentType() {
		return parentType_;
	}

	public String getId() {
		return modelID_;
	}

	public int getMinCardinality(String attrType) {
		if (attrToMinCardinality_.containsKey(attrType)) {
			Integer i = (Integer) attrToMinCardinality_.get(attrType);
			return i.intValue();
		} else
			return 0;
	}

	public int getMaxCardinality(String attrType) {
		if (attrToMaxCardinality_.containsKey(attrType)) {
			Integer i = (Integer) attrToMaxCardinality_.get(attrType);
			return i.intValue();
		} else
			return 0;
	}

	private void initCardinality() throws IdASException {
		attrToMinCardinality_ = new HashMap();
		attrToMaxCardinality_ = new HashMap();
		for (ExtendedIterator itr = valueClass_.listSuperClasses(true); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				Property prop = r.getOnProperty();
				if (prop != null) {
					String propType = prop.getURI();
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						Integer cardinality = new Integer(cr.getCardinality());
						attrToMinCardinality_.put(propType, cardinality);
						attrToMaxCardinality_.put(propType, cardinality);
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						Integer cardinality = new Integer(cr.getMaxCardinality());
						attrToMaxCardinality_.put(propType, cardinality);
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						Integer cardinality = new Integer(cr.getMinCardinality());
						attrToMinCardinality_.put(propType, cardinality);
					}

				}
			}
		}
	}

	public boolean isBlankNode() {
		return true;
	}

	public IEntityModel getParentEntity() {
		// TODO Auto-generated method stub
		return null;
	}

	public List getSubEntities() {
		// TODO Auto-generated method stub
		return null;
	}

	public IDisplayData getDisplayData() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

}
