/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.model.impl;

import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IModel;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.rdf.model.RDFWriter;

public class ContextModel implements IContextModel {

	private Log log = LogFactory.getLog(ContextModel.class);

	private List entityList_ = null;

	private HashMap entityMap_ = null;

	private OntModel model_ = null;

	private URI baseURI_ = null;

	private HashMap entityTree_ = null;

	private HashMap complexValueTree_ = null;

	private HashMap entityTypeByHBName_ = null;

	private AttributeModel ppidOwner_ = null;

	private HigginsVocabulary vocabulary_ = null;

	public ContextModel(OntModel model, URI baseURI) throws IdASException {
		model_ = model;
		baseURI_ = baseURI;
		vocabulary_ = new HigginsVocabulary(model_);
		try {
			init();
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e);
		}
	}

	private void init() throws Exception {
		complexValueTree_ = new HashMap();
		entityTree_ = new HashMap();
		initEntities();
	}

	private void initEntities() throws Exception {
		entityList_ = new ArrayList();
		entityMap_ = new HashMap();
		entityTypeByHBName_ = new HashMap();
		OntClass entityClass = vocabulary_.getEntity();
		if (entityClass == null)
			throw new IdASException("Can not find class " + HigginsVocabulary.Entity + " in model " + baseURI_);
		addEntity(null, entityClass);
	}

	private void addEntity(EntityModel parent, OntClass enityClass) throws Exception {
		log.debug("Create EntityModel for class " + enityClass.getURI());
		EntityModel em = new EntityModel(enityClass, parent, this);
		entityList_.add(em);
		entityMap_.put(enityClass.getURI(), em);
		registerEntity(em);
		entityTypeByHBName_.put(em.getId(), em.getType());
		Iterator itr = enityClass.listSubClasses();
		while (itr.hasNext()) {
			OntClass child = (OntClass) itr.next();
			addEntity(em, child);
		}
	}

	public IModel getModel(URI type) {
		if (type == null)
			return null;
		IEntityModel nm = getEntityModel(type);
		if (nm != null)
			return nm;
		for (int i = 0, size = entityList_.size(); i < size; i++) {
			EntityModel entity = (EntityModel) entityList_.get(i);
			entity.getModel(type);
			if (entity != null)
				return entity;
		}
		return null;
	}

	public IEntityModel getEntityModel(URI entityType) {
		if (entityType == null)
			return null;
		String str = entityType.toString();
		if (entityMap_.containsKey(str))
			return (EntityModel) entityMap_.get(str);
		else
			return null;
	}

	public Iterator getEntityModels() {
		return entityList_.iterator();
	}

	public URI getType() {
		return baseURI_;
	}

	public OntModel getOntModel() {
		return model_;
	}

	public void registerBlankEntityModel(AttributeComplexValueModel bn) {
		String parentType = bn.getParentType();
		ArrayList lst = null;
		if (complexValueTree_.containsKey(parentType)) {
			lst = (ArrayList) complexValueTree_.get(parentType);
		} else {
			lst = new ArrayList();
			complexValueTree_.put(parentType, lst);
		}
		boolean contains = false;
		URI type = bn.getType();
		for (int i = 0, k = lst.size(); i < k; i++) {
			AttributeComplexValueModel v = (AttributeComplexValueModel) lst.get(i);
			if (type != null && type.equals(v.getType())) {
				contains = true;
				break;
			}
		}
		if (contains == false)
			lst.add(bn);
	}

	public void registerEntity(EntityModel nm) {
		String parentType = nm.getParentType();
		ArrayList lst = null;
		if (entityTree_.containsKey(parentType)) {
			lst = (ArrayList) entityTree_.get(parentType);
		} else {
			lst = new ArrayList();
			entityTree_.put(parentType, lst);
		}
		lst.add(nm);
	}

	public HashMap getEntitiesHash() {
		return entityTree_;
	}

	public HashMap getComplexValueHash() {
		return complexValueTree_;
	}

	public EntityModel getTopEntityModel() {
		return (EntityModel) getEntityModel(HigginsVocabulary.EntityURI);
	}

	public String getSchema() throws IdASException {
		String res = null;
		if (model_ != null) {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			RDFWriter writer = model_.getBaseModel().getWriter("RDF/XML-ABBREV");
			// RDFWriter writer = schema.getWriter("RDF/XML-ABBREV");
			writer.setProperty("xmlbase", getType().toString());
			writer.setProperty("allowBadURIs", "false");
			writer.write(model_, baos, getType().toString());
			res = baos.toString();
		}
		return res;
	}

	public AttributeModel getPPIDAttribute() {
		return ppidOwner_;
	}

	public void setPPIDAttribute(AttributeModel ppidOwner) {
		ppidOwner_ = ppidOwner;
	}

	public HigginsVocabulary getHigginsVocabulary() {
		return vocabulary_;
	}

	public String getId() {
		return "context";
	}

	public IAttributeModel getAttributeModel(URI attrID) {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator getAttributeModels() {
		// TODO Auto-generated method stub
		return null;
	}

	public IDisplayData getDisplayData() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

}
