/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.filter;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.BadFilterException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.model.impl.HigginsVocabulary;

import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class FilterAttributeAssertion extends FilterAssertion implements IFilterAttributeAssertion {
	private Log log = LogFactory.getLog(FilterAttributeAssertion.class);

	private CompoundCondition condition_ = null;

	private Context idasContext_ = null;

	private FilterContext filterContext_ = null;

	protected IAttributeValue value_ = null;

	/**
	 * @param context
	 */
	public FilterAttributeAssertion(Context context) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::FilterAttributeAssertion");
		idasContext_ = context;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAssertion#getCondition()
	 */
	public ICondition getCondition() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::getCondition");
		if (condition_ == null)
			throw new IdASException("Can't return condition. FilterAttributeAssertion should be initialized before.");
		return condition_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterPropertyAssertion#setAssertionValue(org.eclipse.higgins.idas.api.IAttributeValue)
	 */
	public void setAssertionValue(IAttributeValue assertionValue) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::setAssertionValue");
		value_ = assertionValue;
	}

	/**
	 * @param val
	 * @param condition
	 * @param attributeNode
	 * @param parent
	 * @throws IdASException
	 */
	private void initWithSimpleValue(INode parentNode, PropertyNode attrType, ISimpleAttrValue val, CompoundCondition condition, GraphPattern parent) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::initWithSimpleValue");
		GraphPattern pattern = filterContext_.registerPattern(new GraphPattern(parentNode, attrType, new LiteralNode(), parent));
		condition.addCondition(new ValueCondition((LiteralNode) pattern.getObject(), comparator_, val));
	}

	/**
	 * @param val
	 * @param condition
	 * @param attributeNode
	 * @param parent
	 * @throws IdASException
	 */
	private void initWithComplexValue(IComplexAttrValue val, CompoundCondition condition, AttributeNode attributeNode, GraphPattern parent)
			throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::initWithComplexValue");
		Iterator itAttr = val.getAttributes();
		if (itAttr != null) {
			while (itAttr.hasNext()) {
				IAttribute attr = (IAttribute) itAttr.next();
				URI attrTypeURI = attr.getAttrID();
				if (attrTypeURI == null)
					throw new IdASException("Attribute returned null AttrID");
				Iterator itVal = attr.getValues();
				if (itVal != null) {
					while (itVal.hasNext()) {
						IAttributeValue value = (IAttributeValue) itVal.next();
						if (value != null) {
							if (value.isSimple()) {
								ISimpleAttrValue simple = (ISimpleAttrValue) value;
								PropertyNode attrType = new PropertyNode(attrTypeURI.toString());
								initWithSimpleValue(attributeNode, attrType, simple, condition,  parent); 
 							} else {
								IComplexAttrValue complexVal = (IComplexAttrValue) value;
								URI attrID = attr.getAttrID();
								if (attrID == null)
									throw new IdASException("Attribute type is null.");
								AttributeNode newAttributeNode = new AttributeNode();
								PropertyNode newPropNode = new PropertyNode(attrID.toString());
								GraphPattern newPattern = filterContext_
										.registerPattern(new GraphPattern(attributeNode, newPropNode, newAttributeNode, parent));
								initWithComplexValue(complexVal, condition, newAttributeNode, newPattern);
							}
						}
					}
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAssertion#init(org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext)
	 */
	public void init(FilterContext filterContext) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::init");
		filterContext_ = filterContext;
		if (idasContext_ == null)
			throw new IdASException("Can't init FilterAttributeAssertion because context is null.");
		if (value_ == null)
			throw new IdASException("Can't init FilterAttributeAssertion because value is null.");
		if (type_ == null)
			throw new IdASException("Can't init FilterAttributeAssertion because type is null.");
		if (comparator_ == null)
			throw new IdASException("Can't init FilterAttributeAssertion because comparator is null.");
		GraphPattern pattern = null;
		String type = type_.toString();
		condition_ = new CompoundCondition(Filter.OP_AND);
		ArrayList types = new ArrayList();
		types.add(type);
		if (includeSubtypes_) {
			OntProperty prop = idasContext_.getModelNoException().getOntProperty(type);
			ExtendedIterator it = prop.listSubProperties();
			while (it.hasNext()) {
				OntProperty subProp = (OntProperty) it.next();
				types.add(subProp.getURI());
			}
		}
		CompoundCondition valuesCondition = new CompoundCondition(Filter.OP_OR);
		condition_.addCondition(valuesCondition);
		int typesCount = types.size();
		for (int i = 0; i < typesCount; i++) {
			CompoundCondition valueCondition = new CompoundCondition(Filter.OP_AND);
			valuesCondition.addCondition(valueCondition);
			String subType = (String) types.get(i);
			PropertyNode propNode = new PropertyNode(subType);
			if (value_ instanceof ISimpleAttrValue) {
				ISimpleAttrValue val = (ISimpleAttrValue) value_;
				// ModelUtils.checkAttribute(idasContext_, subType, val);
				initWithSimpleValue(filterContext_.getSubjectNode(), new PropertyNode(subType), val,  valueCondition, null);
			} else if (value_ instanceof IComplexAttrValue) {
				AttributeNode attributeNode = new AttributeNode();
				pattern = filterContext_.registerPattern(new GraphPattern(filterContext_.getSubjectNode(), propNode, attributeNode, null));
				IComplexAttrValue val = (IComplexAttrValue) value_;
				// ModelUtils.checkAttribute(idasContext_, subType, val);
				initWithComplexValue(val, valueCondition, attributeNode, pattern);
			} else
				throw new IdASException("Unexpected type of assertion value : " + value_.getClass().getName());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAssertion#setID(java.net.URI)
	 */
	public void setID(URI type) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterAttributeAssertion::setID(URI)");
		if (type == null)
			throw new IdASException("Parameter \"type\" is null.");
		String uri = type.toString();
		OntProperty attrID = idasContext_.getModelNoException().getOntProperty(uri);
		if (attrID == null)
			throw new IdASException("Couldn't find object property : " + uri);
		OntProperty higginsAttr = idasContext_.getModelNoException().getOntProperty(type.toString());
		if (higginsAttr == null)
			throw new IdASException("Can not find ont property by URI = " + type.toString());
		if (ModelUtils.isAttributeOfEntity(idasContext_, higginsAttr) == false)
			throw new IdASException("Property  " + uri + " is not an attribute of entity");
		type_ = type;
	}

	public void setAttributeFilter(IFilter attributeFilter) throws IdASException {
		throw new NotImplementedException();
	}

}
