/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.filter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IUserAccount;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;
import org.eclipse.higgins.idas.cp.jena2.impl.authentication.AuthConstants;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.RDFNode;

public class FilterContext {
	private Log log = LogFactory.getLog(FilterContext.class);

	private int varIndex_ = 0;

	private Hashtable pathToPattern_;

	private SubjectNode subjectNode_;

	private Context idasContext_;

	private ICondition queryCondition_;
	
	private boolean ignoreUserToken_ = false;

	/**
	 * @param idasContext
	 * @throws IdASException
	 */
	public FilterContext(Context idasContext) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::FilterContext");
		idasContext_ = idasContext;
	}

	/**
	 * @param idasContext
	 * @throws IdASException
	 */
	public FilterContext(Context idasContext, boolean ignoreUserToken) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::FilterContext");
		idasContext_ = idasContext;
		ignoreUserToken_ = ignoreUserToken;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public Context getIdASContext() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getIdASContext");
		return idasContext_;
	}

	/**
	 * @return
	 */
	public int getNextIndex() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getNextIndex");
		return varIndex_++;
	}

	/**
	 * @return
	 */
	public SubjectNode getSubjectNode() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getSubjectNode");
		return subjectNode_;
	}

	/**
	 * Register pattern to avoid duplication of paterns in RDF query.
	 * 
	 * @param pattern
	 * @return
	 * @throws IdASException
	 */
	public GraphPattern registerPattern(GraphPattern pattern) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::registerPattern");
		String fullPath = pattern.getFullPath();
		if (pathToPattern_.containsKey(fullPath))
			return (GraphPattern) pathToPattern_.get(fullPath);
		else
			pathToPattern_.put(fullPath, pattern);
		return pattern;
	}

	/**
	 * @param node
	 */
	private void indexNode(INode node) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::indexNode");
		if (node.indexRequired() && (node.hasIndex() == false))
			node.setIndex(getNextIndex());
	}

	/**
	 * 
	 */
	private void indexNodes() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::indexNodes");
		Iterator it = pathToPattern_.values().iterator();
		while (it.hasNext()) {
			GraphPattern pattern = (GraphPattern) it.next();
			indexNode(pattern.getObject());
			indexNode(pattern.getPredicate());
			indexNode(pattern.getSubject());
		}
	}

	/**
	 * @return
	 */
	private ArrayList getSortedPatternPaths() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getSortedPatternPaths");
		ArrayList list = new ArrayList();
		Enumeration keys = pathToPattern_.keys();
		while (keys.hasMoreElements())
			list.add(keys.nextElement());
		Collections.sort(list);
		return list;
	}

	/**
	 * @return
	 */
	private StringBuffer getPatterns() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getPatterns");
		StringBuffer buf = new StringBuffer();
		ArrayList list = getSortedPatternPaths();
		int size = list.size();
		for (int i = 0; i < size; i++) {
			String key = (String) list.get(i);
			buf.append("\t\t");
			GraphPattern pattern = (GraphPattern) pathToPattern_.get(key);
			buf.append(pattern.asQueryElement());
			buf.append(". \n");
		}
		return buf;
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	private String assembleQuery() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::assembleQuery");
		StringBuffer buf = new StringBuffer();
		buf.append("PREFIX xsd: <http://www.w3.org/2001/XMLSchema#> \n");
		buf.append("SELECT DISTINCT ");
		buf.append(subjectNode_.asQueryElement());
		buf.append("\n");
		buf.append("WHERE {\n");
		buf.append("\tOPTIONAL { \n");
		buf.append(getPatterns());
		buf.append("\t}\n");
		buf.append("\tFILTER\n");
		buf.append(queryCondition_.getCondition());
		buf.append("}\n");
		return buf.toString();
	}

	/**
	 * @param filter
	 * @return
	 * @throws IdASException
	 */
	public List getSubjectIndividuals(Filter filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getSubjectIndividuals");
		String query = getRDFQuery(filter);
		log.debug(query);	
		ArrayList list = new ArrayList();
		Query q = QueryFactory.create(query);
		QueryExecution qexec = QueryExecutionFactory.create(q, idasContext_.getModelNoException());
		try {
			ResultSet results = qexec.execSelect();
			while (results.hasNext()) {
				QuerySolution soln = results.nextSolution();
				RDFNode a = soln.get(SubjectNode.SUBJ_NODE);
				Individual digitalSubject = (Individual) a.as(Individual.class);
				list.add(digitalSubject);
			}
		} catch (Exception e) {
			log.error(e);
			throw new IdASException(e.getMessage());
		} finally {
			qexec.close();
		}
		log.debug("Number of selected Node individuals by query : " + list.size());	
		return list;
	}

	/**
	 * @param filter
	 * @return
	 * @throws IdASException
	 */
	private synchronized String getRDFQuery(Filter filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::getRDFQuery");
		subjectNode_ = new SubjectNode();
		varIndex_ = 1;
		pathToPattern_ = new Hashtable();
		filter.init(this);
		queryCondition_ = filter.getCondition();
		if (ignoreUserToken_ == false)
			addUserTokenToQuery();
		else
			log.debug("User token was ignored by FilterContext to query subjects.");
		queryCondition_.setLevel(0);
		indexNodes();
		return assembleQuery();
	}

	/**
	 * @throws IdASException
	 */
	public void addUserTokenToQuery() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.FilterContext::addUserTokenToQuery");
		IUserAccount account = idasContext_.getUserAccount();
		if (account != null) {
			String userToken = account.getUserToken();
			if (userToken != null && "".equals(userToken.trim()) == false) {
				GraphPattern pattern = registerPattern(new GraphPattern(getSubjectNode(), new PropertyNode(AuthConstants.USER_TOKEN_PROPERTY),
						new LiteralNode(), null));
				CompoundCondition cc = new CompoundCondition(Filter.OP_AND);
				cc.addCondition(queryCondition_);
				cc.addCondition(new UntypedValueCondition((LiteralNode) pattern.getObject(), IFilterAttributeAssertion.COMP_ATTR_EQ, userToken));
				queryCondition_ = cc;
			}
		}
	}

}
