/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl.filter;

import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.BadFilterException;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAssertion;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityIDAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.impl.Context;

public class Filter implements IFilter {
	private Log log = LogFactory.getLog(Filter.class);

	private IFilterAssertion assertion_ = null;

	private String operator_ = null;

	private ArrayList filters_ = null;

	private Context idasContext_ = null;

	private ICondition condition_ = null;

	public Filter(Context context) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::Filter");
		if (context == null)
			throw new IdASException("Parameter \"context\" is null");
		idasContext_ = context;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#setAssertion(org.eclipse.higgins.idas.IFilterAssertion)
	 */
	public void setAssertion(IFilterAssertion assertion) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::setAssertion");
		if (assertion instanceof FilterAssertion == false)
			throw new BadFilterException("FilterAssertion created by another context provider implementation cannot be added");
		if ((IFilter.OP_AND.equals(operator_) == false) && (IFilter.OP_OR.equals(operator_) == false) && (filters_ == null)) {
			assertion_ = assertion;
		} else {
			throw new BadFilterException("Assertion cannot be added when operator or filter is present");
		}
	}

	/**
	 * @return assertion set by calling {@link #setAssertion(IFilterAssertion)}.
	 *         May be null.
	 * @throws IdASException
	 */
	public IFilterAssertion getAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::getAssertion");
		return assertion_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#setOperator(java.lang.String)
	 */
	public void setOperator(String operator) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::setOperator");
		if (IFilter.OP_NOT.equals(operator)) {
			operator_ = operator;
		} else if (IFilter.OP_AND.equals(operator) || IFilter.OP_OR.equals(operator)) {
			if (assertion_ == null)
				operator_ = operator;
			else
				throw new BadFilterException("Operators \"OR\" or \"AND\" cannot be added when assertion is present");
		} else
			throw new BadFilterException("Unexpected operator: " + operator);
	}

	/**
	 * @return operator set by calling {@link #setOperator(String)}. May be
	 *         null.
	 * @throws IdASException
	 */
	public String getOperator() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::getOperator");
		return operator_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#addFilter(org.eclipse.higgins.idas.IFilter)
	 */
	public void addFilter(IFilter filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::addFilter(IFilter)");
		if (assertion_ == null) {
			if (filters_ == null)
				filters_ = new ArrayList();
			if (filter instanceof Filter)
				filters_.add(filter);
			else
				throw new BadFilterException("Filter created by another context provider implementation cannot be added");
		} else {
			throw new BadFilterException("Filter cannot be added when assertion is present");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#addFilter(org.eclipse.higgins.idas.IFilterAssertion)
	 */
	public void addFilter(IFilterAssertion filterAssertion) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::addFilter(IFilterAssertion)");
		if (filterAssertion instanceof FilterAssertion == false)
			throw new BadFilterException("FilterAssertion created by another context provider implementation cannot be added");
		IFilter filter = new Filter(idasContext_);
		filter.setAssertion(filterAssertion);
		addFilter(filter);
	}

	/**
	 * @return
	 * @throws IdASException
	 */
	public ArrayList getFilters() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::getFilters");
		return filters_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#createAttributeAssertion()
	 */
	public IFilterAttributeAssertion createAttributeAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::IFilterAttributeAssertion");
		return new FilterAttributeAssertion(idasContext_);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#createNodeIDAssertion()
	 */
	public IFilterEntityIDAssertion createNodeIDAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::createNodeIDAssertion");
		return new FilterEntityIDAssertion(idasContext_);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IFilter#createTypeAssertion()
	 */
	public IFilterEntityTypeAssertion createTypeAssertion() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::createTypeAssertion");

		// / Uncomment
		// return new FilterTypeAssertion(idasContext_);
		return null;
	}

	public ICondition getCondition() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::getCondition");
		if (condition_ == null)
			throw new IdASException("Can't return condition. Filter should be initialized before.");
		return condition_;
	}

	public void init(FilterContext filterContext) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::init");
		if (assertion_ == null) {
			if (filters_ == null || filters_.size() == 0)
				throw new IdASException("Filter doesn't contain any filter/assertion");
			int size = filters_.size();
			CompoundCondition condition = new CompoundCondition(operator_);
			for (int i = 0; i < size; i++) {
				Filter filter = (Filter) filters_.get(i);
				filter.init(filterContext);
				condition.addCondition(filter.getCondition());
				condition_ = condition;
			}
		} else {
			FilterAssertion assertion = (FilterAssertion) assertion_;
			assertion.init(filterContext);
			condition_ = assertion.getCondition();
		}
	}

	/**
	 * This method is used when Filter was added to FilterAttributeAssertion as
	 * metadata filter
	 * 
	 * @param root
	 * @param filterContext
	 * @throws IdASException
	 */
	public void initAsAttributeMetadataFilter(GraphPattern root, FilterContext filterContext) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.filter.Filter::initAsAttributeMetadataFilter");
		if (assertion_ == null) {
			if (filters_ == null || filters_.size() == 0)
				throw new IdASException("Filter doesn't contain any filters/assertion");
			int size = filters_.size();
			for (int i = 0; i < size; i++) {
				CompoundCondition condition = new CompoundCondition(operator_);
				Filter filter = (Filter) filters_.get(i);
				filter.initAsAttributeMetadataFilter(root, filterContext);
				condition.addCondition(filter.getCondition());
				condition_ = condition;
			}
		}
	}

}
