/*******************************************************************************
 * Copyright (c) 2006 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.IContextConfiguration;
import org.eclipse.higgins.idas.cp.jena2.IFactoryConfiguration;
import org.eclipse.higgins.idas.cp.jena2.util.ConfigUtil;
import org.eclipse.higgins.idas.model.impl.ContextModel;
import org.eclipse.higgins.idas.model.impl.HigginsVocabulary;

import com.hp.hpl.jena.assembler.assemblers.DocumentManagerAssembler;
import com.hp.hpl.jena.ontology.OntDocumentManager;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntModelSpec;
import com.hp.hpl.jena.rdf.model.ModelFactory;

public abstract class AbstractContextFactory implements IContextFactory {
	private static Log log = LogFactory.getLog(AbstractContextFactory.class);
	protected String id = null;

	protected String name = null;

	protected String policy = null;

	protected File configFile = null;

	protected IFactoryConfiguration config = null;

	protected String cacheLocation = null;

	protected OntDocumentManager documentManager = null;
	// The cache of context models
	private static HashMap models = new HashMap();

	protected AbstractContextFactory(String id, String name) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::AbstractContextFactory");
		setID(id);
		setName(name);
		/*
		 * String cfgDir = System.getProperty("user.home") + File.separator +
		 * ".higgins" + File.separator + ".idas"; File configFile = new
		 * File(cfgDir + File.separator + id + ".ini"); initialize(configFile);
		 */
	}

	public String getID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getID");
		return id;
	}

	protected void setID(String id) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setID");
		this.id = id;
	}

	public String getName() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getName");
		if (config != null) {
			return name;
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	protected void setName(String name) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setName");
		this.name = name;
	}

	public void initialize(File configFile) throws IdASException {
		// log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::initialize");
		// System.out.println(this.getClass().getName() + " Loading
		// configuration from:
		// \"" + configFile + "\"");
		// this.configFile = configFile;
		// config = new FactoryConfiguration(getID());
		// config.load(configFile);
		// cacheLocation = config.getCachePath();
		// documentManager = OntDocumentManager.getInstance();
		// documentManager.clearCache();
		// documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/higgins",
		// ConfigUtil.getCacheFileName(
		// "http://www.eclipse.org/higgins/ontologies/2006/higgins",
		// cacheLocation));
		// documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/higgins.owl",
		// ConfigUtil.getCacheFileName(
		// "http://www.eclipse.org/higgins/ontologies/2006/higgins.owl",
		// cacheLocation));
		// documentManager.addAltEntry("http://www.eclipse.org/higgins/ontologies/2006/display-data",
		// ConfigUtil.getCacheFileName(
		// "http://www.eclipse.org/higgins/ontologies/2006/display-data",
		// cacheLocation));
	}

	public void destroyContext(URI contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::destroyContext");
		if (config != null) {
			if (config.getContextConfig(contextID) != null) {
				config.removeContextConfig(contextID);
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public List getConfigPropNames() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getConfigPropNames");
		if (config != null) {
			List list = new ArrayList();
			for (Enumeration e = config.getDefaultProperties().propertyNames(); e.hasMoreElements();) {
				String s = (String) e.nextElement();
				list.add(s);
			}
			return list;
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public Properties getDefaultConfig() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getDefaultConfig");
		if (config != null) {
			return config.getDefaultProperties();
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public boolean canCreate(URI contextID, Properties configData) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::canCreate(URI, Properties)");
		if (config != null) {
			if (config.getContextConfig(contextID) != null) {
				throw new IdASException("Context " + contextID + " is already configured with different configuration data.");
			}

			return isValidProperties(configData);
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
	}

	public boolean canCreate(URI contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::canCreate(URI)");
		if (config != null) {
			if (config.getContextConfig(contextID) != null) {
				return true;
			} else {
				// TODO check whether it possible to load configuration using
				// contextID
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
		return false;
	}

	protected abstract IContext createContext(IContextConfiguration c) throws IdASException;

	public IContext createContext(URI contextID, Properties configData) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::createContext(URI, Properties)");
		IContext ctx = null;
		if (config != null) {
			IContextConfiguration cfg = config.getContextConfig(contextID);
			if (cfg != null) {
				throw new IdASException("Context " + contextID + " is already configured with different configuration data.");
			}
			if (!isValidProperties(configData)) {
				throw new IdASException("Context " + contextID + " couldn't be created - invalid configuration data.");
			}
			cfg = config.createContextConfig(contextID, configData);
			ctx = createContext(cfg);
			if (ctx != null) {
				config.setContextConfig(cfg);
				config.save(configFile);
			}
		} else {
			throw new IdASException("Factory hasn't been initialized yet.");
		}
		return ctx;
	}

	public IContext createContext(URI contextID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::createContext(URI)");
		IContext ctx = null;
		if (config != null) {
			IContextConfiguration cfg = config.getContextConfig(contextID);
			if (cfg == null) {
				// TODO try to load configuration data using contextID
			}
			if (cfg == null) {
				throw new IdASException("Context " + contextID + "couldn't be created - can't find configuration data.");
			}
			ctx = createContext(cfg);
		} else {
			throw new IdASException("Factory hasn't been initialized properly.");
		}
		return ctx;
	}

	public Iterator getContexts(String filter) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getContexts");
		if (config != null) {
			return config.getContextIDs().iterator();
		} else {
			throw new IdASException("Factory hasn't been initialized properly.");
		}
	}

	public String getPolicy() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::getPolicy");
		return policy;
	}

	public void setPolicy(String policy) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::setPolicy");
		this.policy = policy;
	}

	protected boolean isValidProperties(Properties props) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.AbstractContextFactory::isValidProperties");
		boolean res = false;
		if (config != null) {
			res = true;
			Properties defaults = config.getDefaultProperties();
			Iterator itr = props.keySet().iterator();
			while (itr.hasNext()) {
				if (!defaults.containsKey(itr.next())) {
					res = false;
					break;
				}
			}
		}
		return res;
	}

	/**
	 * @param schemaResource
	 *            The name of java resource or file
	 * @param schemaURI
	 *            URI of the schema, wchich identifies the model
	 * @return
	 */
	public synchronized static ContextModel getContextModel(String schemaResource, String schemaURI) throws IdASException {
		if (models.containsKey(schemaURI))
			return (ContextModel) models.get(schemaURI);
		else {
			ContextModel cm = createContextModel(schemaResource, schemaURI);
			models.put(schemaURI, cm);
			return cm;
		}
	}

	/**
	 * Creates the context model in the following order:
	 * 
	 * 1. Tries to load <code>schemaResource</code> from java resource; 2.
	 * Tries to load <code>schemaResource</code> from file 3. Tries to load
	 * <code>schemaResource</code> from HTTP 4. Tries to load
	 * <code>schemaURI</code> from HTTP
	 * 
	 * @param schemaResource
	 * @param schemaURI
	 * @return
	 */
	private synchronized static ContextModel createContextModel(String schemaResource, String schemaURI) throws IdASException {
		InputStream is = null;
		OntModel model = null;
		initHiggisBaseModel();
		try {
			try {
				is = Thread.currentThread().getContextClassLoader().getResourceAsStream(schemaResource);
				if (is == null) {
					File fileSchema = new File(schemaResource);
					if (fileSchema.exists() && fileSchema.isFile() == true)
						is = new FileInputStream(fileSchema);
				}
				if (is == null) {
					URL schemaURL = new URL(schemaResource);
					is = schemaURL.openStream();
				}
				if (is == null) {
					URL schemaURL = new URL(schemaURI);
					is = schemaURL.openStream();
				}
				if (is == null)
					throw new IdASException("Can not get schema input stream");
				model = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
				model.read(is, schemaURI);
				return new ContextModel(model, new URI(schemaURI));
			} catch (IdASException e) {
				throw e;
			} catch (Exception e) {
				throw new IdASException(e);
			}
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException e) {
			}
		}
	}

	private synchronized static void initHiggisBaseModel() throws IdASException {
		OntDocumentManager dm = OntDocumentManager.getInstance();
		if (dm.getModel(HigginsVocabulary.HIGGINS_BASE) == null) {
			InputStream is = null;
			try {
				is = Thread.currentThread().getContextClassLoader().getResourceAsStream(HigginsVocabulary.HIGGINS_BASE_RESOURCE_NAME);
				if (is != null) {
					OntModel modelHiggins = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM);
					modelHiggins.read(is, HigginsVocabulary.HIGGINS_BASE);
					if (!modelHiggins.isEmpty()) {
						dm.addModel(HigginsVocabulary.HIGGINS_BASE, modelHiggins);
					}
				}
			} catch (Exception e) {
				log.warn(e, e);
			} finally {
				if (is != null) {
					try {
						is.close();
					} catch (IOException e) {
					}
				}
			}
		}
	}

}
