/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.cp.jena2.IJenaContext;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.model.impl.AttributeModel;
import org.eclipse.higgins.idas.model.impl.HigginsVocabulary;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IEntityModel;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;

public class Entity implements IEntity {
	private Log log = LogFactory.getLog(Entity.class);

	protected Context context_ = null;

	protected IEntityModel model_ = null;

	protected Individual subject_ = null;

	protected OntClass ontClass_ = null;

	protected URI type_ = null;

	protected String userToken_ = null;

	/**
	 * Constructs Entity for already stored in ontology Individual
	 * 
	 * @param context
	 * @param subject
	 * @throws IdASException
	 */
	public Entity(Context context, Individual subject, IEntityModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::Entity");
		if (context == null)
			throw new IdASException("Couldn't create Entity with null IContext");
		if (subject == null)
			throw new IdASException("Couldn't create Entity for null subject individual");
		context_ = context;
		subject_ = subject;
		ontClass_ = context.getOntClass(subject.getRDFType().getURI());
		model_ = model;
		userToken_ = ModelUtils.getUserToken(context_, subject_);
	}

	public IContext getContext() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getContext");
		return context_;
	}

	public IJenaContext getJenaContext() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getJenaContext");
		return context_;
	}

	public Individual getSubjectIndividual() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getSubjectIndividual");
		return subject_;
	}

	public IEntityModel getModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getModel");
		return model_;
	}

	public String getEntityID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getEntityID");
		IAttribute sAttr = getAttribute(HigginsVocabulary.entityIdURI);
		if (sAttr == null)
			throw new IdASException("Attribute" + HigginsVocabulary.entityIdURI + " was not found");
		IAttributeValue val = null;
		Iterator itr = sAttr.getValues();
		if (itr != null && itr.hasNext())
			val = (IAttributeValue)itr.next();
		if (val == null)
			throw new IdASException("Value of attribute " + HigginsVocabulary.entityIdURI + " is null");
		if (val.isSimple() == false)
			throw new IdASException("Value of attribute " + HigginsVocabulary.entityIdURI + " is not SimpleValue");
		Object data = ((ISimpleAttrValue) val).getData();
		if (data == null)
			throw new IdASException("Value of attribute " + HigginsVocabulary.entityIdURI + " contains null data");
		return data.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IDigitalSubject#getEntityType()
	 */
	public URI getEntityType() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getEntityType");
		if (type_ == null)
			type_ = URI.create(ontClass_.getURI());
		return type_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.IDigitalSubject#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::remove");
		context_.unregisterChangedSubject(this);
		ModelUtils.removeIndividual(context_, subject_);
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addAttribute");
		return getAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::addAttribute");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		IAttribute newAttr = getAttribute(copyFrom.getAttrID());
		context_.registerChangedSubject(this);
		Iterator itrVal = copyFrom.getValues();
		while (itrVal.hasNext()) {
			IAttributeValue value = (IAttributeValue) itrVal.next();
			newAttr.addValue(value);
		}
		return newAttr;
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttribute");
		if (attrID == null)
			throw new IdASException("Parameter \"attrID\" is null.");
		String uri = attrID.toString();
		IAttributeModel am = model_.getAttributeModel(attrID);
		if (am == null)
			throw new IdASException("Unsupported attribute " + uri + "for entity with type " + getEntityType().toString());
		OntProperty op = context_.getOntProperty(uri);
		int maxCardinality = ModelUtils.getMaxCardinality(context_, ontClass_, op);
		if (maxCardinality == 1)
			return new SingleValuedAttribute(context_, this, op, am);
		else
			return new Attribute(context_, this, op, am);
	}

	public Iterator getAttributes() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getAttributes");
		List attrs = new ArrayList();
		Iterator itr = model_.getAttributeModels();
		while(itr.hasNext()) {
			AttributeModel attrModel = (AttributeModel)itr.next();
			URI attrType = attrModel.getType();
			if (attrType == null)
				throw new IdASException("Can not get the type of attribute model for Entity with type " + ontClass_.getURI());
			OntProperty property = context_.getModel().getOntProperty(attrType.toString());
			if (property == null)
				throw new IdASException("Can not get object property with type " + attrType);
			int maxCardinality = ModelUtils.getMaxCardinality(context_, ontClass_, property);
			if (maxCardinality == 1)
				attrs.add(new SingleValuedAttribute(context_, this, property, attrModel));
			else
				attrs.add(new Attribute(context_, this, property, attrModel));
		}
		return attrs.iterator();
	}

	// TODO need we to add this method to interface?
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getSingleValuedAttribute");
		String idURI = attrID.toString();
		IAttribute attr = getAttribute(attrID);
		if (attr == null)
			return null;
		if (attr instanceof ISingleValuedAttribute == false)
			throw new IdASException("Attribute " + idURI + " is not an instance of ISingleValuedAttribute");
		return (ISingleValuedAttribute) attr;
	}

	public String getUserToken() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::getUserToken");
		return userToken_;
	}

	public boolean equals(Object obj) {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Entity::equals");
		if (obj == null)
			return false;
		if (obj instanceof Entity == false)
			return false;
		Entity ds = (Entity)obj;
		return subject_.equals(ds.getSubjectIndividual());
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		
	}

}
