/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;
import org.eclipse.higgins.idas.api.model.IDisplayData;
import org.eclipse.higgins.idas.api.model.IModel;

import com.hp.hpl.jena.ontology.CardinalityRestriction;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.MaxCardinalityRestriction;
import com.hp.hpl.jena.ontology.MinCardinalityRestriction;
import com.hp.hpl.jena.ontology.ObjectProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class AttributeModel implements IAttributeModel {
	private Log log_ = LogFactory.getLog(AttributeModel.class);

	private URI type_ = null;

	private boolean isDirect_ = false;

	private boolean isSimple_ = false;

	private IAttributeValueModel value_ = null;

	private ContextModel ctx_ = null;

	private OntProperty attr_ = null;

	private IModel parent_ = null;

	private OntClass owner_ = null;

	private String shortName_ = null;

	private int maxCardinality_ = 0;

	private int minCardinality_ = 0;

	public AttributeModel(ContextModel ctx, DatatypeProperty dp, EntityModel parent) throws IdASException {
		ctx_ = ctx;
		attr_ = dp;
		parent_ = parent;
		owner_ = parent.getOntClass();
		isDirect_ = true;
		isSimple_ = true;
		initialize();
	}

	public AttributeModel(ContextModel ctx, DatatypeProperty dp, AttributeComplexValueModel parent) throws IdASException {
		ctx_ = ctx;
		attr_ = dp;
		parent_ = parent;
		owner_ = parent.getOntClass();
		isDirect_ = false;
		isSimple_ = true;
		initialize();
	}

	public AttributeModel(ContextModel ctx, ObjectProperty op, EntityModel parent) throws IdASException {
		ctx_ = ctx;
		attr_ = op;
		parent_ = parent;
		owner_ = parent.getOntClass();
		isDirect_ = true;
		isSimple_ = false;
		initialize();
	}

	public AttributeModel(ContextModel ctx, ObjectProperty op, AttributeComplexValueModel parent) throws IdASException {
		ctx_ = ctx;
		attr_ = op;
		parent_ = parent;
		owner_ = parent.getOntClass();
		isDirect_ = false;
		isSimple_ = false;
		initialize();
	}

	private void initialize() throws IdASException {
		if (AuthConstants.PPID_PROPERTY.equals(attr_.getURI())) {
			shortName_ = AuthConstants.PPID_HIBERNATE_NAME;
			if (isDirect_) {
				AttributeModel oldAM = ctx_.getPPIDAttribute();
				if (oldAM != null) {
					EntityModel oldOwner = (EntityModel) oldAM.getOwner();
					OntClass oldEntityClass = oldOwner.getOntClass();
					if (ModelUtils.isClassRelative(owner_, oldEntityClass))
						ctx_.setPPIDAttribute(this);
				}
				ctx_.setPPIDAttribute(this);
			}
		} else
			shortName_ = ModelUtils.getModelShortName(ctx_.getOntModel(), attr_);
		initType();
		initValue();
		initCardinality();
	}

	private void initType() throws IdASException {
		try {
			type_ = new URI(attr_.getURI());
		} catch (URISyntaxException e) {
			log_.error(e);
			throw new IdASException(e);
		}
	}

	private void initValue() throws IdASException {
		if (isSimple_) {
			value_ = new SimpleValueModel(ctx_, this, attr_.asDatatypeProperty());
		} else {
			value_ = new AttributeComplexValueModel(ctx_, this, attr_.asObjectProperty());
		}
	}

	public URI getType() {
		return type_;
	}

	public boolean isDirect() {
		return isDirect_;
	}

	public int getMaxCardinality() {
		return maxCardinality_;
	}

	public int getMinCardinality() {
		return minCardinality_;
	}

	private void initCardinality() throws IdASException {
		for (ExtendedIterator itr = owner_.listSuperClasses(true); itr.hasNext();) {
			OntClass c = (OntClass) itr.next();
			if (c.isRestriction()) {
				Restriction r = c.asRestriction();
				if (r.onProperty(attr_)) {
					if (r.isCardinalityRestriction()) {
						CardinalityRestriction cr = r.asCardinalityRestriction();
						minCardinality_ = cr.getCardinality();
						maxCardinality_ = cr.getCardinality();
					} else if (r.isMaxCardinalityRestriction()) {
						MaxCardinalityRestriction cr = r.asMaxCardinalityRestriction();
						maxCardinality_ = cr.getMaxCardinality();
					} else if (r.isMinCardinalityRestriction()) {
						MinCardinalityRestriction cr = r.asMinCardinalityRestriction();
						minCardinality_ = cr.getMinCardinality();
					}
				}
			}
		}
	}

	public Iterator getMetadataModels() {
		return null;
	}

	public IAttributeValueModel getValueModel() {
		return value_;
	}

	public String getId() {
		return shortName_;
	}

	public boolean isSimple() {
		return isSimple_;
	}

	public IModel getOwner() {
		return parent_;
	}

	public IDisplayData getDisplayData() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

}
