/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.cp.jena2.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;
import org.eclipse.higgins.idas.cp.jena2.util.ModelUtils;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.ontology.Individual;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.NodeIterator;
import com.hp.hpl.jena.rdf.model.RDFNode;

/**
 * @author sergey
 * 
 */
/**
 * @author sergey
 * 
 */
public class Attribute implements IAttribute {
	private Log log = LogFactory.getLog(Attribute.class);

	protected OntProperty property_ = null;

	protected Context context_ = null;

	protected IAttributeModel attModel_ = null;

	protected OntClass ownerClass_ = null;

	protected OntClass complexValueClass_ = null;

	protected boolean isSimple_ = false;

	// the subject which owns this attribute (directly or through any level of
	// complex value)
	protected Entity subject_ = null;

	protected Individual owner_ = null;

	protected Attribute(Context context, Entity subject, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::Attribute(..., Entity, ...)");
		if (subject == null)
			throw new IdASException("Parameter \"subject\" is null.");
		subject_ = subject;
		init(context, subject.getSubjectIndividual(), property, model);
	}

	protected Attribute(Context context, ComplexValue valueOwner, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::Attribute(..., ComplexValue, ...)");
		if (valueOwner == null)
			throw new IdASException("Parameter \"valueOwner\" is null.");
		subject_ = valueOwner.getSubjectOwner();
		init(context, valueOwner.getValueIndividual(), property, model);
	}

	private void init(Context context, Individual owner, OntProperty property, IAttributeModel model) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::init");
		if (context == null)
			throw new IdASException("Parameter \"context\" is null.");
		if (owner == null)
			throw new IdASException("Parameter \"owner\" is null.");
		if (property == null)
			throw new IdASException("Parameter \"property\" is null.");
		context_ = context;
		owner_ = owner;
		property_ = property;
		ownerClass_ = context_.getOntClass(owner_.getRDFType().getURI());
		attModel_ = model;
		isSimple_ = attModel_.getValueModel().isSimple();
	}

	protected OntClass getComplexValueClass() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getComplexValueClass");
		if (complexValueClass_ == null) {
			IAttributeValueModel valueModel = attModel_.getValueModel();
			if (valueModel.isSimple())
				throw new IdASModelException("Attribute " + attModel_.getType().toString() + " is not complex.");
			String cvClassURI = valueModel.getType().toString();
			complexValueClass_ = context_.getOntClass(cvClassURI);
			if (complexValueClass_ == null)
				throw new IdASException("Can not find class with URI " + cvClassURI);
		}
		return complexValueClass_;
	}

	/**
	 * @return <code>Individual</code> of <code>Entity</code> or
	 *         <code>ComplexValue</code> which owns this attribure
	 */
	public Individual getOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getOwner");
		return owner_;
	}

	/**
	 * @return Ontology class of Individual of Entity or ComplexValue which
	 *         immediately owns this attribure
	 */
	public OntClass getOwnerClass() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getOwnerClass");
		return ownerClass_;
	}

	/**
	 * @return Entity which owns this attribute (directly or through any level
	 *         of complex value)
	 */
	public Entity getSubjectOwner() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getSubjectOwner");
		return subject_;
	}

	/**
	 * @return ObjectProperty of this attribute
	 */
	public Entity getAttributeProperty() {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getSubjectOwner");
		return subject_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#addComplexValue(java.net.URI)
	 */
	public IComplexAttrValue addComplexValue(URI type) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addComplexValue");
		if (isSimple_ == false) {
			OntClass cvClass = getComplexValueClass();
			context_.registerChangedSubject(subject_);
			Individual newValue = context_.getModel().createIndividual(cvClass);
			owner_.addProperty(property_, newValue);
			return new ComplexValue(context_, this, newValue, property_);
		} else
			throw new IdASException("Can not create complex value for simple attribute with type " + getAttrID().toString());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#addSimpleValue(java.net.URI,
	 *      java.lang.Object)
	 */
	public ISimpleAttrValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addSimpleValue");
		if (isSimple_ == true) {
			context_.registerChangedSubject(subject_);
			SimpleValue sv = new SimpleValue(context_, this, owner_, property_.asDatatypeProperty(), null);
			sv.setData(data);
			return sv;
		} else
			throw new IdASException("Can not create simple value for complex attribute with type " + getAttrID().toString());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(java.net.URI)
	 */
	public IAttributeValue addValue(URI type) throws IdASException, InvalidTypeException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addValue(URI)");
		if (isSimple_)
			return addSimpleValue(type, null);
		else
			return addComplexValue(type);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#addValue(org.eclipse.higgins.idas.api.IAttributeValue)
	 */
	public IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::addValue");
		if (copyFrom == null)
			throw new IdASException("Parameter \"copyFrom\" is null.");
		URI newType = copyFrom.getDataType();
		if (newType == null)
			throw new IdASException("Type of passed value \"copyFrom\" is null.");
		if (newType.equals(this.getAttrID()) == false)
			throw new IdASException("Can not create value with type " + newType.toString() + " for attribute with type " + getAttrID().toString());
		if (isSimple_) {
			if (copyFrom instanceof ISimpleAttrValue) {
				ISimpleAttrValue from = (ISimpleAttrValue) copyFrom;
				ISimpleAttrValue newVal = addSimpleValue(newType, from.getData());
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of ISimpleValue");
		} else {
			if (copyFrom instanceof IComplexAttrValue) {
				IComplexAttrValue from = (IComplexAttrValue) copyFrom;
				IComplexAttrValue newVal = addComplexValue(newType);
				Iterator itrAttr = from.getAttributes();
				if (itrAttr != null) {
					while (itrAttr.hasNext()) {
						IAttribute attr = (IAttribute) itrAttr.next();
						newVal.addAttribute(attr);
					}
				}
				return newVal;
			} else
				throw new IdASException("Passed value \"copyFrom\" with type " + newType.toString() + " is not an instance of IComplexValue");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#getAttrID()
	 */
	public URI getAttrID() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getAttrID");
		return attModel_.getType();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#getModel()
	 */
	public IAttributeModel getModel() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getModel");
		return attModel_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#getValues()
	 */
	public Iterator getValues() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::getValues");
		ArrayList list = new ArrayList();
		NodeIterator itr = owner_.listPropertyValues(property_);
		if (isSimple_) {
			while (itr.hasNext()) {
				RDFNode node = itr.nextNode();
				if (node != null) {
					if (node.isLiteral()) {
						Literal lt = (Literal)node.as(Literal.class);
						SimpleValue sv = new SimpleValue(context_, this, owner_, property_.asDatatypeProperty(), lt);
						list.add(sv);
					}
				}
			}
		}
		else {
			while (itr.hasNext()) {
				RDFNode node = itr.nextNode();
				if (node != null) {
					if (node.isLiteral() == false) {
						Individual ind = (Individual) node.as(Individual.class);
						OntClass cls = context_.getOntClass(ind.getRDFType().getURI());
						ComplexValue cv = new ComplexValue(context_, this, ind, property_);
						list.add(cv);
					}
				}
			}
		}
		return list.iterator();
	}

	public Iterator getValues(IExtension[] extensions) throws IdASException {
		if (extensions != null) {
			for (int index = 0; index < extensions.length; index++) {
				if (extensions[index].failIfUnsupported() == true) {
					throw new UnhandledExtensionException(extensions[index].getClass().getName() + " not supported");
				}
			}
		}

		return getValues();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#isSingleValued()
	 */
	public boolean isSingleValued() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::isSingleValued");
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IAttribute#remove()
	 */
	public void remove() throws IdASException {
		log.trace("org.eclipse.higgins.idas.cp.jena2.impl.Attribute::remove");
		ArrayList list = new ArrayList();
		NodeIterator itr = owner_.listPropertyValues(property_);
		while (itr.hasNext()) {
			RDFNode node = itr.nextNode();
			list.add(node);
		}
		for (int i = 0; i < list.size(); i++) {
			RDFNode node = (RDFNode) list.get(i);
			owner_.removeProperty(property_, node);
			if (node.isLiteral() == false) {
				Individual ind = (Individual) node.as(Individual.class);
				ModelUtils.removeIndividual(context_, ind);
			}
		}
	}

	/**
	 * The methods below were added by Jim to replace IMetadata
	 */
	public boolean equals(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		// TODO Auto-generated method stub
		return null;
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public Iterator getAttributes() throws IdASException {
		// TODO Auto-generated method stub
		return null;
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		// TODO Auto-generated method stub
		return null;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		// TODO Auto-generated method stub

	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		// TODO Auto-generated method stub

	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub

	}

}
