/*******************************************************************************
 * Copyright (c) 2006 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.idas.cp.jena2.util;

import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.dv.util.Base64;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.cp.jena2.util.XMLDateConvertor;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.datatypes.TypeMapper;
import com.hp.hpl.jena.datatypes.xsd.XSDDatatype;
import com.hp.hpl.jena.rdf.model.Literal;

public class XMLValueHelper {
	private static Log log = LogFactory.getLog(XMLValueHelper.class);

	protected String type = null;

	protected XSDDatatype xsdType = null;

	protected static Hashtable typesMap = new Hashtable();

	public static synchronized XMLValueHelper getInstance(String xmlType)
			throws IdASException {
		if (xmlType == null)
			return null;
		if (typesMap.contains(xmlType))
			return (XMLValueHelper) typesMap.get(xmlType);
		else {
			XMLValueHelper valueModel = new XMLValueHelper(xmlType);
			typesMap.put(xmlType, valueModel);
			return valueModel;
		}
	}

	protected XMLValueHelper(String xmlType) throws IdASException {
		initType(xmlType);
	}

	protected void initType(String typeUri) throws IdASException {
		try {
			xsdType = findXSDDatatype(typeUri);
			if (xsdType == null) {
				throw new IdASModelException("Unsupported datatype " + typeUri);
			}
			type = typeUri;
		} catch (IdASException e) {
			throw e;
		} catch (Exception e) {
			throw new IdASModelException("Invalid datatype " + typeUri, e);
		}
	}

	public static XSDDatatype findXSDDatatype(String typeUri) {
		if (typeUri == null)
			return null;
		RDFDatatype rdfType = TypeMapper.getInstance().getTypeByName(typeUri);
		if (rdfType == null) {
			log.warn("Couldn't find RDFDatatype by URI = " + typeUri);
			return null;
		}
		if (rdfType instanceof XSDDatatype == false) {
			log.warn("URI " + typeUri + " is not xsd datatype");
			return null;
		}
		return (XSDDatatype) rdfType;
	}

	public static boolean isSupported(String typeUri) {
		return (findXSDDatatype(typeUri) != null);
	}

	public String getType() {
		return type;
	}

	public boolean isValidValue(Object valueForm) {
		return xsdType.isValidValue(valueForm);
	}

	public boolean isValid(String lexicalForm) {
		return xsdType.isValid(lexicalForm);
	}

	public Object toValueForm(String lexicalForm) throws IdASModelException {
		if (lexicalForm == null)
			return "";
		try {
			return xsdType.parse(lexicalForm);
		} catch (Exception e) {
			throw new IdASModelException(e);
		}
	}

	public Object toValueForm(Literal literal) throws IdASModelException {
		if (literal != null) {
			Object obj = parse(literal);
			return obj;
		} else
			return null;
	}

	public String toLexicalForm(Object value) throws IdASModelException {
		if (value == null)
			return null;
		return unparseObj(value);
	}

	private String unparseObj(Object value) throws IdASModelException {
		try {
			if (value instanceof Date) {
				return XMLDateConvertor.format((Date) value);
			} else if (value instanceof ByteBuffer) {
				ByteBuffer bb = (ByteBuffer) value;
				byte[] data = bb.array();
				return Base64.encode(data);
			}
			return xsdType.unparse(value);
		} catch (Exception e) {
			throw new IdASModelException(
					"Can not get lexical form for xml type " + type.toString()
							+ ", object value " + value);
		}
	}

	private Object parse(Literal lt) throws IdASModelException {
		try {
			if (type.equals(ISimpleAttrValue.DATE_ATTR_VALUE_TYPE_URI_STR) ||
				type.equals(ISimpleAttrValue.DATETIME_ATTR_VALUE_TYPE_URI_STR)||
				type.equals(ISimpleAttrValue.TIME_ATTR_VALUE_TYPE_URI_STR))
					return XMLDateConvertor.parse(lt.getString());
			if (type.equals(ISimpleAttrValue.ANYURI_ATTR_VALUE_TYPE_URI_STR)) {
				try {
					return new URI(lt.getString());
				} catch (Exception e) {
					log.error(e);
					throw new IdASModelException(e);
				}
			}
			if (ISimpleAttrValue.BASE64BINARY_ATTR_VALUE_TYPE_URI_STR
					.equals(type)
					|| ISimpleAttrValue.BYTE_ATTR_VALUE_TYPE_URI_STR
							.equals(type)) {
				byte[] data = Base64.decode(lt.getString());
				ByteBuffer bb = ByteBuffer.allocate(data.length);
				bb.put(data);
				return bb;
			}
			return lt.getValue();
		} catch (Exception e) {
			throw new IdASModelException(e);
		}
	}

}
