/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.idas.model.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IdASModelException;

import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.ontology.DataRange;
import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;

public class XMLValueModel {

	private Log log = LogFactory.getLog(XMLValueModel.class);

	private URI type_ = null;

	private boolean isOneOf_ = false;

	private List oneOf_ = null;

	private String pattern_ = null;

	private int length_ = -1;

	private int minLength_ = -1;

	private int maxLength_ = -1;

	private int totalDigits_ = -1;

	private int fractionDigits_ = -1;

	private DatatypeProperty attrProp_ = null;

	private SimpleValueModel parent_ = null;

	public XMLValueModel(DatatypeProperty prop, SimpleValueModel parent) throws IdASException {
		parent_ = parent;
		attrProp_ = prop;
		initialize();
	}

	private void initialize() throws IdASException {
		initRange();
		if (type_ == null)
			throw new IdASException("Can not parse xml type of simple attribute " + parent_.getAttribute().getType());
	}

	private void initRange() throws IdASException {
		initRange(attrProp_);
	}

	private void initRange(OntProperty prop) throws IdASException {
		OntResource r = prop.getRange();
		if (r != null) {
			if (r.isDataRange()) {
				initAsOWL1_0(r);
			} else {
				Resource res = r.getRDFType();
				if (res != null)
					initAsOWL1_1(r);
				else
					initType(r.getURI());
			}
		} else {
			Iterator itr = prop.listSubProperties();
			while (itr.hasNext()) {
				OntProperty p = (OntProperty) itr.next();
				initRange(p);
			}
		}
	}

	private void initAsOWL1_0(OntResource r) throws IdASException {
		oneOf_ = new ArrayList();
		DataRange dr = r.asDataRange();
		isOneOf_ = true;
		for (Iterator itr = dr.listOneOf(); itr.hasNext();) {
			Literal l = (Literal) itr.next();
			RDFDatatype t = l.getDatatype();
			initType(t.getURI());
			oneOf_.add(l.getValue());
		}
	}

	private void initAsOWL1_1(OntResource r) throws IdASException {
		StmtIterator itr = r.listProperties();
		while (itr.hasNext()) {
			Statement st = itr.nextStatement();
			RDFNode node = st.getObject();
			Literal val = node.isLiteral() ? (Literal) node.as(Literal.class) : null;
			String prop = st.getPredicate().getURI();
			if (HigginsVocabulary.OWL11_onDataRange.equals(prop)) {
				if (node.isResource()) {
					Resource res = (Resource) node.as(Resource.class);
					initType(res.getURI());
				}
			} else if (HigginsVocabulary.OWL11_pattern.equals(prop)) {
				pattern_ = val.getString();
			} else if (HigginsVocabulary.OWL11_maxLength.equals(prop)) {
				try {
					maxLength_ = val.getInt();
				} catch (Exception e) {
					log.error("Can not parse maxLength for property " + parent_.getType(), e);
				}
			} else if (HigginsVocabulary.OWL11_totalDigits.equals(prop)) {
				try {
					totalDigits_ = val.getInt();
				} catch (Exception e) {
					log.error("Can not parse totalDigits for property " + parent_.getType(), e);
				}
			} else if (HigginsVocabulary.OWL11_fractionDigits.equals(prop)) {
				try {
					fractionDigits_ = val.getInt();
				} catch (Exception e) {
					log.error("Can not parse fractionDigits for property " + parent_.getType(), e);
				}
			}
		}
	}

	private void initType(String typeUri) throws IdASException {
		log.trace("org.eclipse.higgins.idas.model.impl.XMLValueModel::initType");
		if (type_ == null) {
			try {
				type_ = new URI(typeUri);
			} catch (Exception e) {
				log.error(e);
				throw new IdASException(e);
			}
		} else if (typeUri.equals(type_) == false) {
			throw new IdASModelException("Datatype mismatch for enumerated values: Property=" + attrProp_.getURI() + " Datatype1=" + type_.toString()
					+ " Datatype2=" + typeUri);
		}
	}

	public boolean isOneOf() {
		return isOneOf_;
	}

	public List getOneOf() {
		return oneOf_;
	}

	public URI getType() {
		return type_;
	}

	public String getPattern() {
		return pattern_;
	}

	public int getLength() {
		return length_;
	}

	public int getMinLength() {
		return minLength_;
	}

	public int getMaxLength() {
		return maxLength_;
	}

	public int getTotalDigits() {
		return totalDigits_;
	}

	public int getFractionDigits() {
		return fractionDigits_;
	}

}
