/**
 * Copyright (c) 2006-2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Tom Doman
 *		Jim Sermersheim
 *		Duane Buss
 */

package org.eclipse.higgins.idas.cp.jndi;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.security.Provider;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;
import java.util.Vector;
import java.util.Map.Entry;

import javax.naming.CommunicationException;
import javax.naming.Context;
import javax.naming.NameClassPair;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.StartTlsRequest;
import javax.naming.ldap.StartTlsResponse;
import javax.net.ssl.SSLSession;
//import com.sun.jndi.ldap.ctl.ProxiedAuthorizationControl;

import openxdas.XDasEvents;
import openxdas.XDasException;
import openxdas.XDasOutcomes;
import openxdas.XDasRecord;
import openxdas.XDasSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.dom.DOMDocument;
import org.dom4j.dom.DOMElement;
import org.dom4j.dtd.InternalEntityDecl;
import org.dom4j.tree.DefaultDocumentType;
import org.eclipse.higgins.idas.api.AuthenticationException;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.spi.BasicContext;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;
import org.eclipse.higgins.idas.common.AuthNDigestMaterials;
import org.eclipse.higgins.idas.common.AuthNLeastPrivilegedUserMaterials;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.spi.AttributeNotification;
import org.eclipse.higgins.idas.spi.AttributeValueNotification;
import org.eclipse.higgins.idas.spi.BasicFilter;
import org.eclipse.higgins.idas.spi.BasicFilterAssertion;
import org.eclipse.higgins.idas.spi.BasicFilterAttributeAssertion;
import org.eclipse.higgins.idas.spi.BasicFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.spi.BasicValueBase64Binary;
import org.eclipse.higgins.idas.spi.BasicValueHexBinary;
import org.eclipse.higgins.idas.spi.BasicValueLanguage;
import org.eclipse.higgins.idas.spi.BasicValueNCName;
import org.eclipse.higgins.idas.spi.BasicValueNMTOKEN;
import org.eclipse.higgins.idas.spi.BasicValueName;
import org.eclipse.higgins.idas.spi.BasicValueNormalizedString;
import org.eclipse.higgins.idas.spi.BasicValueString;
import org.eclipse.higgins.idas.spi.BasicValueToken;
import org.eclipse.higgins.idas.spi.EntityNotification;
import org.eclipse.higgins.util.saml.SAML1Assertion;
import org.eclipse.higgins.util.saml.SAML1Subject;
import org.eclipse.higgins.util.idas.cp.jscript.JScriptCPAttributePDPs;
import org.eclipse.higgins.util.jscript.JScriptException;
import org.eclipse.higgins.util.jscript.JScriptExec;
import org.eclipse.higgins.util.jscript.JScriptExecHelper;

import org.bandit.util.jndi.RfcFilter;
import org.bandit.util.misc.CIStringKey;
import org.bandit.util.misc.NonEscapedAttrXMLWriter;

import com.novell.ldap.LDAPLocalException;

/**
 * 
 * @author tdoman@novell.com
 * @author jimse@novell.com
 * @author dbuss@novell.com
 */

public class JNDIContext extends BasicContext
{
	private Log _log = LogFactory.getLog(JNDIContext.class.getName());
	private LdapContext _ctx;
	private Map _contextSettings;	
	private Map _connectionSettings;
	private Hashtable _env;
	private HashMap _schemaAttrDomainHashMap;
	private HashMap _schemaAttrCaseHashMap;
	private HashMap _syntaxHashMap;
	private IContextId _contextID;
	private boolean _bIsOpen = false;
	private String _currAddr, _entityID, _contextURIs = "", _contextTypes = "";
	private XDasSession _xdasSession;
	private Object _identity;
	private boolean _bStartTLS = false;
	private StartTlsResponse _tls;
//	private ProxiedAuthorizationControl proxyAuthZ;

	public static final String consumerEntityIDToProvider = "consumerEntityIDToProvider";
	public static final String providerEntityIDToConsumer = "providerEntityIDToConsumer";
	public static final String consumerEntityTypeToProvider = "consumerEntityTypeToProvider";
	public static final String providerEntityTypeToConsumer = "providerEntityTypeToConsumer";
	public static final String consumerAIDToProvider = "consumerAIDToProvider";
	public static final String providerAIDToConsumer = "providerAIDToConsumer";
	public static final String consumerATypeToProvider = "consumerATypeToProvider";
	public static final String providerATypeToConsumer = "providerATypeToConsumer";
	public static final String consumerAValueToProvider = "consumerAValueToProvider";
	public static final String providerAValueToConsumer = "providerAValueToConsumer";

	private JScriptExec _consumerEntityIDToProviderExec;
	private JScriptExec _providerEntityIDToConsumerExec;
	private JScriptExec _consumerEntityTypeToProviderExec;
	private JScriptExec _providerEntityTypeToConsumerExec;
	private JScriptCPAttributePDPs _attrPDPs;
	private boolean _bSetupPDPs = true;

	private class SyntaxMap
	{
		private String _label;
		private String _urlName;
		private String _owlType;

		SyntaxMap(
			String label,
			String owlType)
		{
			_label = label;
			_urlName = label.replaceAll(" ", "");
			_owlType = owlType;
		}

		public String getLabel()
		{
			return _label;
		}

		public String getURLName()
		{
			return "syntax_" + _urlName;
		}

		public String getOWLType()
		{
			return _owlType;
		}
	};

	/**
	 */
	public JNDIContext(
		IContextId contextID) throws IdASException
	{
		int iLoop = 0;

		_contextID = contextID;
		_contextSettings = contextID.getConfiguration();
		_connectionSettings = (Map)_contextSettings.get("Connection");
		if (_connectionSettings == null)
			throw new IdASException("No JNDI conne" +
					"ction settings specified");
		_env = new Hashtable();

		URI contextURIs[] = _contextID.getUris();
		for (iLoop = 0; iLoop < contextURIs.length; ++iLoop)
			_contextURIs = _contextURIs + contextURIs[iLoop].toString() + ", ";

		String contextTypes[] = _contextID.getTypes();
		for (iLoop = 0; iLoop < contextTypes.length; ++iLoop)
			_contextTypes = _contextTypes + contextTypes[iLoop] + ", ";

		_initSyntaxMap();
		_initXDASSession();
	}

	/**
	 */
	private void _initXDASSession() throws IdASException
	{
		try
		{
			_xdasSession = new XDasSession("Higgins IdAS JNDI Context Provider", null, null,
				null, null, null);
		}
		catch (XDasException e)
		{
			throw new IdASException(e);
		}
		catch (SocketException e)
		{
			;
		}
		catch (IOException e)
		{
			;
		}
	}

	/**
	 */
	private void _initSyntaxMap()
	{
		_syntaxHashMap = new HashMap();

		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.3", new SyntaxMap(
			"Attribute Type Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.6", new SyntaxMap("Bit String",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.7", new SyntaxMap("Boolean",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.11", new SyntaxMap("Country String",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.14", new SyntaxMap(
			"Delivery Method", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.15", new SyntaxMap(
			"Directory String", "&higgins;NormalizedStringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.16", new SyntaxMap(
			"DIT Content Rule Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.17", new SyntaxMap(
			"DIT Structure Rule Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.12", new SyntaxMap("DN",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.21", new SyntaxMap("Enhanced Guide",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.22", new SyntaxMap(
			"Facsimile Telephone Number", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.23", new SyntaxMap("Fax",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.24", new SyntaxMap(
			"Generalized Time", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.25", new SyntaxMap("Guide",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.26", new SyntaxMap("IA5 String",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.27", new SyntaxMap("Integer",
			"&higgins;IntegerSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.28", new SyntaxMap("JPEG",
			"&higgins;Base64BinarySimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.54", new SyntaxMap(
			"LDAP Syntax Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.30", new SyntaxMap(
			"Matching Rule Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.31", new SyntaxMap(
			"Matching Rule Use Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.34", new SyntaxMap(
			"Name and Optional UID", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.35", new SyntaxMap(
			"Name Form Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.36", new SyntaxMap("Numeric String",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.37", new SyntaxMap(
			"Object Class Description", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.40", new SyntaxMap("Octet String",
			"&higgins;Base64BinarySimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.38", new SyntaxMap("OID",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.39", new SyntaxMap("Other Mailbox",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.41", new SyntaxMap("Postal Address",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.44", new SyntaxMap(
			"Printable String", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.58", new SyntaxMap(
			"Substring Assertion", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.50", new SyntaxMap(
			"Telephone Number", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.51", new SyntaxMap(
			"Teletex Terminal Identifier", "&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.52", new SyntaxMap("Telex Number",
			"&higgins;StringSimpleAttribute"));
		_syntaxHashMap.put("1.3.6.1.4.1.1466.115.121.1.53", new SyntaxMap("UTC Time",
			"&higgins;StringSimpleAttribute"));
	}

	/**
	 * @throws IdASException
	 */
	private void _emitXDASRecord(
		int iEventNumber,
		int iOutcome,
		String sInitiatorInfo,
		String sTargetInfo,
		String sEventInfo) throws IdASException
	{
		if (_xdasSession != null)
		{
			if (sTargetInfo == null)
				sTargetInfo = "";
			if (sEventInfo == null)
				sEventInfo = "";
			try
			{
				XDasRecord xdasRecord = _xdasSession.XDasStartRecord(iEventNumber, iOutcome,
					null, null, sEventInfo);
				xdasRecord.setInitiatorInfo(null, null, sInitiatorInfo);
				xdasRecord.setTargetInfo(null, _contextURIs, _contextTypes, null, null, sTargetInfo);
				xdasRecord.commit();
			}
			catch (XDasException e)
			{
				throw new IdASException(e);
			}
			catch (IOException e)
			{
				throw new IdASException(e);
			}
		}
	}

	/**
	 */
	private String _lookupEntity(
		IFilter lookupFilter) throws IdASException
	{
		String entityID = null;

		if (lookupFilter != null)
		{
			Iterator entities = this.getEntities(lookupFilter);
			while (entities.hasNext())
			{
				IEntity entity = (IEntity) entities.next();
				if (entityID == null)
					entityID = entity.getEntityID();
				else
					throw new IdASException("Non-unique identity from AuthN materials.");
			}
		}

		return entityID;
	}

	/**
	 */
	public String open(
		Object identity) throws IdASException
	{
		if (_bIsOpen)
			throw new ContextOpenException();

		boolean bSetupContext = _identity == null ? true : false;
		if (bSetupContext || !_identity.equals(identity))
		{
			if (_bSetupPDPs)
			{
				Map jsPolicySettings = (Map)_contextSettings.get("JSPolicyAction");
				if (jsPolicySettings != null)
				{
					_consumerEntityIDToProviderExec = (JScriptExec)jsPolicySettings.get(consumerEntityIDToProvider);
					_log.debug(consumerEntityIDToProvider + " == " + _consumerEntityIDToProviderExec);
					
					_providerEntityIDToConsumerExec = (JScriptExec)jsPolicySettings.get(providerEntityIDToConsumer);
					_log.debug(providerEntityIDToConsumer + " == " + _providerEntityIDToConsumerExec);
					
					_consumerEntityTypeToProviderExec = (JScriptExec)jsPolicySettings.get(consumerEntityTypeToProvider);
					_providerEntityTypeToConsumerExec = (JScriptExec)jsPolicySettings.get(providerEntityTypeToConsumer);

					_attrPDPs = new JScriptCPAttributePDPs(
							(JScriptExec)jsPolicySettings.get(consumerAIDToProvider),
							(JScriptExec)jsPolicySettings.get(providerAIDToConsumer),
							(JScriptExec)jsPolicySettings.get(consumerATypeToProvider),
							(JScriptExec)jsPolicySettings.get(providerATypeToConsumer),
							(JScriptExec)jsPolicySettings.get(consumerAValueToProvider),
							(JScriptExec)jsPolicySettings.get(providerAValueToConsumer),
							false);

					_bSetupPDPs = false;
				}
			}

			Map envSettings = (Map)_contextSettings.get("env");
			if (envSettings != null)
			{
				Set envSet = envSettings.entrySet();
				Iterator iter = envSet.iterator();

				while (iter.hasNext())
				{
					Entry envEntry = (Entry)iter.next();
					_env.put(envEntry.getKey(), envEntry.getValue());
				}
			}

			IFilter lookupFilter = null;
			_identity = identity;
			// Override specific environment elements.
			if (identity instanceof AuthNNamePasswordMaterials)
			{
				_env.put(Context.SECURITY_AUTHENTICATION, "simple");
				_entityID = ((AuthNNamePasswordMaterials) identity).getUsername();
				_env.put(Context.SECURITY_PRINCIPAL, consumerEntityIDToProvider(_entityID));
				_env.put(Context.SECURITY_CREDENTIALS,
					((AuthNNamePasswordMaterials) identity).getPassword());
				_log.debug("attempt open using simple auth with user: " + _entityID 
						+" mapped to: " + consumerEntityIDToProvider(_entityID));
			}
			else if (identity instanceof AuthNDigestMaterials)
			{
					BasicValueBase64Binary base64Value = new BasicValueBase64Binary(
						((AuthNDigestMaterials)identity).getSHA1Digest(), null);
					BasicFilterAttributeAssertion attrAssertion = new BasicFilterAttributeAssertion();
					attrAssertion.setID(URI.create("cardKeyHash"));
					attrAssertion.setComparator(BasicFilterAttributeAssertion.COMP_ATTR_EQ);
					attrAssertion.setAssertionValue(new BasicValueString(base64Value
						.getLexical(), null));
					lookupFilter = new BasicFilter();
					lookupFilter.setAssertion(attrAssertion);
					_log.debug("attempt open using digestmaterials: " + base64Value.getLexical());
			}
			else if (identity instanceof AuthNLeastPrivilegedUserMaterials)
			{
				_entityID = (String) _env.get(Context.SECURITY_PRINCIPAL);
				_log.debug("attempt open using least privileged user: " + _entityID);
			}
			else if (identity instanceof AuthNAnonymousMaterials)
			{
				_log.debug("attempt open using anonymous materials");
				_env.put(Context.SECURITY_AUTHENTICATION, "none");
				Object obj = _env.get(Context.SECURITY_PRINCIPAL);
				_env.remove(obj);
				obj = _env.get(Context.SECURITY_CREDENTIALS);
				_env.remove(obj);
			}
			else if (identity instanceof SAML1Assertion)
			{
				try
				{
					String saslClientPackage = (String) _env.get("org.eclipse.higgins.idas.cp.jndi.saslClientPackage"),
							saslClientProvider = (String) _env.get("org.eclipse.higgins.idas.cp.jndi.saslClientProvider");

					if ((saslClientPackage == null) || (saslClientProvider == null))
						throw new IdASException("SASL SAML Client Provider not specified");

					// Make sure the SAML/SASL security provider is loaded
					if (java.security.Security.getProvider("SAML SASL Mechanism") == null)
					{
						Class saslClientProviderClass = Class.forName(saslClientPackage + "." + saslClientProvider);
						java.security.Security.addProvider((Provider)saslClientProviderClass.newInstance());
					}

					SAML1Assertion assertion = (SAML1Assertion)identity;
					SAML1Subject subject = assertion.getSubject();
					if (subject.getNameIDFormat().indexOf("X509SubjectName") == -1)
						throw new IdASException("Unsupported name format specified in SAML assertion"); 
					_entityID = subject.getNameID();
					_env.put(Context.SECURITY_PRINCIPAL, _entityID);
					_env.put(Context.SECURITY_PROTOCOL, "ssl");
					_env.put(Context.SECURITY_AUTHENTICATION, "SAML");
					_env.put("java.naming.security.sasl.authorizationId", _entityID);
					_env.put("javax.security.sasl.client.pkgs", saslClientPackage);
					_env.put(Context.SECURITY_CREDENTIALS, assertion.toString());
				}
				catch (ClassNotFoundException e)
				{
					throw new IdASException(e);
				}
				catch (InstantiationException e)
				{
					throw new IdASException(e);
				}
				catch (IllegalAccessException e)
				{
					throw new IdASException(e);
				}
			}
			else
			{
				throw new NotImplementedException("Unsupported AuthN materials: "
					+ identity.toString());
			}

			try
			{
				if (bSetupContext == true)
				{
					_setupContext();

					String str = (String) _env.get("org.eclipse.higgins.idas.cp.jndi.startTLS"), trustStore;
					if ((str != null) && (str.compareTo("true")) == 0)
					{
						_bStartTLS = true;
						_tls = (StartTlsResponse) _ctx.extendedOperation(new StartTlsRequest());
						SSLSession sess;
						if ((trustStore = (String) _env.get("org.eclipse.higgins.idas.cp.jndi.trustStore")) != null)
						{
							_setupCustomSSLSocketFactory(trustStore, (String) _env.get("org.eclipse.higgins.idas.cp.jndi.trustStorePassword"));
							sess = _tls.negotiate((LDAPHigginsSSLSocketFactoryProducer)LDAPHigginsSSLSocketFactoryProducer.getDefault());
						}
						else
						{
							sess = _tls.negotiate();
						}
						_log.debug("Start TLS request successful: Host (" + sess.getPeerHost()
							+ ")" + ", Protocol ("
							+ sess.getProtocol() + ")");
					}

					_emitXDASRecord(XDasEvents.XDAS_AE_CREATE_SESSION,
						XDasOutcomes.XDAS_OUT_SUCCESS, _getIdentity(), null, null);
					_log.debug("Context opened: " + _contextURIs + " as: " + _getIdentity());
				}
				else
				{
					Iterator envIter = _env.entrySet().iterator();
					while (envIter.hasNext())
					{
						Map.Entry me = (Map.Entry) envIter.next();
						_ctx.addToEnvironment((String) me.getKey(), me.getValue());
					}
					_ctx.reconnect(null);
				}
			}
			catch (NamingException e)
			{
				_handleNamingException(e);
			}
			catch (IOException e)
			{
				throw new IdASException(e);
			}
			_bIsOpen = true;

			if (lookupFilter != null)
			{
				_entityID = _lookupEntity(lookupFilter);
				if (_entityID == null)
					throw new IdASException("AuthN materials produced no Entity ID: "
						+ identity.toString());
			}
		}
		else
		{
			/* Context already open for this identity. */
			_bIsOpen = true;
		}
		return _entityID;
	}

	/**
	 */
	private void _handleNamingException(
		NamingException e) throws IdASException
	{
		if (e instanceof javax.naming.CommunicationException)
		{
			_destroyContext();
		}
		else if (e instanceof javax.naming.NameAlreadyBoundException)
		{
			throw new EntityExistsException(e);
		}
		else if (e instanceof javax.naming.AuthenticationException)
		{
			_destroyContext();
			throw new AuthenticationException(e);
		}
		else if (e instanceof javax.naming.NameNotFoundException)
		{
			throw new NoSuchEntityException(e);
		}
		throw new IdASException(e);
	}

	/**
	 * 
	 * @throws IdASException
	 */
	private void _setupCustomSSLSocketFactory(
		String trustStore,
		String trustStorePassword) throws IdASException
	{
		LDAPHigginsSSLSocketFactoryProducer socketFactory = (LDAPHigginsSSLSocketFactoryProducer)LDAPHigginsSSLSocketFactoryProducer.getDefault();		
		socketFactory.setKeyStore(new File(trustStore), trustStorePassword);
	}

	/**
	 */
	private void _setupContext() throws IdASException
	{
		List addressList = (List)_connectionSettings.get("AddressList");
		
		if ((addressList == null) || (addressList.size() == 0))
			throw new IdASException("No JNDI service addresses configured.");

		int iLoop;
		for (iLoop = 0; iLoop < addressList.size(); iLoop++)
		{
			String addr = (String)addressList.get(iLoop), trustStore;
			if ((_currAddr != null) && (_currAddr.compareTo(addr) == 0))
				continue;
			_currAddr = addr;
			_log.debug("Attempting to create initial context: " + _currAddr);

			String ssl = null;
			if ( ((trustStore = (String) _env.get("org.eclipse.higgins.idas.cp.jndi.trustStore")) != null)
				&& ( (_currAddr.indexOf("ldaps") == 0)
					|| ( ((ssl = (String) _env.get("java.naming.security.protocol")) != null) && ssl.equals("ssl")) ) )
			{
				_env.put("java.naming.ldap.factory.socket", "org.eclipse.higgins.idas.cp.jndi.LDAPHigginsSSLSocketFactoryProducer");
				_setupCustomSSLSocketFactory(trustStore, (String) _env.get("org.eclipse.higgins.idas.cp.jndi.trustStorePassword"));
			}
			else
			{
				_env.remove("java.naming.ldap.factory.socket");
			}

			String jndiProvider = (String)_connectionSettings.get("jndiProvider");
			if (jndiProvider == null)
				jndiProvider = "com.sun.jndi.ldap.LdapCtxFactory";
			_env.put(Context.INITIAL_CONTEXT_FACTORY, jndiProvider);
			_env.put(Context.PROVIDER_URL, _currAddr);

			try
			{
				_ctx = new InitialLdapContext(_env, null);
				break;
			}
			catch (CommunicationException e)
			{
				continue;
			}
			catch (NamingException e)
			{
				_handleNamingException(e);
			}
		}
		if (iLoop == addressList.size())
			throw new IdASException("Unable to connect to any specified JNDI service address.");
	}

	/**
	 */
	private void _destroyContext() throws IdASException
	{
		try
		{
			if (_bStartTLS)
			{
				if (_tls != null)
					_tls.close();
				_bStartTLS = false;
			}
			if (_ctx != null)
			{
				_ctx.close();
			}
		}
		catch (NamingException e)
		{
			throw new IdASException(e);
		}
		catch (IOException e)
		{
			throw new IdASException(e);
		}
		_currAddr = null;
		_identity = null;
		_ctx = null;
	}

	private String _getIdentity()
	{
		return (_identity == null ? null : _identity.toString());
	}

	/**
	 */
	public void close() throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		_bIsOpen = false;
		_emitXDASRecord(XDasEvents.XDAS_AE_TERMINATE_SESSION,
			XDasOutcomes.XDAS_OUT_SUCCESS, _getIdentity(), null, null);
		_log.debug("Context closed: " + _contextURIs + " as: " + _getIdentity());
	}

	/**
	 */
	public boolean isOpen(
		Object identity) throws IdASException
	{
		return _bIsOpen;
	}

	/**
	 * 
	 */
	public String getAuthEntityID()
	{
		return _entityID;
	}

	/**
	 */
	private DOMDocument _getInitialSchemaDoc()
	{
		/* Create an OWL document in RDF representing the schema. */
		DOMDocument doc = new DOMDocument();
		Vector entityDeclList = new Vector();

		entityDeclList.add(new InternalEntityDecl("ldapowl",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins/ldap#"));
		entityDeclList.add(new InternalEntityDecl("higgins",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins#"));
		entityDeclList.add(new InternalEntityDecl("xsd",
			"http://www.w3.org/2001/XMLSchema#"));

		DefaultDocumentType docType = new DefaultDocumentType();
		docType.setInternalDeclarations(entityDeclList);
		docType.setName("rdf:RDF");
		doc.setDocType(docType);
		doc.addComment("\r\n*******************************************************************************\r\n"
				+ "* Copyright (c) 2006 Novell Inc.\r\n"
				+ "* All rights reserved. This document and accompanying materials\r\n"
				+ "* are made available under the terms of the Eclipse Public License v1.0\r\n"
				+ "* which accompanies this distribution, and is available at\r\n"
				+ "* http://www.eclipse.org/legal/epl-v10.html\r\n"
				+ "*\r\n"
				+ "* Contributors:\r\n"
				+ "*     Tom Doman\r\n"
				+ "*     Jim Sermersheim\r\n"
				+ "*******************************************************************************\r\n");

		DOMElement elem = (DOMElement) doc.addElement("rdf:RDF");
		elem.addAttribute("xml:base",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins/ldap#");
		elem.addNamespace("rdf", "http://www.w3.org/1999/02/22-rdf-syntax-ns#");
		elem.addNamespace("rdfs", "http://www.w3.org/2000/01/rdf-schema#");
		elem.addNamespace("owl", "http://www.w3.org/2002/07/owl#");
		elem.addNamespace("ldapowl",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins/ldap#");

		DOMElement ontElem = (DOMElement) elem.addElement("owl:Ontology");
		ontElem.addAttribute("rdf:about", "");
		DOMElement childElem = (DOMElement) ontElem.addElement("rdfs:label");
		childElem.addText("Dynamically Generated Higgins-based LDAP Ontology (Context: "
			+ _contextURIs + ")");
		childElem = (DOMElement) ontElem.addElement("owl:imports");
		childElem.addAttribute("rdf:resource",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins");

		ontElem = (DOMElement) elem.addElement("owl:Ontology");
		ontElem.addAttribute("rdf:about",
			"http://www.eclipse.org/higgins/ontologies/2006/higgins");

		return (doc);
	}

	/**
	 */
	private void _getSchemaClass(
		DOMDocument schemaDoc,
		String schemaName,
		Attributes attrs) throws IdASException
	{
		try
		{
			boolean bFoundSup = false;
			DOMElement rootElem = (DOMElement) schemaDoc.getRootElement(), classElem, childElem;

			rootElem.addComment(schemaName);
			classElem = (DOMElement) rootElem.addElement("owl:Class");
			classElem.addAttribute("rdf:ID", "class_" + schemaName);
			childElem = (DOMElement) classElem.addElement("rdfs:label");
			childElem.addAttribute("rdf:datatype", "&xsd;string");
			childElem.addText(schemaName);

			NamingEnumeration attrEnum = attrs.getAll();
			while (attrEnum.hasMore())
			{
				Attribute attr = (Attribute) attrEnum.next();

				NamingEnumeration attrVals = attr.getAll();
				if (attr.getID().compareTo("SUP") == 0)
				{
					childElem = (DOMElement) classElem.addElement("rdfs:subClassOf");
					childElem.addAttribute("rdf:resource", "#class_" + attrVals.next());
					bFoundSup = true;
				}
				else if (attr.getID().compareTo("NUMERICOID") == 0)
				{
					Object oid = attrVals.next();
					childElem = (DOMElement) classElem.addElement("owl:equivalentClass");
					childElem.addAttribute("rdf:resource", "urn:oid:" + oid);
					// Define equivalent class.
					DOMElement oidClassElem = (DOMElement) rootElem.addElement("owl:Class");
					oidClassElem.addAttribute("rdf:about", "urn:oid:" + oid);
					childElem = (DOMElement) oidClassElem.addElement("rdfs:label");
					childElem.addAttribute("rdf:datatype", "&xsd;string");
					childElem.addText(schemaName);
				}
				else if (attr.getID().compareTo("MAY") == 0)
				{
					while (attrVals.hasMore())
						_addAttributeDomain(schemaName, (String) attrVals.next());
				}
				else if (attr.getID().compareTo("MUST") == 0)
				{
					while (attrVals.hasMore())
					{
						String attrName = (String) attrVals.next();
						DOMElement subClassElem = (DOMElement) classElem.addElement("rdfs:subClassOf");
						DOMElement restrictElem = (DOMElement) subClassElem.addElement("owl:Restriction");

						_addAttributeDomain(schemaName, attrName);
						childElem = (DOMElement) restrictElem.addElement("owl:onProperty");
						childElem.addAttribute("rdf:resource", attrName);
						childElem = (DOMElement) restrictElem.addElement("owl:minCardinality");
						childElem.addAttribute("rdf:datatype", "&xsd;nonNegativeInteger");
						childElem.addText("1");
					}
				}
				else if (attr.getID().compareTo("DESC") == 0)
				{
					childElem = (DOMElement) classElem.addElement("rdfs:comment");
					childElem.addAttribute("rdf:datatype", "&xsd;string");
					childElem.addText((String) attrVals.next());
				}
				else if ((attr.getID().compareTo("NAME") == 0)
					|| (attr.getID().compareTo("ABSTRACT") == 0)
					|| (attr.getID().compareTo("STRUCTURAL") == 0)
					|| (attr.getID().compareTo("AUXILIARY") == 0)
					|| (attr.getID().compareTo("OBSOLETE") == 0))
				{
					;
				}
			}
			if (!bFoundSup)
			{
				childElem = (DOMElement) classElem.addElement("rdfs:subClassOf");
				childElem.addAttribute("rdf:resource", "&higgins;Entity");
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}
	}

	/*
	private String _getAttributeNameCase(
		String attrName)
	{
		String attrCaseName = attrName;
		// TODO: Need the schema cache for this as well as other things.
		if (_schemaAttrCaseHashMap != null)
		{
			CIStringKey ciAttrName = new CIStringKey(attrName);

			attrCaseName = (String) _schemaAttrCaseHashMap.get(ciAttrName);
			if (attrCaseName == null)
			{
				_schemaAttrCaseHashMap.put(ciAttrName, attrName);
				attrCaseName = attrName;
			}
		}
		return attrCaseName;
	}
	*/

	/**
	 */
	private void _addAttributeDomain(
		String className,
		String attrName)
	{
		CIStringKey ciAttrName = new CIStringKey(attrName);

		Vector classNameVect = (Vector) _schemaAttrDomainHashMap.get(ciAttrName);
		if (classNameVect == null)
			classNameVect = new Vector();
		classNameVect.add(className);

		_schemaAttrDomainHashMap.put(ciAttrName, classNameVect);
	}

	/**
	 */
	private Vector _getAttrDomains(
		String schemaName)
	{
		CIStringKey ciAttrName = new CIStringKey(schemaName);

		return (Vector) _schemaAttrDomainHashMap.get(ciAttrName);
	}

	/**
	 */
	private void _getSchemaAttrDef(
		DOMDocument schemaDoc,
		String schemaName,
		Attributes attrs) throws IdASException
	{
		DOMElement rootElem = (DOMElement) schemaDoc.getRootElement(), propertyElem, childElem;

		rootElem.addComment(schemaName);
		propertyElem = (DOMElement) rootElem.addElement("owl:ObjectProperty");
		propertyElem.addAttribute("rdf:about", schemaName);
		childElem = (DOMElement) propertyElem.addElement("rdfs:label");
		childElem.addAttribute("rdf:datatype", "&xsd;string");
		childElem.addText(schemaName);

		try
		{
			boolean bFoundSup = false;
			NamingEnumeration attrEnum = attrs.getAll();
			while (attrEnum.hasMore())
			{
				Attribute attr = (Attribute) attrEnum.next();
				NamingEnumeration attrVals = attr.getAll();

				if (attr.getID().compareTo("SUP") == 0)
				{
					childElem = (DOMElement) propertyElem.addElement("rdfs:subPropertyOf");
					childElem.addAttribute("rdf:resource", (String) attrVals.next());
					bFoundSup = true;
				}
				else if (attr.getID().compareTo("NUMERICOID") == 0)
				{
					Object oid = attrVals.next();
					childElem = (DOMElement) propertyElem.addElement("owl:equivalentProperty");
					childElem.addAttribute("rdf:resource", "urn:oid:" + oid);
					// Define equivalent property.
					DOMElement oidPropertyElem = (DOMElement) rootElem
						.addElement("owl:ObjectProperty");
					oidPropertyElem.addAttribute("rdf:about", "urn:oid:" + oid);
					childElem = (DOMElement) oidPropertyElem.addElement("rdfs:label");
					childElem.addAttribute("rdf:datatype", "&xsd;string");
					childElem.addText(schemaName);
				}
				else if (attr.getID().compareTo("SYNTAX") == 0)
				{
					String syntax = (String) attrVals.next(), syntaxOID = syntax;
					if (syntax.indexOf('{') > 0)
						syntaxOID = syntax.substring(0, syntax.indexOf('{'));
					// Create the reference to the syntax class we'll define later.
					childElem = (DOMElement) propertyElem.addElement("rdfs:range");
					SyntaxMap syntaxMap = (SyntaxMap) _syntaxHashMap.get(syntaxOID);
					childElem.addAttribute("rdf:resource",
						syntaxMap == null ? ("urn:oid:" + syntaxOID) : ("#" + syntaxMap.getURLName()));
				}
				else if ((attr.getID().compareTo("DESC") == 0))
				{
					childElem = (DOMElement) propertyElem.addElement("rdfs:comment");
					childElem.addAttribute("rdf:datatype", "&xsd;string");
					childElem.addText((String) attrVals.next());
				}
				else if ((attr.getID().compareTo("SINGLE-VALUE") == 0))
				{
					DOMElement singleValElem = (DOMElement) rootElem.addElement("rdf:Description");
					singleValElem.addAttribute("rdf:about", "#class_top");
					DOMElement subClassElem = (DOMElement) singleValElem.addElement("rdfs:subClassOf");
					DOMElement restrictElem = (DOMElement) subClassElem.addElement("owl:Restriction");
					childElem = (DOMElement) restrictElem.addElement("owl:maxCardinality");
					childElem.addText("1");
					childElem = (DOMElement) restrictElem.addElement("owl:onProperty");
					childElem.addAttribute("rdf:resource", schemaName);
				}
				else if ((attr.getID().compareTo("NAME") == 0))
				{
					String name = (String) attrVals.next();
					if (schemaName.compareTo(name) != 0)
					{
						childElem = (DOMElement) propertyElem.addElement("owl:equivalentProperty");
						childElem.addAttribute("rdf:resource", name);
						// Define equivalent property.
						DOMElement equivPropertyElem = (DOMElement) rootElem.addElement("owl:ObjectProperty");
						equivPropertyElem.addAttribute("rdf:about", name);
						childElem = (DOMElement) equivPropertyElem.addElement("rdfs:label");
						childElem.addAttribute("rdf:datatype", "&xsd;string");
						childElem.addText(schemaName);
					}
				}
				else if ((attr.getID().compareTo("EQUALITY") == 0)
					|| (attr.getID().compareTo("ORDERING") == 0)
					|| (attr.getID().compareTo("SUBSTRING") == 0)
					|| (attr.getID().compareTo("COLLECTIVE") == 0)
					|| (attr.getID().compareTo("NO-USER-MODIFICATION") == 0)
					|| (attr.getID().compareTo("USAGE") == 0)
					|| (attr.getID().compareTo("OBSOLETE") == 0))
				{
					;
				}
			}

			// Add references to all the classes that this attribute is legal on.
			Vector attrClasses = _getAttrDomains(schemaName);
			if (attrClasses != null)
			{
				for (int iLoop = 0; iLoop < attrClasses.size(); ++iLoop)
				{
					childElem = (DOMElement) propertyElem.addElement("rdfs:domain");
					childElem.addAttribute("rdf:resource", "#class_" + attrClasses.get(iLoop));
				}
			}
			// Add the subproperty of the property.
			if (!bFoundSup)
			{
				childElem = (DOMElement) propertyElem.addElement("rdfs:subPropertyOf");
				childElem.addAttribute("rdf:resource", "&higgins;attribute");
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}
	}

	/**
	 */
	private void _getSchemaSyntaxDef(
		DOMDocument schemaDoc,
		String schemaName,
		Attributes attrs) throws IdASException
	{
		try
		{
			DOMElement rootElem = (DOMElement) schemaDoc.getRootElement(), classElem, childElem;
			SyntaxMap syntaxMap = (SyntaxMap) _syntaxHashMap.get(schemaName);

			if (syntaxMap == null)
				rootElem.addComment(schemaName);
			else
				rootElem.addComment(syntaxMap.getLabel());

			classElem = (DOMElement) rootElem.addElement("owl:Class");
			if (syntaxMap == null)
				classElem.addAttribute("rdf:about", "urn:oid:" + schemaName);
			else
				classElem.addAttribute("rdf:ID", syntaxMap.getURLName());
			childElem = (DOMElement) classElem.addElement("rdfs:label");
			childElem.addAttribute("rdf:datatype", "&xsd;string");
			childElem.addText(syntaxMap == null ? schemaName : syntaxMap.getLabel());
			childElem = (DOMElement) classElem.addElement("rdfs:subClassOf");
			childElem.addAttribute("rdf:resource",
					syntaxMap == null ? "&higgins;StringSimpleAttribute" : syntaxMap.getOWLType());

			NamingEnumeration attrEnum = attrs.getAll();
			while (attrEnum.hasMore())
			{
				Attribute attr = (Attribute) attrEnum.next();

				NamingEnumeration attrVals = attr.getAll();
				if (attr.getID().compareTo("DESC") == 0)
				{
					childElem = (DOMElement) classElem.addElement("rdfs:comment");
					childElem.addAttribute("rdf:datatype", "&xsd;string");
					childElem.addText((String) attrVals.next());
				}
				else if (attr.getID().compareTo("NUMERICOID") == 0)
				{
					if (syntaxMap != null)
					{
						Object oid = attrVals.next();
						childElem = (DOMElement) classElem.addElement("owl:equivalentClass");
						childElem.addAttribute("rdf:resource", "urn:oid:" + oid);

						// Define equivalent class.
						DOMElement oidClassElem = (DOMElement) rootElem.addElement("owl:Class");
						oidClassElem.addAttribute("rdf:about", "urn:oid:" + oid);
						childElem = (DOMElement) oidClassElem.addElement("rdfs:label");
						childElem.addAttribute("rdf:datatype", "&xsd;string");
						childElem.addText(syntaxMap.getLabel());
					}
				}
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}
	}

	/**
	 */
	public String getSchema() throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		if (_schemaAttrDomainHashMap == null)
			_schemaAttrDomainHashMap = new HashMap();
		else
			_schemaAttrDomainHashMap.clear();

		if (_schemaAttrCaseHashMap == null)
			_schemaAttrCaseHashMap = new HashMap();
		else
			_schemaAttrCaseHashMap.clear();

		DOMDocument schemaDoc = _getInitialSchemaDoc();
		// TODO: Get the schema associated with the identity used to open this
		// context instead of the root.
		try
		{
			DirContext schemaCtx;
			try {
				schemaCtx = _ctx.getSchema(_ctx.getNameInNamespace());
			} catch (NamingException e1)	{
				try {
					schemaCtx = _ctx.getSchema("");
				} catch (NamingException e2)	{
					schemaCtx = _ctx.getSchema(consumerEntityIDToProvider(_entityID));
				}
			}
			// The OWL generator depends on these containers being visited in this
			// order.
			Vector schemaContainers = new Vector();
			schemaContainers.add("ClassDefinition");
			schemaContainers.add("AttributeDefinition");
			schemaContainers.add("SyntaxDefinition");

			for (int iLoop = 0; iLoop < schemaContainers.size(); iLoop++)
			{
				NamingEnumeration list = schemaCtx.list((String)schemaContainers.elementAt(iLoop));
				DOMElement rootElem = (DOMElement) schemaDoc.getRootElement();
				rootElem.addComment((String) schemaContainers.elementAt(iLoop));

				while (list.hasMore())
				{
					NameClassPair ncPair = (NameClassPair) list.next();
					DirContext ctx = (DirContext) schemaCtx.lookup(schemaContainers.elementAt(iLoop)
						+ "/" + ncPair.getName());
					Attributes attrs = ctx.getAttributes("");

					if (((String)schemaContainers.elementAt(iLoop)).compareTo("ClassDefinition") == 0)
						_getSchemaClass(schemaDoc, ncPair.getName(), attrs);
					else if (((String)schemaContainers.elementAt(iLoop)).compareTo("AttributeDefinition") == 0)
						_getSchemaAttrDef(schemaDoc, ncPair.getName(), attrs);
					else if (((String)schemaContainers.elementAt(iLoop)).compareTo("SyntaxDefinition") == 0)
						_getSchemaSyntaxDef(schemaDoc, ncPair.getName(), attrs);
				}
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		StringWriter strWriter = new StringWriter();
		NonEscapedAttrXMLWriter xmlWriter = new NonEscapedAttrXMLWriter(strWriter);
		try
		{
			xmlWriter.write(schemaDoc);
		}
		catch (IOException e)
		{
			throw new IdASException(e);
		}
		return strWriter.toString();
	}

	/**
	 */
	public IEntity getEntity(
		String entityID) throws IdASException
	{
		return getEntity(entityID, null);
	}

	/**
	 */
	public IEntity getEntity(
		String entityID,
		Iterator consumerSelectionList) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		_emitXDASRecord(XDasEvents.XDAS_AE_QUERY_ACCOUNT, XDasOutcomes.XDAS_OUT_SUCCESS,
			_getIdentity(), entityID, null);
		_log.debug("Searching for Entity: " + entityID + " as: "
			+ _getIdentity() + " in context: " + _contextURIs);
		JNDIEntity entity = null;
		NamingEnumeration results;
		SearchControls controls = new SearchControls();
		HashSet providerSelectionList = new HashSet();
		if (consumerSelectionList != null)
		{
			_mapConsumerSelectionList(consumerSelectionList, providerSelectionList);
			/*
			 * We need these attributes always so that we can adequately answer all
			 * calls.
			 */
			_addRequiredAttrs(providerSelectionList);
			URI[] uriAttrArray = (URI[]) providerSelectionList.toArray(new URI[0]);
			String[] attrArray = new String[uriAttrArray.length];
			for (int iLoop = 0; iLoop < uriAttrArray.length; ++iLoop)
				attrArray[iLoop] = uriAttrArray[iLoop].toString();
			controls.setReturningAttributes(attrArray);
		}
		controls.setSearchScope(SearchControls.OBJECT_SCOPE);

		try
		{
			String realEntityID = consumerEntityIDToProvider(entityID);
			results = _ctx.search(realEntityID, "(objectclass=*)",
				controls);

			if (results.hasMore())
			{
				SearchResult result = (SearchResult) results.next();
				entity = new JNDIEntity(this, result, realEntityID, _attrPDPs);
				if (results.hasMore())
					throw new IdASException("Non-unique Entity ID");
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		return entity;
	}

	/**
	 */
	private void _mapConsumerSelectionList(
		Iterator consumerSelectionList,
		HashSet providerSelectionList) throws IdASException
	{
		while (consumerSelectionList.hasNext())
		{
			URI consumerAttrName = (URI) consumerSelectionList.next();
			Vector providerAttrList;

			if ((providerAttrList = _mapConsumerAttrName(consumerAttrName)) != null)
			{
				Iterator providerAttrIter = providerAttrList.iterator();
				while (providerAttrIter.hasNext())
				{
					URI providerAttrName = (URI) providerAttrIter.next();
					providerSelectionList.add(providerAttrName);
				}
			}
			else
			{
				providerSelectionList.add(consumerAttrName);
			}
		}
	}

	/**
	 */
	private Vector _mapConsumerAttrName(
		URI consumerAttrName) throws IdASException
	{
		Vector attrList = new Vector();

		if (_attrPDPs != null)
		{
			Iterator itr = _attrPDPs.consumerIDToProviders(consumerAttrName);
			while (itr.hasNext())
				attrList.add((URI) itr.next());
		}
		else
		{
			attrList.add(consumerAttrName);
		}
		return attrList;
	}

	/**
	 */
	private void _addRequiredAttrs(
		HashSet providerSelectionList) throws IdASException
	{
		try
		{
			providerSelectionList.add(new URI("objectclass"));
			providerSelectionList.add(new URI("structuralObjectClass"));
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	private void _convertFilter(
		BasicFilter basicFilter,
		RfcFilter ldapFilter) throws IdASException
	{
		try
		{
			BasicFilterAssertion assertion = (BasicFilterAssertion) basicFilter.getAssertion();
			if (assertion != null)
			{
				URI attrURI = assertion.getID();
				if (assertion instanceof BasicFilterEntityTypeAssertion)
					attrURI = new URI("objectclass");
				String comparator = assertion.getComparator();
				int rfcType = _mapComparator(comparator);
				Vector providerAttrList = _mapConsumerAttrName(attrURI);

				if (providerAttrList.size() > 1)
					ldapFilter.startNestedFilter(rfcType);

				Iterator providerAttrIter = providerAttrList.iterator();
				while (providerAttrIter.hasNext())
				{
					URI attrName = (URI) providerAttrIter.next();
					if (assertion instanceof BasicFilterAttributeAssertion)
					{
						BasicFilterAttributeAssertion attributeAssertion = (BasicFilterAttributeAssertion) assertion;
						IAttributeValue assertionValue = attributeAssertion.getAssertionValue();

						if (assertionValue == null)
						{
							switch ( rfcType )
							{
							case RfcFilter.PRESENT:
								ldapFilter.addPresent(attrName.toString());
								break;
							default:
								throw new NotImplementedException("Comparator (" + comparator
									+ ") not supported.");
							}
						}
						else if (assertionValue.isSimple())
						{
							// assertionValue.getType();
							if ((rfcType == RfcFilter.EQUALITY_MATCH)
								|| (rfcType == RfcFilter.GREATER_OR_EQUAL)
								|| (rfcType == RfcFilter.LESS_OR_EQUAL))
							{
								if ((assertionValue instanceof BasicValueBase64Binary)
									|| (assertionValue instanceof BasicValueHexBinary))
								{
									ByteBuffer byteBuf = (ByteBuffer) ((ISimpleAttrValue)assertionValue).getData();
									ldapFilter.addAttributeValueAssertion(rfcType, attrName.toString(),
										byteBuf.array());
								}
								else if ((assertionValue instanceof BasicValueString)
									|| (assertionValue instanceof BasicValueNormalizedString)
									|| (assertionValue instanceof BasicValueToken)
									|| (assertionValue instanceof BasicValueLanguage)
									|| (assertionValue instanceof BasicValueNMTOKEN)
									|| (assertionValue instanceof BasicValueName)
									|| (assertionValue instanceof BasicValueNCName))
								{
									ldapFilter.addAttributeValueAssertion(rfcType, attrName.toString(),
										((String) ((ISimpleAttrValue)assertionValue).getData()).getBytes());
								}
								else
								{
									ldapFilter.addAttributeValueAssertion(rfcType, attrName.toString(),
										((ISimpleAttrValue) assertionValue).getLexical().getBytes());
								}
							}
							else
							{
								// TODO: Handle substrings
								// if (rfcType == RfcFilter.SUBSTRINGS)
								// ldapFilter.startSubstrings();
								// if (rfcType == RfcFilter.EXTENSIBLE_MATCH)
								throw new NotImplementedException("Comparator (" + comparator
									+ ") not supported.");
							}
						}
						else
						{
							throw new NotImplementedException(
								"Complex assertion values in filters not supported.");
						}
					}
					else if (assertion instanceof BasicFilterEntityTypeAssertion)
					{
						BasicFilterEntityTypeAssertion entityTypeAssertion = (BasicFilterEntityTypeAssertion) assertion;
						ldapFilter.addAttributeValueAssertion(RfcFilter.EQUALITY_MATCH, attrName.toString(),
							consumerEntityTypeToProvider(entityTypeAssertion.getAssertionValue()).toString().getBytes());
					}
				}

				if (providerAttrList.size() > 1)
					ldapFilter.endNestedFilter(rfcType);
			}
			else
			{
				String operator = basicFilter.getOperator();
				int rfcType = _mapOperator(operator);
				ldapFilter.startNestedFilter(rfcType);
				Iterator filters = basicFilter.getFilters();
				while (filters.hasNext())
				{
					IFilter filter = (IFilter) filters.next();
					_convertFilter((BasicFilter) filter, ldapFilter);
				}
				ldapFilter.endNestedFilter(rfcType);
			}
		}
		catch (LDAPLocalException e)
		{
			throw new IdASException(e);
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	private int _mapComparator(
		String comparator)
	{
		int iRet = -1;

		if (comparator.compareTo(IFilterAttributeAssertion.COMP_ATTR_EQ) == 0)
			iRet = RfcFilter.EQUALITY_MATCH;
		else if (comparator.compareTo(IFilterAttributeAssertion.COMP_ATTR_GE) == 0)
			iRet = RfcFilter.GREATER_OR_EQUAL;
		else if (comparator.compareTo(IFilterAttributeAssertion.COMP_ATTR_LE) == 0)
			iRet = RfcFilter.LESS_OR_EQUAL;
		else if (comparator.compareTo(IFilterAttributeAssertion.COMP_ATTR_PRESENT) == 0)
			iRet = RfcFilter.PRESENT;
		else if (comparator.compareTo(IFilterAttributeAssertion.COMP_ATTR_SUBSTR) == 0)
			iRet = RfcFilter.SUBSTRINGS;

		return iRet;
	}

	/**
	 */
	private int _mapOperator(
		String operator) throws IdASException
	{
		int iRet = -1;

		if (operator.compareTo(IFilter.OP_NOT) == 0)
			iRet = RfcFilter.NOT;
		else if (operator.compareTo(IFilter.OP_AND) == 0)
			iRet = RfcFilter.AND;
		else if (operator.compareTo(IFilter.OP_OR) == 0)
			iRet = RfcFilter.OR;
		else
			throw new NotImplementedException("Operator (" + operator
				+ ") not supported.");

		return iRet;
	}

	/**
	 */
	public Iterator getEntities(
		IFilter filter) throws IdASException
	{
		return getEntities(filter, null);
	}

	/**
	 */
	public Iterator getEntities(
		IFilter filter,
		Iterator consumerSelectionList) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		_emitXDASRecord(XDasEvents.XDAS_AE_QUERY_ACCOUNT, XDasOutcomes.XDAS_OUT_SUCCESS,
			_getIdentity(), filter == null ? null : filter.toString(), null);
		_log.debug("Searching for Entities matching filter: " + filter == null ? null : filter.toString()
			+ " as: " + _getIdentity() + " in context: " + _contextURIs);
		HashSet subs = new HashSet();
		String strFilter = "(objectclass=*)";

		if (filter != null)
		{
			BasicFilter basicFilter = (BasicFilter) filter;
			// Convert the IdAS filter to an LDAP filter.
			RfcFilter ldapFilter = new RfcFilter();
			_convertFilter(basicFilter, ldapFilter);
			strFilter = ldapFilter.filterToString();
			_log.debug("LDAP String Filter: " + strFilter);
		}
		try
		{
			JNDIEntity entity;
			SearchControls controls = new SearchControls();
			NamingEnumeration results;
			HashSet providerSelectionList = new HashSet();
			if (consumerSelectionList != null)
			{
				_mapConsumerSelectionList(consumerSelectionList, providerSelectionList);
				/*
				 * We need these attributes always so that we can adequately answer
				 * all calls.
				 */
				_addRequiredAttrs(providerSelectionList);
				URI[] uriAttrArray = (URI[]) providerSelectionList.toArray(new URI[0]);
				String[] attrArray = new String[uriAttrArray.length];
				for (int iLoop = 0; iLoop < uriAttrArray.length; ++iLoop)
					attrArray[iLoop] = uriAttrArray[iLoop].toString();
				controls.setReturningAttributes(attrArray);
			}
			controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
			String searchBase = _getSearchBase();
			results = _ctx.search(searchBase, strFilter, controls);

			while (results.hasMore())
			{
				SearchResult result = (SearchResult) results.next();
				entity = new JNDIEntity(this, result, searchBase, _attrPDPs);
				subs.add(entity);
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		return subs.iterator();
	}

	/**
	 */
	public void removeEntity(
		String entityID) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();
		try
		{
			_ctx.destroySubcontext(entityID);
		}
		catch (NamingException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	public URI getSearchResultType(
		SearchResult searchResult) throws IdASException
	{
		String structuralClass = null;
		String resultName = searchResult.getName();
		out: try
		{
			Attribute ocAttr = _getObjectClass(searchResult);
			NamingEnumeration jndiAttrValEnum = ocAttr.getAll();

			if (ocAttr.size() == 1)
			{
				structuralClass = (String) jndiAttrValEnum.next();
			}
			else
			{
				while (jndiAttrValEnum.hasMore())
				{
					String jndiValue = (String) jndiAttrValEnum.next();
					if (_isStructural(resultName, jndiValue, ocAttr))
					{
						structuralClass = jndiValue;
						break out;
					}
				}
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		if (structuralClass == null)
			throw new IdASException("No structural class found.");

		URI uriType = null;
		try
		{
			uriType = new URI(structuralClass);
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}

		return providerEntityTypeToConsumer(uriType);
	}

	/**
	 */
	private boolean _isStructural(
		String entityName,
		String className,
		Attribute ocAttr) throws IdASException
	{
		NamingEnumeration jndiAttrValEnum;
		boolean bRet = false;

		out: try
		{
			jndiAttrValEnum = ocAttr.getAll();
			while (jndiAttrValEnum.hasMore())
			{
				Object jndiValue = jndiAttrValEnum.next();
				if (className.compareToIgnoreCase((String) jndiValue) != 0)
				{
					DirContext schemaCtx = _ctx.getSchema(_ctx.getNameInNamespace());
					DirContext ctx = (DirContext) schemaCtx.lookup("ClassDefinition" + "/"
						+ jndiValue);
					Attributes attrs = ctx.getAttributes("");

					String supClass = _getSuperiorClass(attrs);
					if (supClass != null)
					{
						if (className.compareToIgnoreCase(supClass) == 0)
							break out;
					}
				}
			}
			/*
			 * This class is not listed as the superior class of any class in the
			 * set of object class values.
			 */
			bRet = true;
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		return bRet;
	}

	/**
	 */
	private String _getSuperiorClass(
		Attributes attrs) throws IdASException
	{
		String superiorClass = null;
		NamingEnumeration attrEnum = attrs.getAll();

		try
		{
			while (attrEnum.hasMore())
			{
				Attribute attr = (Attribute) attrEnum.next();

				NamingEnumeration attrVals = attr.getAll();
				if (attr.getID().compareTo("SUP") == 0)
				{
					superiorClass = (String) attrVals.next();
				}
				else if ((attr.getID().compareTo("AUXILIARY") == 0)
					|| (attr.getID().compareTo("ABSTRACT") == 0))
				{
					superiorClass = null;
					break;
				}
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		return superiorClass;
	}

	/**
	 */
	private Attribute _getObjectClass(
		SearchResult searchResult) throws IdASException
	{
		Attributes jndiAttrs = searchResult.getAttributes();
		Attribute retAttr = null;
		NamingEnumeration jndiEnum;
		boolean bFoundStructural = false;

		jndiEnum = jndiAttrs.getAll();
		out: try
		{
			while (jndiEnum.hasMore())
			{
				Attribute jndiAttr = (Attribute) jndiEnum.next();
				if (jndiAttr.getID().compareToIgnoreCase("structuralObjectClass") == 0)
				{
					bFoundStructural = true;
					NamingEnumeration jndiAttrValEnum = jndiAttr.getAll();
					if (jndiAttr.size() != 1)
						throw new IdASException("Only 1 value expected, found: "
							+ jndiAttr.size());

					String jndiValue = (String) jndiAttrValEnum.next();
					if (jndiValue.compareToIgnoreCase("top") != 0)
					{
						retAttr = jndiAttr;
						break out;
					}
				}
				else if (jndiAttr.getID().compareToIgnoreCase("objectclass") == 0)
				{
					retAttr = jndiAttr;
					if (bFoundStructural)
						break out;
				}
			}
		}
		catch (NamingException e)
		{
			_handleNamingException(e);
		}

		if (retAttr == null)
			throw new IdASException("No structural or object class found.");
		return retAttr;
	}

	/**
	 * internal functions and classes
	 */

	/**
	 */
	protected String consumerEntityIDToProvider(
		String entityID) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformString(_consumerEntityIDToProviderExec,
					JScriptCPAttributePDPs.consumerIDParamName, entityID);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected String providerEntityIDToConsumer(
		String entityID) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformString(_providerEntityIDToConsumerExec,
					JScriptCPAttributePDPs.providerIDParamName, entityID);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected URI consumerEntityTypeToProvider(
		URI type) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformURI(_consumerEntityTypeToProviderExec,
					JScriptCPAttributePDPs.consumerTypeParamName, type);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected URI providerEntityTypeToConsumer(
		URI type) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformURI(_providerEntityTypeToConsumerExec,
					JScriptCPAttributePDPs.providerTypeParamName, type);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 * 
	 */
	public IEntity addEntity(
		URI type,
		String entityID) throws IdASException
	{
		JNDIEntity entity = new JNDIEntity(this, consumerEntityTypeToProvider(type),
			entityID, _attrPDPs);
		this.updateNotification(new EntityNotification(entity,
			EntityNotification.UPDATE_ADD, null));
		return entity;
	}

	/**
	 * 
	 */
	protected DirContext getDirContext()
	{
		return _ctx;
	}

	/**
	 * 
	 */
	private String _getSearchBase()
	{
		String searchBase = (String)_connectionSettings.get("searchBase");
		if (searchBase == null)
			searchBase = "";
		return searchBase;
	}

	/**
	 */
	public IEntity addEntity(
		IEntity copyFrom) throws IdASException, EntityExistsException
	{
		JNDIEntity entity = new JNDIEntity(this, copyFrom, _attrPDPs);
		this.updateNotification(new EntityNotification(entity,
			EntityNotification.UPDATE_ADD, null));
		return entity;
	}

	public void applyUpdates() throws IdASException
	{
		Hashtable updateList = super.getUpdateList();
		try
		{
			_log.debug("Applying updates ...");
			int iCount = 0;
			Iterator keyIter = updateList.keySet().iterator();
			while (keyIter.hasNext())
			{
				String entityID = (String)keyIter.next();
				_updateEntry(entityID, (Vector)updateList.get(entityID));
				iCount++;
			}
			_log.debug(iCount + ((iCount == 1) ? " change " : " changes ") + "applied");
		}
		catch (IdASException e)
		{
			throw e;
		}
		finally
		{
			updateList.clear();
		}
	}

	private void _updateEntry(
		String entityID,
		Vector entityNotifs) throws IdASException
	{
		try
		{
			EntityNotification [] aEntityNotifs = (EntityNotification []) entityNotifs.toArray(new EntityNotification[entityNotifs.size()]);
			_log.debug("Updating entry [" + entityID + "], " + aEntityNotifs[0].getAction());
			if (aEntityNotifs[0].getAction().compareTo(EntityNotification.UPDATE_ADD) == 0)
			{
				BasicAttributes jndiAttrs = _entityNotificationToBasicAttributes(aEntityNotifs);
				_ctx.createSubcontext(consumerEntityIDToProvider(entityID), jndiAttrs);
			}
			else if (aEntityNotifs[0].getAction().compareTo(EntityNotification.UPDATE_REMOVE) == 0)
			{
				_ctx.destroySubcontext(consumerEntityIDToProvider(aEntityNotifs[0].getEntity().getEntityID()));
			}
			else if (aEntityNotifs[0].getAction().compareTo(EntityNotification.UPDATE_ATTR_NOTIFY) == 0)
			{
				Vector jndiMods = _entityNotificationToModificationItems(aEntityNotifs);
				_ctx.modifyAttributes(consumerEntityIDToProvider(entityID),
					(ModificationItem [])jndiMods.toArray(new ModificationItem[jndiMods.size()]));
			}
			else
			{
				throw new NotImplementedException(aEntityNotifs[0].getAction());
			}
		}
		catch (NamingException e)
		{
			throw new IdASException(e);
		}
	}

	private BasicAttributes _entityNotificationToBasicAttributes(
		EntityNotification[] entityNotifs) throws IdASException
	{
		BasicAttributes jndiAttrs = new BasicAttributes();
		jndiAttrs.put(new BasicAttribute("objectClass", entityNotifs[0].getEntity().getModel().getType().toString()));

		for (int iLoop = 0; iLoop < entityNotifs.length; ++iLoop)
		{
			if (entityNotifs[iLoop].getAction().compareTo(EntityNotification.UPDATE_ATTR_NOTIFY) == 0)
			{
				AttributeNotification attrNotif = entityNotifs[iLoop].getAttributeNotification();
				if (attrNotif.getAction().compareTo(AttributeNotification.UPDATE_VALUE_NOTIFY) == 0)
				{
					AttributeValueNotification attrValNotif = attrNotif.getAttributeValueNotification();
					if (attrValNotif.getAction().compareTo(AttributeValueNotification.UPDATE_ADD) != 0)
						throw new IdASException("Unallowed attribute value notification type during entity creation: "
							+ attrValNotif.getAction());

					/* If we got one attribute value added, we got them all. */
					if (jndiAttrs.get(attrNotif.getAttr().getAttrID().toString()) == null)
						jndiAttrs.put(_IAttributeToAttribute(attrNotif.getAttr()));
				}
				else if (attrNotif.getAction().compareTo(AttributeNotification.UPDATE_ADD) == 0) 
				{
					/* If we get an associated value notify, we'll handle it here or above. */
					if (jndiAttrs.get(attrNotif.getAttr().getAttrID().toString()) == null)
						jndiAttrs.put(_IAttributeToAttribute(attrNotif.getAttr()));
				}
				else
				{
					/* Any other kind of attribute notification should not occur during entity creation. */ 
					throw new IdASException("Unallowed attribute notification type during entity creation: "
						+ attrNotif.getAction());
				}
			}
			else if (entityNotifs[iLoop].getAction().compareTo(EntityNotification.UPDATE_ADD) != 0)
			{
				throw new IdASException("Unallowed entity notification type during entity creation: "
					+ entityNotifs[iLoop].getAction());
			}
		}
		return jndiAttrs;
	}

	private Vector _entityNotificationToModificationItems(
		EntityNotification[] entityNotifs) throws IdASException
	{
		Vector jndiMods = new Vector();
		_log.debug("Entity notifications: " + entityNotifs.length);
		for (int iLoop = 0; iLoop < entityNotifs.length; ++iLoop)
		{
			AttributeNotification attrNotif = entityNotifs[iLoop].getAttributeNotification();
			_log.debug("Converting attribute notification to modification item: " + attrNotif.getAction()
					+ ", " + attrNotif.getAttr().getAttrID().toString());

			if (attrNotif.getAction().compareTo(AttributeNotification.UPDATE_VALUE_NOTIFY) == 0)
			{
				jndiMods.add(_attributeValueNotificationToModificationItem(attrNotif));
			}
			else if (attrNotif.getAction().compareTo(AttributeNotification.UPDATE_REMOVE) == 0)
			{
				AttributeNotification addNotif = null, addValueNotif = null;

				if (iLoop + 2 < entityNotifs.length)
				{
					addNotif = entityNotifs[iLoop + 1].getAttributeNotification();
					addValueNotif = entityNotifs[iLoop + 2].getAttributeNotification();
				}

				if ((addNotif != null) && (addValueNotif != null)
					&& (attrNotif.getAttr().getAttrID().toString().compareTo(
						addNotif.getAttr().getAttrID().toString()) == 0)
					&& (addNotif.getAction().compareTo(AttributeNotification.UPDATE_ADD) == 0)
					&& (addValueNotif.getAction().compareTo(AttributeNotification.UPDATE_VALUE_NOTIFY) == 0))
				{
					Attribute attr = _AttrNotifToJNDIAttr(addValueNotif);
					jndiMods.add(new ModificationItem(DirContext.REPLACE_ATTRIBUTE, attr));
					iLoop += 2;
					continue;
				}
				else
				{
					jndiMods.add(new ModificationItem(DirContext.REMOVE_ATTRIBUTE,
							new BasicAttribute(attrNotif.getAttr().getAttrID().toString())));					
				}
			}
			/* We should receive a value notify for every attribute value added. */
			else if (attrNotif.getAction().compareTo(AttributeNotification.UPDATE_ADD) != 0)
			{
				throw new NotImplementedException(attrNotif.getAction());
			}
		}

		return jndiMods;
	}

	private Attribute _AttrNotifToJNDIAttr(
		AttributeNotification attrNotif) throws IdASException
	{
		Attribute attr = null;
		try
		{
			String attrID = attrNotif.getAttr().getAttrID().toString();
			attr = new BasicAttribute(attrID);
			AttributeValueNotification attrValNotif = attrNotif.getAttributeValueNotification();
			IAttributeValue attrVal = attrValNotif.getAttributeValue();
			IAttributeValue providerVal;
			URI providerType = (URI)_mapConsumerAttrName(new URI(attrID)).firstElement();
			
			if (attrVal.isSimple()) {
				ISimpleAttrValue val = (ISimpleAttrValue)attrVal;
				providerVal = (IAttributeValue)_attrPDPs.consumerValueToProvider(
					val.getModel().getType(), null, providerType, null, attrVal);
				String strVal = ((ISimpleAttrValue)providerVal).getLexical();
				_log.debug("Adding value to JNDI attribute: " + strVal);
				attr.add(strVal);
			}
			else {
				throw new NotImplementedException("Cannot convert complex attributes to JNDI attributes");
			}
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}

		return attr;
	}

	private ModificationItem _attributeValueNotificationToModificationItem(
		AttributeNotification attrNotif) throws IdASException
	{
		ModificationItem retModItem = null;
		Attribute attr = _AttrNotifToJNDIAttr(attrNotif);
		AttributeValueNotification attrValNotif = attrNotif.getAttributeValueNotification();

		if (attrValNotif.getAction().compareTo(AttributeValueNotification.UPDATE_ADD) == 0)
			retModItem = new ModificationItem(DirContext.ADD_ATTRIBUTE, attr);
		else if (attrValNotif.getAction().compareTo(AttributeValueNotification.UPDATE_REMOVE) == 0)
			retModItem = new ModificationItem(DirContext.REMOVE_ATTRIBUTE, attr);
		else if (attrValNotif.getAction().compareTo(AttributeValueNotification.UPDATE_SET_DATA) == 0)
			retModItem = new ModificationItem(DirContext.REPLACE_ATTRIBUTE, attr);
		else
			throw new NotImplementedException(attrValNotif.getAction());

		return retModItem;
	}

	private Attribute _IAttributeToAttribute(
		IAttribute attr) throws IdASException
	{
		BasicAttribute jndiAttr = null;
		try
		{
			URI providerType = (URI) _mapConsumerAttrName(
				new URI(attr.getAttrID().toString())).firstElement();
			IAttributeValue providerVal;
			jndiAttr = new BasicAttribute(providerType.toString());
			_log.debug("Converting consumer attribute to provider attribute: " + attr.getAttrID().toString());

			// put each value into the jndi attr
			Iterator consumerVals = attr.getValues();
			while (consumerVals.hasNext())
			{
				IAttributeValue consumerVal = (IAttributeValue) consumerVals.next();
				if (consumerVal.isSimple()) {
					ISimpleAttrValue val = (ISimpleAttrValue)consumerVal;
					providerVal = (ISimpleAttrValue)_attrPDPs.consumerValueToProvider(
						val.getModel().getType(), null, providerType, null, consumerVal);
					jndiAttr.add(((ISimpleAttrValue)providerVal).getLexical());
				}
				else {
					throw new NotImplementedException("Cannot convert complex attributes to JNDI attributes: "
						+ consumerVal.getClass().toString());
				}
			}
		}
		catch (URISyntaxException e)
		{
			throw new IdASException(e);
		}
		return jndiAttr;
	}

	/**
	 * 
	 */
	public IContextModel getContextModel()
	{
		return new JNDIContextModel(this);
	}
}
