/**
 * Copyright (c) 2007-2008 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 *		Jim Sermersheim
 */

package org.eclipse.higgins.idas.cp.jspolicy;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.util.idas.cp.BasicAttributePDP;
import org.eclipse.higgins.util.idas.cp.BasicAttributePDPIter;
import org.eclipse.higgins.util.idas.cp.MetadataPDPIter;
import org.eclipse.higgins.util.idas.cp.jscript.JScriptCPAttributePDPs;
import org.eclipse.higgins.util.idas.cp.jscript.JScriptCPMetadataPDPs;

/**
 * A <a href="http://www.eclipse.org/higgins/">Higgins</a> Entity
 * implementation allowing for javascript policy at critical points. Utilizing
 * Mozilla's <a href"http://www.mozilla.org/rhino/">Rhino</a> project, this
 * code currently supports JavaScript PDPs at the following points:
 * 
 * <UL>
 * <li>{@link #getEntityID()} Entity IDs may be mapped from stored (real) values to
 * what is presented by consumers of this Context Provider.</li>
 * <li>{@link #getEntityType()} Entity types may be mapped from stored
 * (real) values to what is presented by consumers of this Context Provider.</li>
  * </UL>
 * 
 * Additional PDPs found in
 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext},
 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyEntity}, and
 * TODO add doc
 * 
 * @author dbuss@novell.com
 * @author tdoman@novell.com
 */

public class JSPolicyEntity implements IEntity, IHasMetadata
{
	private JSPolicyContext _policyContext;
	private IEntity _providerEntity;
	private String _uniqueID;
	private URI _type;
	private JScriptCPMetadataPDPs _dsMetadataPDPs;
	private JScriptCPMetadataPDPs _attributeValueMetadataPDP;
	private JScriptCPMetadataPDPs _attributeMetadataPDP;
	private JScriptCPAttributePDPs _attributePDP;
	
	/*
	 */
	protected JSPolicyEntity(
			JScriptCPMetadataPDPs dsMetadataPDPs,
			JScriptCPAttributePDPs attributePDP,
			JScriptCPMetadataPDPs attributeValueMetadataPDP,
			JScriptCPMetadataPDPs attributeMetadataPDP,
			JSPolicyContext policyContext,
			IEntity providerEntity,
			String consumerEntityID,
			URI consumerURI) throws IdASException
	{
		_policyContext = policyContext;
		_providerEntity = providerEntity;
		_uniqueID = consumerEntityID;
		_type = consumerURI;

		_dsMetadataPDPs = dsMetadataPDPs; 
		_attributeValueMetadataPDP =  attributeValueMetadataPDP;
		_attributeMetadataPDP = attributeMetadataPDP;
		_attributePDP =  attributePDP;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IEntity#getContext
	 */
	public IContext getContext() throws IdASException
	{
		return _policyContext;
	}

	/**
	 * 
	 * Returns the Subject ID of this Entity.
	 * 
	 * This Context Provider allows Entity IDs to be mapped from stored (real) values
	 * to what is presented by consumers of this Context Provider.
	 * <p>
	 * 
	 * Please see
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityIDToProvider}
	 * for usage and examples.
	 * 
	 * @see org.eclipse.higgins.idas.api.IEntity#getEntityID
	 */
	public String getEntityID() throws IdASException
	{
		if (_uniqueID == null)
			_uniqueID = _policyContext.providerEntityIDToConsumer(_providerEntity.getEntityID());

		return _uniqueID;
	}

	/**
	 * Returns the type of this Entity.
	 * 
	 * This Context Provider allows Entity types to be mapped from
	 * stored (real) values to what is presented by consumers of this Context
	 * Provider by inserting JavaScript into the configuration file.
	 * <p>
	 * 
	 * Please see
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityIDToProvider}
	 * 
	 * 
	 */
	public URI getEntityType() throws IdASException
	{
		if (_type == null)
			_type = _policyContext.providerEntityTypeToConsumer(_providerEntity.getEntityType());
		return _type;
	}

	/**
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException
	{
		return new BasicAttributePDPIter( _attributePDP, _attributeMetadataPDP, 
				_attributeValueMetadataPDP,
			_providerEntity.getAttributes(), true);
	}

	/**
	 * This Context Provider allows Attribute types to be mapped from stored
	 * (real) values to what is presented by consumers of this Context Provider
	 * by inserting JavaScript into the configuration file. Since one consumer
	 * type may map to more than one virtual type each potential type is examined
	 * to see if the provider object has values.
	 * 
	 * Because of the error prone nature of this call it is recomended that it be
	 * used with extreme caution.
	 * 
	 * The PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerATypeToProvider}
	 * contains the type mappings.
	 * <p>
	 * 
	 * TODO: Should we have a PDP just for this function which controls the
	 * behavior of error handling and type mapping?
	 *
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(
		URI attributeID) throws IdASException
	{
		if (_attributePDP != null)
		{
			IAttribute providerAttr = null;
			Iterator itr = _attributePDP.consumerIDToProviders(attributeID);

			while (itr.hasNext())
			{
				Object obj = itr.next();
				if (obj instanceof URI)
				{
					providerAttr = _providerEntity.getAttribute((URI) obj);
				}
				else if (obj instanceof String)
				{
					try
					{
						providerAttr = _providerEntity.getAttribute(new URI(((String) (obj))));
					}
					catch (URISyntaxException e)
					{
						throw new IdASException(e);
					}
				}
				else
					throw new IdASException(
						"Illegal object type returned from mapping, consumer URI, got: "
							+ obj.getClass().toString());
				if (providerAttr != null)
					break;
			}
			if (providerAttr != null)
				return new BasicAttributePDP(_attributePDP, 
						_attributeMetadataPDP, _attributeValueMetadataPDP,
						providerAttr,true);
			else
				return null;
		}
		else
		{
			/* no policy */
			return _providerEntity.getAttribute(attributeID);
		}
	}
	
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrType) throws IdASException, NotSingleValuedAttributeException {
		IAttribute attr = getAttribute(attrType);
		if (attr.isSingleValued()) {
			return (ISingleValuedAttribute)attr;
		}
		else {
			throw new NotSingleValuedAttributeException(attrType + "is not a single-valued attribute");
		}
	}	

	
	/**
	 * TODO: this may return unmapped metadata, this should be fixed.
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadataSet()
	 */
	public Iterator getMetadataSet() throws IdASException
	{
		if ( _dsMetadataPDPs != null)
			return  new MetadataPDPIter( _dsMetadataPDPs, 
					_providerEntity.getMetadataSet(), true);
		else
			return _providerEntity.getMetadataSet();
	}

	/**
	 * 
	 */
	public IMetadata getMetadata(URI metadataID) throws IdASException
	{
		IMetadata metadataItem = null;
		Iterator iter = this.getMetadataSet();
		while (iter.hasNext())
		{
			metadataItem = (IMetadata)iter.next();
			if (metadataID.equals(metadataItem.getModel().getType()))
				break;
		}
		return metadataItem;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(java.net.URI)
	 */
	public IMetadata addMetadata(URI type) throws IdASException, InvalidTypeException 
	{
		throw new NotImplementedException();
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(org.eclipse.higgins.idas.api.IMetadata)
	 */
	public IMetadata addMetadata(IMetadata copyFrom) throws IdASException 
	{
		throw new NotImplementedException();
	}

	/**
	 * TODO: implement this
	 */
	public IEntityModel getModel() throws IdASException
	{
		throw new NotImplementedException();
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IEntity#remove()
	 */
	public void remove() throws IdASException {
		_providerEntity.remove();
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(URI type) throws IdASException, InvalidTypeException 
	{
		return new BasicAttributePDP(_attributePDP, _attributeMetadataPDP, 
				_attributeValueMetadataPDP,  null, false);
	}

	/**
	 * TODO there are three cases we have to deal with 
	 * 1 - The attribute is a generic IAttribute created from some other iContext
	 * 2 - The attribute was returned from this Policy Context and the values are already mapped
	 * 3 - The attribute was created using addAttribute and values need to be mapped. 
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException 
	{
		if ( copyFrom instanceof BasicAttributePDP)
			return new BasicAttributePDP(_attributePDP, _attributeMetadataPDP, 
					_attributeValueMetadataPDP,  copyFrom,  true);
		else
			return new BasicAttributePDP(_attributePDP, _attributeMetadataPDP, 
					_attributeValueMetadataPDP, copyFrom, true);
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		// TODO Auto-generated method stub
		return false;
	}

	public void removeAttribute(URI attrType) throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(URI attrType, Object value)
			throws IdASException {
		// TODO Auto-generated method stub
		
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		// TODO Auto-generated method stub
		
	}

}
