/**
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *		Tom Doman
 */

package org.eclipse.higgins.idas.cp.jspolicy;

import java.lang.reflect.Constructor;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidEntityIDException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.idas.spi.BasicContext;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.util.idas.cp.BasicAttributePDPIter;
import org.eclipse.higgins.util.idas.cp.jscript.JScriptCPAttributePDPs;
import org.eclipse.higgins.util.idas.cp.jscript.JScriptCPMetadataPDPs;
import org.eclipse.higgins.util.jscript.JScriptException;
import org.eclipse.higgins.util.jscript.JScriptExec;
import org.eclipse.higgins.util.jscript.JScriptExecHelper;

/**
 * A <a href="http://www.eclipse.org/higgins/">Higgins</a> Context Provider
 * implementation allowing for javascript policy at critical points. Utilizing
 * Mozilla's <a href"http://www.mozilla.org/rhino/">Rhino</a> project, this
 * code currently supports JavaScript PDPs at the following points:
 * 
 * <UL>
 * <li>{@link #getEntity(String entityID)}, and
 * {@link #getEntity(java.lang.String, java.util.Iterator)} calls allow entity IDs
 * to be mapped from stored (provider) values to what is presented by consumers of
 * this Context Provider.</li>
 * </UL>
 * 
 * Additional PDPs found in
 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext},
 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyEntity}, and
 * 
 * @author dbuss@novell.com
 * @author tdoman@novell.com
 */

public class JSPolicyContext extends BasicContext
{
	public static final String connectionType = "org.eclipse.higgins.idas.cp.jspolicy";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Entity IDs passed to this context provider before being passed to the real
	 * context provider.
	 * <p>
	 * When this PDP is invoked the Entity ID is converted to a string, stored in an
	 * evaluation specific JavaScript scope using the name
	 * {@link #consumerIDParamName}. It is presented that either the result of the
	 * script <b>or</b> a JavaScript variable with the name
	 * {@link #genericResultName} is the new Entity ID.
	 * <p>
	 * To avoid problems it is recomended that the reverse mapping also be
	 * defined using the id {@link #providerEntityIDToConsumer}
	 * <p>
	 * A simple nonsensical sample where the provider Entity IDs all start with the
	 * letters 'vc' and those are replaced by the letters 'js':
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerEntityIDToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			var re = new RegExp(&quot;&circ;js&quot;, &quot;g&quot;);
	 * 			RESULT = consumerID.replace(re, &quot;vc&quot;);
	 * 		&lt;/JSPolicyAction&gt;
	 * 		&lt;JSPolicyAction id=&quot;providerEntityIDToConsumer&quot; evalType=&quot;javascript&quot;&gt;
	 * 			var re = new RegExp(&quot;&circ;vc&quot;, &quot;g&quot;);
	 * 			RESULT = consumerID.replace(re, &quot;js&quot;);
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 */
	public static final String consumerEntityIDToProvider = "consumerEntityIDToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Entity IDs as they are returned from the real context provider to this context
	 * provider. The inverse of this mapping is {@link #consumerEntityIDToProvider}
	 */
	public static final String providerEntityIDToConsumer = "providerEntityIDToConsumer";

	/* cached PDP for the above named functions */
	private JScriptExec _consumerEntityIDToProviderExec;
	private JScriptExec _providerEntityIDToConsumerExec;

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Metadata ID passed to this context provider before being passed to the
	 * real context provider.  Applies to metadata on Entities, Attributes, and
	 * Attribute Values.  
	 * <p>
	 * @see org.eclipse.higgins.util.idas.cp.jscript.JScriptCPMetadataPDPs#consumerIDToProvider
	 */
	public static final String consumerMIDToProvider = "consumerMIDToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute ID as they are returned from the real context provider to
	 * this context provider. The inverse of this mapping is
	 * {@link #consumerMIDToProvider}
	 */
	public static final String providerMIDToConsumer = "providerMIDToConsumer";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute ID passed to this context provider before being passed to the
	 * real context provider.
	 * <p>
	 * @see org.eclipse.higgins.util.idas.cp.jscript.JScriptCPAttributePDPs#consumerIDToProvider
	 */
	public static final String consumerAIDToProvider = "consumerAIDToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute ID as they are returned from the real context provider to
	 * this context provider. The inverse of this mapping is
	 * {@link #consumerAIDToProvider}
	 */
	public static final String providerAIDToConsumer = "providerAIDToConsumer";
	
	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Metadata Types passed to this context provider before being passed to the
	 * real context provider.
	 * <p>
	 * @see org.eclipse.higgins.util.idas.cp.jscript.JScriptCPMetadataPDPs#consumerTypeToProvider
	 */
	public static final String consumerMTypeToProvider = "consumerMTypeToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Metadata Types as they are returned from the real context provider to
	 * this context provider. The inverse of this mapping is
	 * {@link #consumerMTypeToProvider}
	 */
	public static final String providerMTypeToConsumer = "providerMTypeToConsumer";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute Types passed to this context provider before being passed to the
	 * real context provider.
	 * <p>
	 * @see org.eclipse.higgins.util.idas.cp.jscript.JScriptCPAttributePDPs#consumerTypeToProvider
	 */
	public static final String consumerATypeToProvider = "consumerATypeToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute Types as they are returned from the real context provider to
	 * this context provider. The inverse of this mapping is
	 * {@link #consumerATypeToProvider}
	 */
	public static final String providerATypeToConsumer = "providerATypeToConsumer";
	
	
	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Entity Types passed to this context provider before being passed
	 * to the real context provider.
	 * <p>
	 * When this PDP is invoked the Entity Type is stored in an
	 * evaluation specific JavaScript scope using the name
	 * {@link #consumerTypeParamName}. It is presented that either the result of the
	 * script <b>or</b> a JavaScript variable with the name
	 * {@link #genericResultName} is the new Entity Type.
	 * <p>
	 * To avoid problems it is recomended that the reverse mapping also be
	 * defined using the id {@link #providerEntityTypeToConsumer}
	 * <p>
	 * 
	 * A simple nonsensical sample where the provider Entity Type mappings
	 * are stored in a globally scoped attribute multimap, and are referenced
	 * using JavaScript Associative Arrays.
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerEntityTypeToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			RESULT = multimap.consumer[String(PARAM.toString())];
	 * 		&lt;/JSPolicyAction&gt;
	 * 		&lt;JSPolicyAction id=&quot;providerEntityTypeToConsumer&quot; evalType=&quot;javascript&quot;&gt;
	 * 			RESULT = multimap.provider[String(PARAM.toString())];
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 * 
	 * <p>
	 * TODO: currently this is for Attribute types, Entity MetaData
	 * Types, and Attribute MetaData Types, should we have seperate PDPs for
	 * those? The internal code is already fully instrumented for the addtional
	 * PDPs.
	 */
	public static final String consumerEntityTypeToProvider = "consumerEntityTypeToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Entity Types as they are returned from the real context provider
	 * to this context provider. The inverse of this mapping is
	 * {@link #consumerEntityTypeToProvider}
	 */
	public static final String providerEntityTypeToConsumer = "providerEntityTypeToConsumer";

	/* cached PDP for the above named functions */
	private JScriptExec _consumerEntityTypeToProviderExec;
	private JScriptExec _providerEntityTypeToConsumerExec;

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute Values passed to this context provider before being passed to
	 * the real context provider.
	 * <p>
	 * When this PDP is invoked the consumer type, the provider type and the
	 * Attribute Value are all stored in an evaluation specific JavaScript scope
	 * using the names {@link #consumerTypeParamName},
	 * {@link #providerTypeParamName}, {@link #genericParamName} respectivly. It is
	 * presented that either the result of the script <b>or</b> a JavaScript
	 * variable with the name {@link #genericResultName} is the new value
	 * <p>
	 * To avoid problems it is recomended that the reverse mapping also be
	 * defined using the id {@link #providerAValueToConsumer}
	 * <p>
	 * 
	 * A simple nonsensical sample where the for anything which is Entity ID the same
	 * mappings is preformed as noted in the sample for
	 * {@link #consumerEntityTypeToProvider}
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerEntityTypeToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			importPackage(Packages.org.eclipse.higgins.idas);
	 * 			importPackage(Packages.org.eclipse.higgins.idas.impl);
	 * 
	 * 			var re = new RegExp(&quot;&circ;vc&quot;, &quot;g&quot;);
	 * 			if (consumerType == &quot;EntityIDSyntax&quot;) {
	 * 				RESULT = new Packages.org.eclipse.higgins.idas.impl.BasicValueString(
	 * 				String(PARAM.getData()).replace(re, &quot;js&quot;));
	 * 			}
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 */
	public static final String consumerAValueToProvider = "consumerAValueToProvider";
	public static final String consumerMValueToProvider = "consumerMValueToProvider";

	
	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute values passed to this context provider before being passed to
	 * the real context provider. The inverse of this mapping is
	 * {@link #consumerMValueToProvider}
	 */
	public static final String providerAValueToConsumer = "providerAValueToConsumer";
	public static final String providerMValueToConsumer = "providerMValueToConsumer";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute Values passed to this context provider before being passed to
	 * the real context provider during a context open call.
	 * <p>
	 * When this PDP is invoked the consumer type, the provider type and the
	 * Attribute Value are all stored in an evaluation specific JavaScript scope
	 * using the names {@link #consumerTypeParamName},
	 * {@link #providerTypeParamName}, {@link #genericParamName} respectivly. It is
	 * presented that either the result of the script <b>or</b> a JavaScript
	 * variable with the name {@link #genericResultName} is the new value
	 * <p>
	 * Please note this is a unidirctional mapping, there is no inverse.
	 * 
	 * A simple nonsensical sample where the for anything which is Entity ID the same
	 * mappings is preformed as noted in the sample for
	 * {@link #consumerEntityTypeToProvider}
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerOpenValueToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			importPackage(Packages.org.eclipse.higgins.idas);
	 * 			importPackage(Packages.org.eclipse.higgins.idas.impl);
	 * 
	 * 			var re = new RegExp(&quot;&circ;vc&quot;, &quot;g&quot;);
	 * 			if (consumerType == &quot;EntityIDSyntax&quot;) {
	 * 				RESULT = new Packages.org.eclipse.higgins.idas.impl.BasicValueString(
	 * 				String(PARAM.getData()).replace(re, &quot;js&quot;));
	 * 			}
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 */
	public static final String consumerOpenValueToProvider = "consumerOpenValueToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Attribute Types passed to this context provider before being passed to the
	 * real context provider in the course of a context open call.
	 * <p>
	 * When this PDP is invoked the Attribute Type is stored in an evaluation
	 * specific JavaScript scope using the name {@link #genericParamName}. It is
	 * presented that either the result of the script <b>or</b> a JavaScript
	 * variable with the name {@link #genericResultName} is the new Attribute
	 * Type.
	 * <p>
	 * This is a one way map, no values are returned.
	 * 
	 * A simple nonsensical sample where the provider Entity Type mappings
	 * are stored in a globally scoped attribute multimap, and are referenced
	 * using JavaScript Associative Arrays.
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerOpenAIDToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			RESULT = multimap.consumer[PARAM];
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 */
	public static final String consumerOpenAIDToProvider = "consumerOpenAIDToProvider";

	/**
	 * Identifier used in the <code>id</code> attribute of a
	 * <code>JSPolicyAction</code> element to specify the mapping performed on
	 * Context URI during the context open call. When this PDP is invoked the
	 * consumer type, the mapping context provider's Context Ref is stored in an
	 * evaluation specific JavaScript scope using the name
	 * {@link #genericParamName}. It is presented that either the result of the
	 * script <b>or</b> a JavaScript variable with the name
	 * {@link #genericResultName} is the new value
	 * <p>
	 * Please note this is a unidirctional mapping, there is no inverse.
	 * 
	 * A simple sample which ignores the passed in PARAM and just returns a new
	 * value
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerOpenContextRefToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			new java.lang.URI(&quot;http://bogus.data.frogspawn:port&quot;);
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 */
	public static final String consumerOpenContextRefToProvider = "consumerOpenContextRefToProvider";

	/* cached PDP for the above named functions */
	private JScriptExec _consumerOpenContextRefToProviderExec;

	/**
	 * Generic name which is used on many of the PDPs to pass a single parameter
	 * from java to the script.
	 */
	public static final String genericParamName = "PARAM";

	/**
	 * Generic name which is used on many of the PDPs to pass a single parameter
	 * from java to the script.
	 */
	public static final String genericParamName2 = "PARAM2";

	/**
	 * Generic name which is used on many of the PDPs to pass the result from
	 * JavaScript back to JAVA PDP code.
	 */
	public static final String genericResultName = "RESULT";

	/**
	 * Generic parameter name used by selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String consumerIDParamName = "consumerID";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String providerIDParamName = "providerID";
	
	/**
	 * Generic parameter name used by selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String consumerTypeParamName = "consumerType";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String providerTypeParamName = "providerType";

	/**
	 * Generic parameter name used by selected PDPs. Often the variable name for
	 * the presenting Context.
	 */
	public static final String consumerContext = "consumerContext";

	/**
	 * Generic parameter name used by selected PDPs. Often the variable name for
	 * the provider Context.
	 */
	public static final String providerContext = "providerContext";

	/**
	 * Generic parameter name used by selected PDPs. Often the variable name for
	 * the attr PDPCache
	 */
	public static final String attrPDPCache = "attrPDPCache";

	/**
	 * Generic parameter name used by selected PDPs. Often the variable name for
	 * the attrMetadataPDPCache
	 */
	public static final String attrMetadataPDPCache = "attrMetadataPDPCache";

	// TODO - define PDP string names for these and populate them on open,
	// provide meaningful documentation and samples
	public static final String getSchema = "getSchema";
	public static final String setSchema = "setSchema";
	public static final String exportData = "exportData ";
	private JScriptExec _getSchemaExec;
	private JScriptExec _setSchemaExec;
	// private JScriptExec _importDataExec;
	private JScriptExec _exportDataExec;

	/**
	 * Misc private members
	 */
	private IContextId _contextID;
	private Map _contextSettings;
	private boolean _bIsOpen;
	private boolean _providerContextOwned;
	private IContext _providerContext;

	private JScriptCPMetadataPDPs _dsMetadataPDPs;
	private JScriptCPAttributePDPs _attributePDP;
	private JScriptCPMetadataPDPs _attributeValueMetadataPDP;
	private JScriptCPMetadataPDPs _attributeMetadataPDP;
	private JScriptCPAttributePDPs _openPDPs;

	/**
	 * @param contextRef
	 * @param contextSettings
	 * @throws IdASException
	 */
	protected JSPolicyContext(
			IContextId contextIDs) throws IdASException
	{

		_contextID = contextIDs;
		_contextSettings = _contextID.getConfiguration();
		_bIsOpen = false;
		_providerContext = null;
		_providerContextOwned = false;
		Map jsPolicySettings = (Map)_contextSettings.get("JSPolicyAction");
		if (jsPolicySettings != null)
		{
			_consumerEntityIDToProviderExec = (JScriptExec)jsPolicySettings.get(consumerEntityIDToProvider);
			_providerEntityIDToConsumerExec = (JScriptExec)jsPolicySettings.get(providerEntityIDToConsumer);

			_consumerEntityTypeToProviderExec = (JScriptExec)jsPolicySettings.get(consumerEntityTypeToProvider);
			_providerEntityTypeToConsumerExec = (JScriptExec)jsPolicySettings.get(providerEntityTypeToConsumer);

			_consumerOpenContextRefToProviderExec = (JScriptExec)jsPolicySettings.get(consumerOpenContextRefToProvider);
			_getSchemaExec = (JScriptExec)jsPolicySettings.get(getSchema);
			_setSchemaExec = (JScriptExec)jsPolicySettings.get(setSchema);
			_exportDataExec = (JScriptExec)jsPolicySettings.get(exportData);

			_dsMetadataPDPs = new JScriptCPMetadataPDPs(
									(JScriptExec)jsPolicySettings.get(consumerMIDToProvider),
									(JScriptExec)jsPolicySettings.get(providerMIDToConsumer),
									(JScriptExec)jsPolicySettings.get(consumerMTypeToProvider),
									(JScriptExec)jsPolicySettings.get(providerMTypeToConsumer),
									(JScriptExec)jsPolicySettings.get(consumerMValueToProvider),
									(JScriptExec)jsPolicySettings.get(providerMValueToConsumer), 
									false);

			// Currently these meta data PDPs are shared
			_attributeValueMetadataPDP = _dsMetadataPDPs;
			_attributeMetadataPDP = _dsMetadataPDPs;

			// get the attribute and open pdps setup
			_attributePDP = new JScriptCPAttributePDPs(
									(JScriptExec)jsPolicySettings.get(consumerAIDToProvider), 
									(JScriptExec)jsPolicySettings.get(providerAIDToConsumer),																	
									(JScriptExec)jsPolicySettings.get(consumerATypeToProvider), 
									(JScriptExec)jsPolicySettings.get(providerATypeToConsumer),
									(JScriptExec)jsPolicySettings.get(consumerAValueToProvider), 
									(JScriptExec)jsPolicySettings.get(providerAValueToConsumer), 
									false);

			_openPDPs = new JScriptCPAttributePDPs(
									(JScriptExec)jsPolicySettings.get(consumerOpenAIDToProvider), 
									null, null, null,
									(JScriptExec)jsPolicySettings.get(consumerOpenValueToProvider),
									null, true);
		}
	}
	
	/**
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object, IExtension[])
	 */
	// TODO: Add a new PDP for extensions?
	public String open(
		Object identity, IExtension[] extensions) throws IdASException 
	{
		String entityID = null;

		if (_bIsOpen)
			throw new ContextOpenException();

		if (identity instanceof IContext)
		{
			// this case is only for debug/unit tests!!!
			_providerContext = (IContext) identity;
			_providerContextOwned = false;
		}
		else if (identity instanceof IAuthNAttributesMaterials)
		{
			IdASRegistry reg = IdASRegistry.getInstance();
			try
			{
				IContextId providerContextID = (IContextId)_consumerOpenContextRefToProviderExec.evaluate(
							genericParamName,	(Object)_contextID);
		
				List factories =  reg.getContextFactories(providerContextID);
				for (Iterator itr = factories.iterator(); itr.hasNext(); ) 
				{
					IContextFactory factory = (IContextFactory) itr.next();
					_providerContext = factory.createContext(providerContextID);
					if (_providerContext != null)
						break;
				}

				if (_providerContext == null)
					throw new IdASException("Unable to open Context" + "  identity : "
						+ identity.toString() + "  context ID : " + _contextID.toString());

				// convert from consumer to provider properties on the identity
				// preserving the original identity class
				Class c = identity.getClass();
				BasicAttributePDPIter props = new BasicAttributePDPIter(_openPDPs,
						null, null,
					((IAuthNAttributesMaterials) identity).getAttributes(), false);
				Object params[] = {props};
				Object newIdentity = null;
				Constructor[] cons = c.getConstructors();
				for (int iPos = 0; iPos < cons.length; iPos++)
				{
					Class potentialParams[] = cons[iPos].getParameterTypes();
					//TODO: How are our auth materials now?
					if (potentialParams.length == 1
						&& potentialParams[0].isAssignableFrom(Class.forName("java.util.Iterator")))
					{
						newIdentity = cons[iPos].newInstance(params);
						break;
					}
				}
				// open the real Context Provider
				if (newIdentity != null)
					entityID = providerEntityIDToConsumer(_providerContext.open(newIdentity, extensions));
				else
					entityID = providerEntityIDToConsumer(_providerContext.open(identity, extensions));
			}
			catch (Exception e)
			{
				throw new IdASException(e);
			}
			_providerContextOwned = true;
		}

		_bIsOpen = true;

		return entityID;
		
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object)
	 */
	public String open(
		Object identity) throws IdASException
	{
		return this.open(identity, null);
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#close()
	 */
	public void close() throws IdASException
	{
		if (_bIsOpen == false)
			throw new ContextNotOpenException();

		_bIsOpen = false;

		if (_providerContext != null && _providerContextOwned == true)
			_providerContext.close();
		_providerContext = null;
		_providerContextOwned = false;
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#isOpen(java.lang.Object)
	 */
	public boolean isOpen(
		Object identity) throws IdASException
	{
		if (!_bIsOpen || _providerContext == null)
			return false;
		else
			return _providerContext.isOpen(identity);
	}

	/**
	 * This Context Provider allows types and values to be mapped from stored
	 * (provider) values to what is presented by consumers of this Context Provider
	 * by inserting JavaScript into the configuration file.
	 * 
	 * The PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityTypeToProvider}
	 * contains the type mappings performed on the Entity ID, the PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerMTypeToProvider}
	 * describes the mappings performed on the attribute types. <p> TODO: Attrs
	 * are assumed to always map consistiently, but with generated attrs it's
	 * possible that both the generated and the real attribute should be returned
	 * if passed in a selection list.
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String,
	 *      java.util.Iterator)
	 */
	public IEntity getEntity(
		String entityID,
		Iterator consumerSelectionList) throws IdASException
	{
		if (!_bIsOpen || _providerContext == null)
			throw new ContextNotOpenException();

		HashSet providerSelectionList = null;
		if (consumerSelectionList != null)
		{
			providerSelectionList = new HashSet();
			while (consumerSelectionList.hasNext())
			{
				URI consumerAttrName = (URI) consumerSelectionList.next();
				if (_attributePDP != null)
				{
					Iterator itr = _attributePDP.consumerIDToProviders(consumerAttrName);
					while (itr.hasNext())
					{
						Object obj = itr.next();
						if (obj instanceof URI)
						{
							providerSelectionList.add((URI) obj);
						}
						else if (obj instanceof String)
						{
							try
							{
								providerSelectionList.add(new URI(((String) (obj))));
							}
							catch (URISyntaxException e)
							{
								throw new IdASException(e);
							}
						}
						else
							throw new IdASException("Illegal object type returned"
								+ " from mapping, consumer URI, got: "
								+ obj.getClass().toString());
					}
				}
				else
				{
					providerSelectionList.add(consumerAttrName);
				}
			}
		}

		return new JSPolicyEntity(_dsMetadataPDPs,  _attributePDP, _attributeMetadataPDP,
				_attributeMetadataPDP, this, _providerContext.getEntity(consumerEntityIDToProvider(entityID),
				(providerSelectionList != null) ? providerSelectionList.iterator() : null), entityID, null);
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String)
	 */
	public IEntity getEntity(
		String entityID) throws IdASException
	{
		return getEntity(entityID, null);
	}
	
	/**
	 * This Context Provider allows Entity Types, Entity ID and
	 * Attribute values to be mapped from values passed by consumers of this
	 * Context Provider to types and values for storage.
	 * 
	 * The PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityTypeToProvider}
	 * contains the type mappings performed on the Subjec type, the PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityIDToProvider}
	 * describes the mappings performed on the Entity ID.
	 * 
	 */
	public IEntity addEntity(URI entityType, String entityID) 
		throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException 
	{
		if (!_bIsOpen || _providerContext == null)
			throw new ContextNotOpenException();

		IEntity providerDS = _providerContext.addEntity(
				consumerEntityTypeToProvider(entityType), consumerEntityIDToProvider(entityID));
		
		return new JSPolicyEntity(	_dsMetadataPDPs,  
				_attributePDP, _attributeMetadataPDP, _attributeMetadataPDP,
				this, providerDS, entityID, entityType);
	}

	/**
	 * TODO implement this function
	 * 
	 * This Context Provider allows Entity Types, Entity ID, metadata,
	 * and attribute values to be mapped from values passed by consumers of this
	 * Context Provider to types and values for storage.
	 * 
	 * The PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityTypeToProvider}
	 * contains the type mappings performed on the Entity type, the PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerEntityIDToProvider}
	 * describes the mappings performed on the Entity ID.  the PDP described by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerATypeToProvider}
	 * describes the mappings performed on the attribute types. The PDP described
	 * by
	 * {@link org.eclipse.higgins.idas.cp.jspolicy.JSPolicyContext#consumerAValueToProvider}
	 * describes the value mappings which will occur. <p>
	 * 
	 */
	public IEntity addEntity(IEntity copyFrom) throws IdASException, EntityExistsException 
	{
		JSPolicyEntity consumerDS = (JSPolicyEntity)this.addEntity(
						copyFrom.getModel().getType(), copyFrom.getEntityID());

		/* deep copy of attributes */
		Iterator itr = copyFrom.getAttributes();
		while (itr != null && itr.hasNext()) 
		{
			consumerDS.addAttribute((IAttribute)itr.next());
		}
		
		return consumerDS;
	}
	
	/**
	 * @see org.eclipse.higgins.idas.api.IContext#exportData(java.lang.String,
	 *      java.lang.String)
	 */
	public String exportData(
		String filter,
		String representationFormat) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		if (_exportDataExec != null)
		{
			/* paramNames and params must be in the same order */
			final String[] paramNames =
			{genericParamName, genericParamName2, consumerContext, providerContext,
				attrPDPCache, attrMetadataPDPCache};
			Object[] params =
			{filter, representationFormat, this, _providerContext, _attributePDP, _attributeMetadataPDP};

			Object value;
			try
			{
				value = _exportDataExec.evaluate(paramNames, params);
			}
			catch (JScriptException e)
			{
				throw new IdASException(e);
			}

			return (String) value;
		}
		else
		{
			return _providerContext.exportData(filter, representationFormat);
		}
	}

	/**
	 * TODO Implement this function?
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#importData(java.lang.String,
	 *      java.lang.String)
	 */
	public void importData(
		String filter,
		String representationFormat) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		/* paramNames and params must be in the same order */
		// final String[] paramNames = {genericParamName, genericParamName2,
		// consumerContext, providerContext,
		// attrPDPCache, attrMetadataPDPCache};
		// Object[] params = {filter, representationFormat,
		// this, _providerContext,
		// _attrPropertyPDP, _attrMetadataPDP};
		throw new NotImplementedException();
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#getSchema()
	 */
	public String getSchema() throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		if (_getSchemaExec != null)
		{
			/* paramNames and params must be in the same order */
			final String[] paramNames =
			{genericParamName, consumerContext, providerContext, attrPDPCache,
				attrMetadataPDPCache};
			Object[] params =
			{_providerContext.getSchema(), this, _providerContext, _attributePDP,
				_attributeMetadataPDP};

			Object value;
			try
			{
				value = _getSchemaExec.evaluate(paramNames, params);
			}
			catch (JScriptException e)
			{
				throw new IdASException(e);
			}
			return (String) value;
		}
		else
			return _providerContext.getSchema();
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#setSchema(java.lang.String)
	 */
	public void setSchema(
		String schema) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		if (_setSchemaExec != null)
		{
			/* paramNames and params must be in the same order */
//			final String[] paramNames =
//				{genericParamName, consumerContext, providerContext, attrPDPCache,
//				attrMetadataPDPCache};
//			Object[] params =
//				{schema, this, _providerContext, _attrPropertyPDP, _attrMetadataPDP};

			// _providerContext.setSchema(
			// (String)_setSchemaExec.evaluate(paramNames, params));
		}
		else
		{
			_providerContext.setSchema(schema);
		}
	}

	/**
	 * TODO implement this function
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntities(org.eclipse.higgins.idas.api.IFilter)
	 */
	public Iterator getEntities(
		IFilter filter) throws IdASException
	{
		return getEntities(filter, null);
	}

	/**
	 * TODO implement this function
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getEntities(org.eclipse.higgins.idas.api.IFilter,
	 *      java.util.Iterator)
	 */
	public Iterator getEntities(
		IFilter filter,
		Iterator consumerSelectionList) throws IdASException		
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();

		throw new NotImplementedException();
	}

	/**
	 * @see org.eclipse.higgins.idas.api.IContext#verifyEntityAttributes(java.lang.String,
	 *      java.util.Iterator)
	 */
	public boolean verifyEntityAttributes(
		String entityID,
		Iterator attributes) throws IdASException
	{
		if (!_bIsOpen)
			throw new ContextNotOpenException();
	
		return _providerContext.verifyEntityAttributes(consumerEntityIDToProvider(entityID),
			new BasicAttributePDPIter(_attributePDP, _attributeMetadataPDP,
					_attributeValueMetadataPDP, attributes, false));
	}

	/**
	 * TODO implement this function
	 * 
	 * @see org.eclipse.higgins.idas.api.IContext#getContextModel()
	 */
	public IContextModel getContextModel() throws IdASException
	{
		throw new NotImplementedException();
	}

	/**
	 * internal functions and classes
	 */

	/**
	 */
	protected String consumerEntityIDToProvider(
		String entityID) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformString(_consumerEntityIDToProviderExec,
					consumerIDParamName, entityID);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected String providerEntityIDToConsumer(
		String entityID) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformString(_providerEntityIDToConsumerExec,
					providerIDParamName, entityID);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected URI consumerEntityTypeToProvider(
		URI type) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformURI(_consumerEntityTypeToProviderExec,
					consumerTypeParamName, type);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}

	/**
	 */
	protected URI providerEntityTypeToConsumer(
		URI type) throws IdASException
	{
		try
		{
			return JScriptExecHelper.transformURI(_providerEntityTypeToConsumerExec,
					providerTypeParamName, type);
		}
		catch (JScriptException e)
		{
			throw new IdASException(e);
		}
	}
}
