/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import org.apache.log4j.Logger;

import org.eclipse.higgins.idas.spi.BasicContext;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;

import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;

/**
 * 
 * @author dsanders@novell.com
 * 
 */

abstract public class SharedAttrContainer
{
	private java.util.Map			_attrMap;
	private java.util.ArrayList	_attrList;

	SharedAttrContainer()
	{
		_attrMap = new java.util.HashMap();
		_attrList = new java.util.ArrayList();
	}
	
	abstract String getContainerID();
	
	abstract String getContainerType();
	
	abstract Logger getLogger();
	
	java.util.ArrayList getAttrList()
	{
		return( _attrList);
	}
	
	java.util.Map getAttrMap()
	{
		return( _attrMap);
	}
	
	void copyAttr(
		SharedAttribute	copyAttribute) throws IdASException
	{
		SharedAttribute		sharedAttr;
		java.util.ArrayList	valueList = copyAttribute.getValueList();
		String					szAttrID = copyAttribute.getAttrID();
		String					szDataType = copyAttribute.getAttrDataType();

		sharedAttr = new SharedAttribute( this, szAttrID, szDataType, (szDataType != null) ? copyAttribute.isSimpleDataType() : false);
		_attrMap.put( szAttrID, sharedAttr);
		_attrList.add( sharedAttr);
		for (int iLoop = 0; iLoop < valueList.size(); iLoop++)
		{
			sharedAttr.copyValue( (SharedAttributeValue)valueList.get( iLoop));
		}
	}
	
	/**
	 */
	SharedAttribute getAttribute(
		String	szAttrID) throws IdASException
	{
		return( (SharedAttribute)_attrMap.get( szAttrID));
	}
	
	void addAttr(
		SharedAttribute	sharedAttr) throws IdASException
	{
		if (_attrMap.get( sharedAttr.getAttrID()) != null)
		{
			String	szErrMsg = "Attribute '" + sharedAttr.getAttrID() + "' defined multiple times on value for " + getContainerType() + " '" + getContainerID() + "'";
			getLogger().debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		_attrMap.put( sharedAttr.getAttrID(), sharedAttr);
		_attrList.add( sharedAttr);
	}
	
	/**
	 */
	SharedAttribute getSingleValuedAttribute(
		String	szAttrID) throws IdASException, NotSingleValuedAttributeException
	{
		SharedAttribute	attr;
		
		if ((attr = (SharedAttribute)_attrMap.get( szAttrID)) != null)
		{
			if (attr.getValueList().size() > 1)
			{
				String	szErrMsg = "getSingleValuedAttribute: Attribute '" + szAttrID + "' on " + getContainerType() +
					" '" + getContainerID() + "' is NOT single-valued";
				getLogger().debug( szErrMsg);
				throw new NotSingleValuedAttributeException( szErrMsg);
			}
		}
		return( attr);
	}

	/**
	 */
	SharedAttribute addAttribute(
		String	szAttrID) throws IdASException, InvalidTypeException
	{
		SharedAttribute	attr = null;
		
		if ((attr = (SharedAttribute)_attrMap.get( szAttrID)) == null)
		{
			attr = new SharedAttribute( this, szAttrID, null, false);
			_attrMap.put( szAttrID, attr);
			_attrList.add( attr);
		}
		return( attr);
	}
	
	/**
	 */
	SharedAttribute addAttribute(
		IAttribute copyFrom) throws IdASException
	{
		SharedAttribute		sharedAttr = null;
		java.util.Iterator	valueIterator;
		IAttributeValue		attrValue;
		String					szErrMsg;
		IAttributeModel		attrModel;
		IAttributeValueModel	attrValueModel;
		String					szAttrID;
		String					szAttrDataType;
		java.net.URI			attrDataType;
		boolean					bIsSimpleDataType;
		
		// Get the attribute ID - verify that it is not already defined
		
		szAttrID = copyFrom.getAttrID().toString();
		if (_attrMap.get( szAttrID) != null)
		{
			szErrMsg = "addAttribute(copyFrom): Attempting to add attribute ID '" + szAttrID + "' that already exists on " + getContainerType() +
			" '" + getContainerID() + "'";
			getLogger().debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		// Get the attribute type
		
		szAttrDataType = null;
		attrDataType = null;
		bIsSimpleDataType = false;
		if ((valueIterator = copyFrom.getValues()) == null ||
			 !valueIterator.hasNext())
		{
			
			// Do a try/catch here in case either getModel or getValueModel are
			// not implemented on the copyFrom attribute.
			
			try
			{
				if ((attrModel = copyFrom.getModel()) != null)
				{
					if ((attrValueModel = attrModel.getValueModel()) != null)
					{
						attrDataType = attrValueModel.getType();
						szAttrDataType = attrDataType.toString();
						bIsSimpleDataType = attrValueModel.isSimple();
					}
				}
			}
			catch (NotImplementedException e)
			{
				szAttrDataType = null;
			}
		}
		else
		{
			attrValue = (IAttributeValue)valueIterator.next();
			attrDataType = attrValue.getDataType();
			szAttrDataType = attrDataType.toString();
			bIsSimpleDataType = attrValue.isSimple();
		}
		
		// Create the value list
		
		sharedAttr = new SharedAttribute( this, szAttrID, szAttrDataType, bIsSimpleDataType);
		
		// Now add all of the values from the copy attribute to this new attribute.
		
		if ((valueIterator = copyFrom.getValues()) != null)
		{
			while (valueIterator.hasNext())
			{
				attrValue = (IAttributeValue)valueIterator.next();
				if (bIsSimpleDataType)
				{
					sharedAttr.addValueToList( new SharedSimpleAttrValue( sharedAttr, ((ISimpleAttrValue)attrValue).getData()));
				}
				else
				{
					java.util.Iterator		attrIter = ((IComplexAttrValue)attrValue).getAttributes();
					SharedComplexAttrValue	complexAttrValue = new SharedComplexAttrValue( sharedAttr);
					
					while (attrIter.hasNext())
					{
						complexAttrValue.addAttribute( (IAttribute)attrIter.next());
					}
					sharedAttr.addValueToList( complexAttrValue);
				}
			}
		}
		
		_attrMap.put( szAttrID, sharedAttr);
		_attrList.add( sharedAttr);
		return( sharedAttr);
	}

	/**
	 */
	void removeAttribute(
		String	szAttrID) throws IdASException
	{
		String				szErrMsg;
		SharedAttribute	attr;
		
		if ((attr = (SharedAttribute)_attrMap.remove( szAttrID)) == null)
		{
			szErrMsg = "removeAttribute: Could not find attribute '" + szAttrID + "' on " + getContainerType() + " '" + getContainerID() + "'";
			getLogger().debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		else
		{
			// Find and remove the attribute in the list
			
			attr.setDeleted();
			for (int iLoop = 0; iLoop < _attrList.size(); iLoop++)
			{
				if (attr == (SharedAttribute)_attrList.get( iLoop))
				{
					_attrList.remove( iLoop);
					break;
				}
			}
		}
	}

	/**
	 */
	void removeAttributeValue(
		String	szAttrID,
		Object	value) throws IdASException
	{
		String						szErrMsg;
		SharedAttribute			sharedAttr;
		int							iFoundIndex;
		SharedSimpleAttrValue	simpleValue;
		java.util.ArrayList		valueList;
		ISimpleAttrValue			attrValue;
		
		if ((sharedAttr = (SharedAttribute)_attrMap.get( szAttrID)) == null)
		{
			szErrMsg = "removeAttributeValue(attrID,value): Could not find attribute '" + szAttrID + "' on " + getContainerType() + " '" + getContainerID() + "'";
			getLogger().debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		if (sharedAttr.getAttrDataType() != null)
		{
			if (!sharedAttr.isSimpleDataType())
			{
				szErrMsg = "removeAttributeValue(attrID,value): Type for values on attribute '" + szAttrID + "' on " + getContainerType() + " '" + getContainerID() + "' is NOT a simple type";
				getLogger().debug( szErrMsg);
				throw new IdASException( szErrMsg);
			}
			attrValue = BasicContext.createSimpleValue( java.net.URI.create( sharedAttr.getAttrDataType()), value, null);
			if ((iFoundIndex = sharedAttr.findAttrValue( attrValue)) == -1)
			{
				szErrMsg = "removeAttributeValue(attrID,value): Value on attribute '" + szAttrID + "' on " + getContainerType() + " '" + getContainerID() + "' not found";
				getLogger().debug( szErrMsg);
				throw new IdASException( szErrMsg);
			}
			valueList = sharedAttr.getValueList();
			simpleValue = (SharedSimpleAttrValue)valueList.get( iFoundIndex);
			simpleValue.setDeleted();
			valueList.remove( iFoundIndex);
		}
	}

	/**
	 */
	//VISIT: Double check semantics with jimse here - needs to be all or nothing?
	void removeAttributeValue(
		IAttribute	attr) throws IdASException
	{
		SharedAttribute		sharedAttr;
		String					szErrMsg;
		String					szAttrID = attr.getAttrID().toString();
		java.util.Iterator	valueIterator;
		SharedAttributeValue	sharedAttrValue;
		int						iFoundIndex;
		java.util.ArrayList	valueList;
		
		if ((sharedAttr = (SharedAttribute)_attrMap.get( szAttrID)) == null)
		{
			szErrMsg = "removeAttributeValue(attr): Could not find attribute '" + szAttrID + "' on " + getContainerType() + " '" + getContainerID() + "'";
			getLogger().debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		if ((valueIterator = attr.getValues()) != null && sharedAttr.getAttrDataType() != null)
		{
			valueList = sharedAttr.getValueList();
			while (valueIterator.hasNext())
			{
				if ((iFoundIndex = sharedAttr.findAttrValue( (IAttributeValue)valueIterator.next())) != -1)
				{
					sharedAttrValue = (SharedAttributeValue)valueList.get( iFoundIndex);
					sharedAttrValue.setDeleted();
					valueList.remove( iFoundIndex);
				}
			}
		}
	}

	/**
	 */
	boolean equals(
		IHasAttributes	attributes) throws IdASException
	{
		java.util.Iterator	attrIterator = attributes.getAttributes();
		IAttribute				attr;
		String					szAttrID;
		java.util.HashMap		attrsMatchedMap = new java.util.HashMap();
		SharedAttribute		testValueAttr;
		SharedAttribute		testValueAttr2;

		while (attrIterator.hasNext())
		{
			attr = (IAttribute)attrIterator.next();
			szAttrID = attr.getAttrID().toString();
			
			// See if we have this attribute.
			
			if ((testValueAttr = (SharedAttribute)_attrMap.get( szAttrID)) == null)
			{
				return( false);
			}
			
			// Compare the two attribute's to see if they are equal.
			
			if (!testValueAttr.equals( attr))
			{
				return( false);
			}
			
			// If we have already matched this attribute, it means it is in there
			// twice, which would be an error.  We should probably throw an
			// exception.  For now, we will just return false.
			
			if (attrsMatchedMap.get( szAttrID) != null)
			{
				return( false);
			}
			attrsMatchedMap.put( szAttrID, testValueAttr);
		}
		
		// Make sure we found each of our attributes in the attributes list.
		
		for (int iLoop = 0; iLoop < _attrList.size(); iLoop++)
		{
			testValueAttr = (SharedAttribute)_attrList.get( iLoop);
			if ((testValueAttr2 = (SharedAttribute)attrsMatchedMap.get( testValueAttr.getAttrID())) == null ||
				 testValueAttr2 != testValueAttr)
			{
				return( false);
			}
		}
		return( true);
	}
	
	public boolean equals(
		SharedAttributeValue	testValue) throws IdASException
	{
		if (!(testValue instanceof SharedComplexAttrValue))
		{
			return( false);
		}
		return( equals( (SharedAttrContainer)((SharedComplexAttrValue)testValue)));
	}
	
	boolean equals(
		SharedAttrContainer	testValue) throws IdASException
	{
		java.util.HashMap		attrsMatchedMap = new java.util.HashMap();
		java.util.Map			testValueAttrMap = testValue.getAttrMap();
		java.util.ArrayList	testValueAttrList = testValue.getAttrList();
		SharedAttribute		sharedAttr;
		SharedAttribute		testValueAttr;
		SharedAttribute		testValueAttr2;
		String					szAttrID;
		
		// Make sure each of the attributes is found in the other list
		
		for (int iLoop = 0; iLoop < _attrList.size(); iLoop++)
		{
			sharedAttr = (SharedAttribute)_attrList.get( iLoop);
			szAttrID = sharedAttr.getAttrID();
			
			// See if the complex value we are comparing to has this attribute.
			
			if ((testValueAttr = (SharedAttribute)testValueAttrMap.get( szAttrID)) == null)
			{
				return( false);
			}
			
			// Compare the two attribute's to see if they are equal.
			
			if (!testValueAttr.equals( sharedAttr))
			{
				return( false);
			}
			
			// If we have already matched this attribute, it means it is in there
			// twice, which would be an error.  We should probably throw an
			// exception.  For now, we will just return false.
			
			if (attrsMatchedMap.get( szAttrID) != null)
			{
				return( false);
			}
			
			// Remember the test attribute we compared.  We will use this
			// later to see if all of the attributes in the test value were matched.
			
			attrsMatchedMap.put( szAttrID, testValueAttr);
		}
		
		// Make sure we found each of the attributes in value
		
		for (int iLoop = 0; iLoop < testValueAttrList.size(); iLoop++)
		{
			testValueAttr = (SharedAttribute)testValueAttrList.get( iLoop);
			if ((testValueAttr2 = (SharedAttribute)attrsMatchedMap.get( testValueAttr.getAttrID())) == null ||
				 testValueAttr2 != testValueAttr)
			{
				return( false);
			}
		}
		return( true);
	}
}

