/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.ValueAlreadyExistsException;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;

/**
 */
public class SharedAttribute
{
	private Log							_log = LogFactory.getLog( SharedAttribute.class.getName());
	private SharedAttrContainer	_sharedAttrContainer;
	private String						_szAttrID;
	private String						_szAttrDataType;
	private boolean					_bIsSimpleDataType;
	private boolean					_bDeleted;
	private java.util.ArrayList	_attrValueList;

	SharedAttribute(
		SharedAttrContainer	sharedContainer,
		String					szAttrID,
		String					szAttrDataType,
		boolean					bIsSimpleDataType)
	{
		_bDeleted = false;
		_sharedAttrContainer = sharedContainer;
		_szAttrID = szAttrID;
		_szAttrDataType = szAttrDataType;
		_bIsSimpleDataType = bIsSimpleDataType;
		_attrValueList = new java.util.ArrayList();
	}
	
	void copyValue(
		SharedAttributeValue	copyVal) throws IdASException
	{
		SharedAttributeValue	sharedAttrValue;
		
		if (copyVal instanceof SharedSimpleAttrValue)
		{
			sharedAttrValue = new SharedSimpleAttrValue( this, ((SharedSimpleAttrValue)copyVal).getData());
		}
		else
		{
			java.util.ArrayList		copyAttrList = ((SharedComplexAttrValue)copyVal).getAttrList();
			SharedComplexAttrValue	complexValue = new SharedComplexAttrValue( this);
			
			for (int iLoop = 0; iLoop < copyAttrList.size(); iLoop++)
			{
				complexValue.copyAttr( (SharedAttribute)copyAttrList.get( iLoop));
			}
			sharedAttrValue = complexValue;
		}
		_attrValueList.add( sharedAttrValue);
	}
	
	void addValueToList(
		SharedAttributeValue	attrValue)
	{
		_attrValueList.add( attrValue);
	}
	
	void setDeleted()
	{
		_bDeleted = true;
	}
	
	boolean isDeleted()
	{
		return( _bDeleted);
	}
	
	java.util.ArrayList getValueList()
	{
		return( _attrValueList);
	}
	
	SharedAttrContainer getSharedAttrContainer()
	{
		return( _sharedAttrContainer);
	}

	String getAttrID()
	{
		return( _szAttrID);
	}
	
	String getAttrDataType()
	{
		return( _szAttrDataType);
	}
	
	void setAttrDataType(
		String	szAttrDataType,
		boolean	bIsSimpleDataType)
	{
		_szAttrDataType = szAttrDataType;
		_bIsSimpleDataType = bIsSimpleDataType;
	}
	
	boolean isSimpleDataType() throws IdASException
	{
		if (_szAttrDataType == null)
		{
			String	szErrMsg = "Type for value on attribute '" + _szAttrID + "' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "' not defined";
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		return( _bIsSimpleDataType);
	}

	SharedSimpleAttrValue addSimpleValue(
		String	szDataType,
		Object	data) throws IdASException, InvalidTypeException
	{
		String						szErrMsg;
		SharedSimpleAttrValue	simpleValue = null;
		
		// Make sure the value is of the correct type.
		
		if (!_szAttrDataType.equals( szDataType))
		{
			szErrMsg = "addSimpleValue(type,data): Invalid value type '" + szDataType + "' specified for attribute '" + _szAttrID +
							"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new InvalidTypeException( szErrMsg);
		}
		
		simpleValue = new SharedSimpleAttrValue( this, data);
		
		// See if the simple value already exists.  If so, it is an error to add it twice.
		
		if (findAttrValue( simpleValue) != -1)
		{
			szErrMsg = "addSimpleValue(type,data): Attempting to add a duplicate value for attribute '" + _szAttrID +
							"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new ValueAlreadyExistsException( szErrMsg);
		}
		_attrValueList.add( simpleValue);
		return( simpleValue);
	}

	SharedComplexAttrValue addComplexValue(
		String	szDataType) throws IdASException, InvalidTypeException
	{
		String						szErrMsg;
		SharedComplexAttrValue	complexValue = null;
		
		// Make sure the value is of the correct type.
		
		if (!_szAttrDataType.equals( szDataType))
		{
			szErrMsg = "addComplexValue(type): Invalid value type '" + szDataType + "' specified for attribute '" + _szAttrID +
							"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new InvalidTypeException( szErrMsg);
		}
		complexValue = new SharedComplexAttrValue( this);
		_attrValueList.add( complexValue);
		return( complexValue);
	}

	SharedSimpleAttrValue addSimpleValue(
		IAttributeValue	copyFrom) throws IdASException
	{
		String						szErrMsg;
		String						szDataType = copyFrom.getValueType().toString();
		SharedSimpleAttrValue		simpleValue = null;
				
		// Make sure the value is of the correct type.
		
		if (!_szAttrDataType.equals( szDataType))
		{
			szErrMsg = "addSimpleValue(copy): Invalid value type '" + szDataType + "' specified for attribute '" + _szAttrID +
				"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new InvalidTypeException( szErrMsg);
		}
		
		simpleValue = new SharedSimpleAttrValue( this, ((ISimpleAttrValue)copyFrom).getData());
		
		// See if the simple value already exists.  If so, it is an error to add it twice.
		
		if (findAttrValue( simpleValue) != -1)
		{
			szErrMsg = "addSimpleValue(copyFrom): Attempting to add a duplicate value for attribute '" + _szAttrID +
							"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new ValueAlreadyExistsException( szErrMsg);
		}
		_attrValueList.add( simpleValue);
		return( simpleValue);
	}

	SharedComplexAttrValue addComplexValue(
		IAttributeValue	copyFrom) throws IdASException
	{
		String						szErrMsg;
		String						szDataType = copyFrom.getValueType().toString();
		SharedComplexAttrValue	complexValue = null;
		java.util.Iterator		attrIter;
		
		// Make sure the value is of the correct type.
		
		if (!_szAttrDataType.equals( szDataType))
		{
			szErrMsg = "addSimpleValue(copy): Invalid value type '" + szDataType + "' specified for attribute '" + _szAttrID +
				"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new InvalidTypeException( szErrMsg);
		}
		
		attrIter = ((IComplexAttrValue)copyFrom).getAttributes();
		complexValue = new SharedComplexAttrValue( this);
		while (attrIter.hasNext())
		{
			complexValue.addAttribute( (IAttribute)attrIter.next());
		}
		
		// See if the complex value already exists.  If so, it is an error to add it twice.
		
		if (findAttrValue( complexValue) != -1)
		{
			szErrMsg = "addComplexValue(copyFrom): Attempting to add a duplicate value for attribute '" + _szAttrID +
							"' on " + _sharedAttrContainer.getContainerType() + " '" + _sharedAttrContainer.getContainerID() + "'";
			_log.debug( szErrMsg);
			throw new ValueAlreadyExistsException( szErrMsg);
		}
		_attrValueList.add( complexValue);
		return( complexValue);
	}

	void remove() throws IdASException
	{
		_sharedAttrContainer.removeAttribute( _szAttrID);
		_bDeleted = true;
	}
	
	boolean equals(
		SharedAttribute	testAttr) throws IdASException
	{
		String					szTestAttrType;
		java.util.ArrayList	testValueList;
		java.util.HashMap		matchedValuesMap;
		int						iFoundIndex;
		
		// See if the attribute IDs are the same
		
		if (!_szAttrID.equals( testAttr.getAttrID()))
		{
			return( false);
		}
		
		// See if the attribute value types are the same.
		
		if ((szTestAttrType = testAttr.getAttrDataType()) != null)
		{
			if (_szAttrDataType == null ||
				 !_szAttrDataType.equals( szTestAttrType) ||
				 _bIsSimpleDataType != testAttr.isSimpleDataType())
			{
				return( false);
			}
		}
		else
		{
			if (_szAttrDataType != null)
			{
				return( false);
			}
			
			// Both attribute types are null, which means that neither has
			// any values.  Hence, they are equal.
			
			return( true);
		}
		
		// The number of values in each value list must be the same.
		
		testValueList = testAttr.getValueList();
		if (testValueList.size() != _attrValueList.size())
		{
			return( false);
		}
		
		// See if all of the values in the test attribute match a value in our value list.
		
		matchedValuesMap = new java.util.HashMap();
		for (int iLoop = 0; iLoop < testValueList.size(); iLoop++)
		{
			if ((iFoundIndex = findAttrValue( (SharedAttributeValue)testValueList.get( iLoop))) == -1)
			{
				return( false);
			}
			
			// If we have already matched this value, it cannot be legal to match it again.
			// THIS SHOULD PROBABLY BE AN ERROR AS IT IS ILLEGAL TO HAVE TWO IDENTICAL VALUES IN THE LIST
			
			if (matchedValuesMap.get( "" + iFoundIndex) != null)
			{
				return( false);
			}
			matchedValuesMap.put( "" + iFoundIndex, "");
		}
		
		// Verify that we matched all of the values in our value list.
		
		for (int iLoop = 0; iLoop < _attrValueList.size(); iLoop++)
		{
			if (matchedValuesMap.get( "" + iLoop) == null)
			{
				return( false);
			}
		}
		
		// If we get to here, everything that needs to match does.
		
		return( true);
	}
	
	/**
	 */
	boolean equals(
		IAttribute	testAttr) throws IdASException
	{
		java.util.Iterator	testAttrValueIterator;
		IAttributeValue		testAttrValue;
		int						iTestAttrValueCount;
		String					szDataType = null;
		boolean					bIsSimpleDataType = false;
		IAttributeModel		testAttrModel;
		IAttributeValueModel	testAttrValueModel;
		java.util.HashMap		matchedValuesMap;
		int						iFoundIndex;
		
		// Attribute ID must match
		
		if (!_szAttrID.equals( testAttr.getAttrID().toString()))
		{
			return( false);
		}
		
		// See if all of the values of the test attribute match this attribute's values.
		
		iTestAttrValueCount = 0;
		matchedValuesMap = new java.util.HashMap();
		if ((testAttrValueIterator = testAttr.getValues()) != null)
		{
			while (testAttrValueIterator.hasNext())
			{
				iTestAttrValueCount++;
				testAttrValue = (IAttributeValue)testAttrValueIterator.next();
				
				// Verify that the type matches once we have the first value.
				
				if (szDataType == null)
				{
					bIsSimpleDataType = testAttrValue.isSimple();
					if (bIsSimpleDataType != _bIsSimpleDataType)
					{
						return( false);
					}
					szDataType = testAttrValue.getValueType().toString();
					if (_szAttrDataType == null || !_szAttrDataType.equals( szDataType))
					{
						return( false);
					}
				}
				if ((iFoundIndex = findAttrValue( testAttrValue)) == -1)
				{
					return( false);
				}
				
				// If we have already matched this value, it cannot be legal to match it again.
				// THIS SHOULD PROBABLY BE AN ERROR AS IT IS ILLEGAL TO HAVE TWO IDENTICAL VALUES IN THE LIST
				
				if (matchedValuesMap.get( "" + iFoundIndex) != null)
				{
					return( false);
				}
				matchedValuesMap.put( "" + iFoundIndex, "");
			}
			
			// If we didn't get any values, we need to get the test attribute's value model
			// to see if it matches this attribute's type.
			
			if (iTestAttrValueCount == 0)
			{
				
				// Do a try/catch here in case either getModel or getValueModel are
				// not implemented, or we cannot get the type or simple flag.
				
				try
				{
					if ((testAttrModel = testAttr.getModel()) != null)
					{
						if ((testAttrValueModel = testAttrModel.getValueModel()) != null)
						{
							szDataType = testAttrValueModel.getType().toString();
							bIsSimpleDataType = testAttrValueModel.isSimple();
							if (_szAttrDataType == null || !_szAttrDataType.equals( szDataType))
							{
								return( false);
							}
							else if (bIsSimpleDataType != _bIsSimpleDataType)
							{
								return( false);
							}
						}
					}
				}
				catch (Exception e)
				{
					return( false);
				}
			}
		}
			
		// See if all of this attribute's values were matched
		
		if (iTestAttrValueCount != _attrValueList.size())
		{
			return( false);
		}
		for (int iLoop = 0; iLoop < _attrValueList.size(); iLoop++)
		{
			if (matchedValuesMap.get( "" + iLoop) == null)
			{
				return( false);
			}
		}
		return( true);
	}
	
	int findAttrValue(
		SharedAttributeValue	attrValue) throws IdASException
	{
		for (int iLoop = 0; iLoop < _attrValueList.size(); iLoop++)
		{
			if (attrValue.equals( (SharedAttributeValue)_attrValueList.get( iLoop)))
			{
				return( iLoop);
			}
		}
		return( -1);
	}

	int findAttrValue(
		IAttributeValue	attrValue) throws IdASException
	{
		SharedAttributeValue	testValue;
		
		for (int iLoop = 0; iLoop < _attrValueList.size(); iLoop++)
		{
			testValue = (SharedAttributeValue)_attrValueList.get( iLoop);
			
			if (testValue.equals( attrValue))
			{
				return( iLoop);
			}
		}
		return( -1);
	}
}

