/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.higgins.idas.spi.BasicFilter;
import org.eclipse.higgins.idas.spi.BasicFilterAssertion;
import org.eclipse.higgins.idas.spi.BasicFilterAttributeAssertion;
import org.eclipse.higgins.idas.spi.BasicValueBase64Binary;

import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.api.AuthenticationException;

import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.common.AuthNSelfIssuedMaterials;
import org.eclipse.higgins.idas.common.AuthNDigestMaterials;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;

/**
 * 
 * @author dsanders@novell.com
 */

class SharedContext extends Thread
{
	public static final String		XML_CONTEXT_ELEMENT = "Context";
	public static final String		XML_ENTITY_ELEMENT = "Entity";
	public static final String		XML_ATTRIBUTE_ELEMENT = "Attribute";
	public static final String		XML_ATTRIBUTE_VALUE_ELEMENT = "AttributeValue";
	
	public static final String		XML_ATTR_ID_ATTR = "AttrID";
	public static final String		XML_ENTITY_TYPE_ATTR = "EntityType";
	public static final String		XML_DATA_TYPE_ATTR = "DataType";
	
	private Log							_log = LogFactory.getLog( SharedContext.class.getName());
	private int							_iOpenCount;
	private long						_lLastCloseTime;
	private java.util.Map			_entityMap;
	private java.util.ArrayList	_entityList;
	private String						_szFileName;
	private java.io.File				_file;
	private long						_lLastModifiedTime;
	private boolean					_bLocked;
	private String						_szUserNameAttr;
	private String						_szPasswordAttr;
	private String						_szCardKeyHashAttr;
	
	// Constructor that is called by the nonsharedContextFactory.
	SharedContext(
		String	szFileName) throws IdASException
	{
		String			szErrMsg;
		
		_bLocked = false;
		_file = new java.io.File( szFileName);
		try
		{
			_szFileName = _file.getCanonicalPath();
		}
		catch (java.io.IOException e)
		{
			szErrMsg = "Error getting canonical path for file '" + szFileName + "': " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		_initPresetAttrs();
		_entityMap = new java.util.HashMap();
		_entityList = new java.util.ArrayList();
		
		if (!_file.exists())
		{
			_createFile();
		}
		else if (_file.isDirectory())
		{
			szErrMsg = "Specified XML file '" + _szFileName + " is a directory, cannot open";
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		else
		{
			_parseFileToDOM();
		}
	}
	
	private void _initPresetAttrs()
	{
		_szUserNameAttr = AuthNNamePasswordMaterials.ATTR_TYPE_USERNAME.toString();
		_szPasswordAttr = AuthNNamePasswordMaterials.ATTR_MANAGED_URI.toString();
		_szCardKeyHashAttr = AuthNSelfIssuedMaterials.MANAGED_ATTR_URI.toString();
	}
	
	// This constructor is called by nonsharedContext to create a private context for update.
	SharedContext(
		SharedContext	copyContext) throws IdASException
	{
		java.util.ArrayList	copyEntityList = copyContext.getEntityList();
		
		_bLocked = false;
		_szFileName = copyContext.getFileName();
		_file = new java.io.File( _szFileName);
		_initPresetAttrs();		
		_lLastModifiedTime = copyContext.getLastModifiedTime();
		_entityMap = new java.util.HashMap();
		_entityList = new java.util.ArrayList();
		
		for (int iLoop = 0; iLoop < copyEntityList.size(); iLoop++)
		{
			copyEntity( (SharedEntity)copyEntityList.get( iLoop));
		}
	}
	
	// This moves the data from one context to another.  This is only done
	// after an update context does the applyUpdates.  It then moves the data
	// from the update context to the read-only context which is shared
	// by everyone.  This should only be called while the context is locked.
	void setDataFromContext(
		SharedContext	srcContext) throws IdASException
	{
		_entityList = srcContext.getEntityList();
		_entityMap = srcContext.getEntityMap();
		_lLastModifiedTime = srcContext.getLastModifiedTime();
	}
	
	void copyEntity(
		SharedEntity	copyEntity) throws IdASException
	{
		SharedEntity		sharedEntity;
		java.util.ArrayList		attrList = copyEntity.getAttrList();
		String						szEntityID = copyEntity.getEntityID();
		
		sharedEntity = new SharedEntity( this, copyEntity.getEntityType(), szEntityID);
		_entityMap.put( szEntityID, sharedEntity);
		_entityList.add( sharedEntity);
		for (int iLoop = 0; iLoop < attrList.size(); iLoop++)
		{
			sharedEntity.copyAttr( (SharedAttribute)attrList.get( iLoop));
		}
	}
	
	// Methods used by the copy constructor
	
	String getFileName()
	{
		return( _szFileName);
	}
	
	java.util.ArrayList getEntityList()
	{
		return( _entityList);
	}
	
	java.util.Map getEntityMap()
	{
		return( _entityMap);
	}
	
	int getOpenCount()
	{
		return( _iOpenCount);
	}
	
	void setOpenCount(
		int	iOpenCount)
	{
		_iOpenCount = iOpenCount;
	}
	
	void incrOpenCount()
	{
		_iOpenCount++;
	}
	
	void decrOpenCount()
	{
		_iOpenCount--;
	}
	
	long getLastCloseTime()
	{
		return( _lLastCloseTime);
	}
	
	void setLastCloseTime(
		long	lLastCloseTime)
	{
		_lLastCloseTime = lLastCloseTime;
	}
	
	/**
	 */
	private void _createFile() throws IdASException
	{
		javax.xml.parsers.DocumentBuilder			documentBuilder = null;
		javax.xml.parsers.DocumentBuilderFactory	documentBuilderFactory = null;
		String												szErrMsg;
		org.w3c.dom.Document								doc;
		org.w3c.dom.Element								rootElement;
		
		try
		{
			documentBuilderFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
			documentBuilderFactory.setNamespaceAware( true);
			documentBuilder = documentBuilderFactory.newDocumentBuilder();
			doc = documentBuilder.newDocument();
			rootElement = doc.createElement( XML_CONTEXT_ELEMENT);
			doc.appendChild( rootElement);
		}
		catch (Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " creating XML DOM: " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		_writeXMLDocument( doc);
		_lLastModifiedTime = _file.lastModified();
	}
	
	/**
	 */
	private void _writeXMLDocument(
		org.w3c.dom.Document	doc) throws IdASException
	{
		org.dom4j.io.XMLWriter		writer;
		java.io.FileWriter			fileWriter;
		java.io.File					fileName;
		java.io.File					fileDir;
		java.io.File					renameFileTo;
		java.io.File					tmpFileName;
		String							szTmpFileName = null;
		String							szRenameFileTo = null;
		org.dom4j.io.OutputFormat	outputFormat;
		org.dom4j.io.DOMReader		reader;		
		org.dom4j.Document			dom4jDoc;
		String							szErrMsg;
		
		try
		{
			reader = new org.dom4j.io.DOMReader();
			dom4jDoc = reader.read( doc);
		}
		catch (Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " getting DOMReader object: " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		try
		{
		
			// Get File objects for the file and the directory it is in
		
			fileName = new java.io.File( _szFileName);
			fileDir = fileName.getParentFile();
			
			// Get a file name we can rename our current file to temporarily.
			
			renameFileTo = new java.io.File( fileDir, fileName.getName() + ".save");
			szRenameFileTo = renameFileTo.getCanonicalPath();
			
			// Get a temporary file name we can write to in the same directory as the
			// file we want to overwrite.
			
			tmpFileName = java.io.File.createTempFile( "tmp_", ".xml", fileDir);
			szTmpFileName = tmpFileName.getCanonicalPath();
		}
		catch (java.io.IOException e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " getting temporary and/or rename file names: " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		try
		{
			fileWriter = new java.io.FileWriter( szTmpFileName);
			outputFormat = org.dom4j.io.OutputFormat.createPrettyPrint();
			outputFormat.setIndent( "\t");
			writer = new org.dom4j.io.XMLWriter( fileWriter, outputFormat);
			writer.write( dom4jDoc);
			writer.close();
		}
		catch (java.lang.Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " writing DOM to file '" + szTmpFileName + "': " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
			
		// Delete the rename file if it exists.
		
		try
		{
			if (renameFileTo.exists())
			{
				renameFileTo.delete();
			}
		}
		catch (java.lang.Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " deleting old version of XML file '" + szRenameFileTo + "': " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
			
		// Rename the current file to the rename file.

		try
		{			
			if (fileName.exists())
			{
				if (!fileName.renameTo( renameFileTo))
				{
					szErrMsg = "Could not rename '" + _szFileName + "' to '" + szRenameFileTo + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
			}
		}
		catch (IdASException e1)
		{
			throw e1;
		}
		catch (java.lang.SecurityException e2)
		{
			szErrMsg = "Exception " + e2.getClass().getName() + " renaming '" + _szFileName + "' to '" + szRenameFileTo + "': " + e2.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
			
		// Rename the temporary file to the current file.
		
		try
		{
			if (!tmpFileName.renameTo( fileName))
			{
				// Rename the file back, if possible
				
				renameFileTo.renameTo( fileName);
				throw new IdASException( "Could not rename '" + _szFileName + "' to '" + _szFileName + "'");
			}
		}
		catch (IdASException e1)
		{
			throw e1;
		}
		catch (java.lang.SecurityException e3)
		{
			szErrMsg = "Exception " + e3.getClass().getName() + " renaming '" + szTmpFileName + "' to '" + _szFileName + "': " + e3.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
	}

	/**
	 */
	private String _getNodeText(
		org.w3c.dom.Node	domNode)
	{
		org.w3c.dom.Node	textDomNode = domNode.getFirstChild();
		
		while (textDomNode != null && textDomNode.getNodeType() != org.w3c.dom.Node.TEXT_NODE)
		{
			textDomNode = textDomNode.getNextSibling();
		}
		if (textDomNode != null)
		{
			return( textDomNode.getNodeValue());
		}
		else
		{
			return( null);
		}
	}
	
	/**
	 */
	private boolean _isElementNode(
		org.w3c.dom.Node	domNode,
		String				szElementName)
	{
		String	szNodeElementName;
		
		if (domNode.getNodeType() == org.w3c.dom.Node.ELEMENT_NODE)
		{
			szNodeElementName = domNode.getLocalName();
			if (szNodeElementName != null && szNodeElementName.equals( szElementName))
			{
				return( true);
			}
		}
		return( false);
	}
	
	/**
	 */
	private String _getElementAttr(
		org.w3c.dom.Element	element,
		String					szXmlAttrName) throws IdASException
	{
		org.w3c.dom.Attr	domAttr;
		String				szAttrValue = null;

		try
		{
			if ((domAttr = element.getAttributeNode( szXmlAttrName)) != null)
			{
				szAttrValue = domAttr.getValue();
			}
		}
		catch (Exception e)
		{
			throw new IdASException( "Error getting XML attribute '" + szXmlAttrName + "' for element: " + e.getMessage());
		}
		return( szAttrValue);
	}
	
	/**
	 */
	private SharedAttributeValue _parseComplexAttributeValue(
		SharedAttribute		sharedAttr,
		org.w3c.dom.Node		attrValueNode,
		String					szAttrID) throws IdASException
	{
		org.w3c.dom.Node			attrNode;
		SharedAttribute			valueSharedAttr;
		SharedComplexAttrValue	complexValue = new SharedComplexAttrValue( sharedAttr);
		
		attrNode = attrValueNode.getFirstChild();
		while (attrNode != null)
		{
			if (_isElementNode( attrNode, XML_ATTRIBUTE_ELEMENT))
			{
				valueSharedAttr = _parseAttribute( complexValue, attrNode);
				complexValue.addAttr( valueSharedAttr);
			}
			attrNode = attrNode.getNextSibling();
		}
		return( complexValue);
	}
	
	/**
	 */
	private SharedAttribute _parseAttribute(
		SharedAttrContainer	sharedContainer,
		org.w3c.dom.Node		attrNode) throws IdASException
	{
		String						szErrMsg;
		org.w3c.dom.Node			attrValueNode;
		SharedAttributeValue		sharedAttrValue;
		String						szAttrID = _getElementAttr( (org.w3c.dom.Element)attrNode, XML_ATTR_ID_ATTR);
		String						szAttrDataType = _getElementAttr( (org.w3c.dom.Element)attrNode, XML_DATA_TYPE_ATTR);
		boolean						bIsSimpleDataType = false;
		SharedAttribute			sharedAttr;
		
		// Must at least have an attribute ID.
		
		if (szAttrID == null)
		{
			szErrMsg = "No " + XML_ATTR_ID_ATTR + " attribute defined for attribute element in file " + _szFileName + "'";
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		if (szAttrDataType != null)
		{
			bIsSimpleDataType = NonsharedContextFactory.isSimpleDataType( szAttrDataType);
		}
		sharedAttr = new SharedAttribute( sharedContainer, szAttrID, szAttrDataType, bIsSimpleDataType);
		
		// Get the value elements.
		
		attrValueNode = attrNode.getFirstChild();
		while (attrValueNode != null)
		{
			if (_isElementNode( attrValueNode, XML_ATTRIBUTE_VALUE_ELEMENT))
			{
				if (szAttrDataType == null)
				{
					szErrMsg = "Type not set for attribute '" + szAttrID + "', type must be set when attribute has values, file: '" + _szFileName + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				sharedAttrValue = (bIsSimpleDataType)
										? new SharedSimpleAttrValue( sharedAttr, _getNodeText( attrValueNode))
										: _parseComplexAttributeValue( sharedAttr, attrValueNode, szAttrID);
				
				// See if the value is already defined
				
				if (sharedAttr.findAttrValue( sharedAttrValue) != -1)
				{
					szErrMsg = "Value for attribute '" + szAttrID + "' defined multiple times on entity in file '" + _szFileName + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				sharedAttr.addValueToList( sharedAttrValue);
			}
			attrValueNode = attrValueNode.getNextSibling();
		}
		return( sharedAttr);
	}
	
	/**
	 */
	private SharedEntity _parseEntity(
		org.w3c.dom.Node		nodeDOMNode,
		String					szUserNameAttr) throws IdASException
	{
		String						szErrMsg;
		org.w3c.dom.Node			attrNode;
		SharedEntity				sharedEntity;
		SharedAttribute			sharedAttr;
		String						szEntityID = null;
		String						szEntityType;
		String						szAttrID;
		
		// Must have a type attribute on the entity
		
		if ((szEntityType = _getElementAttr( (org.w3c.dom.Element)nodeDOMNode, XML_ENTITY_TYPE_ATTR)) == null)
		{
			szErrMsg = "Entity type XML attribute '" + XML_ENTITY_TYPE_ATTR + "' missing on XML element file " + _szFileName + "'";
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		sharedEntity = new SharedEntity( this, szEntityType, null);		
		
		attrNode = nodeDOMNode.getFirstChild();
		while (attrNode != null)
		{
			if (_isElementNode( attrNode, XML_ATTRIBUTE_ELEMENT))
			{
				sharedAttr = _parseAttribute( sharedEntity, attrNode);
				sharedEntity.addAttr( sharedAttr);
				szAttrID = sharedAttr.getAttrID();
				
				// If the attribute is the user name attribute, get its value as the entity ID
				
				if (szAttrID.equals( _szUserNameAttr))
				{
					java.util.ArrayList	valueList = sharedAttr.getValueList();
					
					if (sharedAttr.getAttrDataType() == null || valueList.size() == 0)
					{
						szErrMsg = "No value defined for entity ID attribute '" + szAttrID + "' on Entity in file " + _szFileName + "'";
						_log.debug( szErrMsg);
						throw new IdASException( szErrMsg);
					}
					if (!sharedAttr.isSimpleDataType())
					{
						szErrMsg = "Value for entity ID attribute '" + szAttrID + "' on Entity in file " + _szFileName + "' is NOT a simple type";
						_log.debug( szErrMsg);
						throw new IdASException( szErrMsg);
					}
					if (valueList.size() > 1)
					{
						szErrMsg = "Multiple value for entity ID attribute '" + szAttrID + "' on Entity in file " + _szFileName + "'";
						_log.debug( szErrMsg);
						throw new IdASException( szErrMsg);
					}
					szEntityID = ((SharedSimpleAttrValue)valueList.get( 0)).getCanonical();
				}
			}
			attrNode = attrNode.getNextSibling();
		}
		
		// Make sure we got a entity ID attribute
		
		if (szEntityID == null)
		{
			szErrMsg = "No " + _szUserNameAttr + " defined for Entity in file " + _szFileName + "'";
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		sharedEntity.setEntityID( szEntityID);
		return( sharedEntity);
	}
	
	/**
	 */
	private void _parseFileToDOM() throws IdASException
	{
		java.io.FileInputStream							inputStream;
		javax.xml.parsers.DocumentBuilder			documentBuilder;
		javax.xml.parsers.DocumentBuilderFactory	documentBuilderFactory;
		String												szErrMsg;
		org.w3c.dom.Document								doc;
		org.w3c.dom.Element								root;
		org.w3c.dom.Node									node;
		java.util.HashMap									entityMap = new java.util.HashMap();
		java.util.ArrayList								entityList = new java.util.ArrayList();
		SharedEntity										sharedEntity;
		String												szEntityID;
		
		// First, get the document into DOM format
	
		try
		{
			inputStream = new java.io.FileInputStream( new java.io.File( _szFileName));
			documentBuilderFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
			documentBuilderFactory.setNamespaceAware( true);
			documentBuilder = documentBuilderFactory.newDocumentBuilder();
			doc = documentBuilder.parse( inputStream);
			root = doc.getDocumentElement();
		}
		catch (Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " parsing DOM from file '" + _szFileName + "': " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		// Now make a second pass to get the entities
		
		node = root.getFirstChild();
		while (node != null)
		{
			if (_isElementNode( node, XML_ENTITY_ELEMENT))
			{
				sharedEntity = _parseEntity( node, _szUserNameAttr);
				
				// See if the entity is already defined.
				
				szEntityID = sharedEntity.getEntityID();
				if (entityMap.get( szEntityID) != null)
				{
					szErrMsg = "Entity '" + szEntityID + "' defined multiple times in file '" + _szFileName + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				entityMap.put( szEntityID, sharedEntity);
				entityList.add( sharedEntity);
			}
			node = node.getNextSibling();
		}
		
		// If we were successful, save things into the member variables.
		
		_entityMap = entityMap;
		_entityList = entityList;
		_lLastModifiedTime = _file.lastModified();
	}
	
	public void close()
	{
		_entityMap = null;
		_entityList = null;
		_szFileName = null;
		_file = null;
		_szUserNameAttr = null;
		_szPasswordAttr = null;
		_szCardKeyHashAttr = null;
		_log = null;
	}
	
	public long getLastModifiedTime()
	{
		return( _lLastModifiedTime);
	}
	
	boolean syncFile() throws IdASException
	{
		if (_lLastModifiedTime != _file.lastModified())
		{
			if (!_file.exists())
			{
				_createFile();
			}
			else if (_file.isDirectory())
			{
				String	szErrMsg = "Specified XML file '" + _szFileName + " is a directory, cannot open";
				_log.debug( szErrMsg);
				throw new IdASException( szErrMsg);
			}
			else
			{
				_parseFileToDOM();
			}
			return( true);
		}
		return( false);
	}
	
	public synchronized void lock()
	{
		while (_bLocked)
		{
			try
			{
				wait();
			}
			catch (Exception e)
			{
			}
		}
		_bLocked = true;
	}
	
	public synchronized void unlock()
	{
		_bLocked = false;
		notify();
	}
	
	private int compareBigDecimalToLong(
		java.math.BigDecimal	value1,
		long						value2)
	{
		return( value1.compareTo( java.math.BigDecimal.valueOf( value2)));
	}
	
	private int compareBigDecimalToDouble(
		java.math.BigDecimal	value1,
		double					value2)
	{
		return( value1.compareTo( new java.math.BigDecimal( value2)));
	}
	
	private int compareBigDecimalToBigInteger(
		java.math.BigDecimal	value1,
		java.math.BigInteger	value2)
	{
		return( value1.compareTo( new java.math.BigDecimal( value2)));
	}
	
	private int compareBigIntegerToLong(
		java.math.BigInteger	value1,
		long						value2)
	{
		return( value1.compareTo( java.math.BigInteger.valueOf( value2)));
	}
	
	private int compareBigIntegerToDouble(
		java.math.BigInteger	value1,
		double					value2)
	{
		java.math.BigDecimal	v1 = new java.math.BigDecimal( value1);
		
		return( v1.compareTo( new java.math.BigDecimal( value2)));
	}
	
	private int compareBigIntegerToBigDecimal(
		java.math.BigInteger	value1,
		java.math.BigDecimal	value2)
	{
		java.math.BigDecimal	v1 = new java.math.BigDecimal( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareLongToBigDecimal(
		long						value1,
		java.math.BigDecimal	value2)
	{
		java.math.BigDecimal	v1 = java.math.BigDecimal.valueOf( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareDoubleToBigDecimal(
		double					value1,
		java.math.BigDecimal	value2)
	{
		java.math.BigDecimal	v1 = new java.math.BigDecimal( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareDoubleToBigInteger(
		double					value1,
		java.math.BigInteger	value2)
	{
		java.math.BigDecimal	v1 = new java.math.BigDecimal( value1);
		
		return( v1.compareTo( new java.math.BigDecimal( value2)));
	}
	
	private int compareLongToBigInteger(
		long						value1,
		java.math.BigInteger	value2)
	{
		java.math.BigInteger	v1 = java.math.BigInteger.valueOf( value1);
		
		return( v1.compareTo( value2));
	}
	
	int compareDoubles(
		double	value1,
		double	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	int compareFloats(
		float	value1,
		float	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	int compareLongs(
		long	value1,
		long	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	int compareIntegers(
		int	value1,
		int	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	int compareShorts(
		short	value1,
		short	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	int compareBytes(
		byte	value1,
		byte	value2)
	{
		if (value1 == value2)
		{
			return( 0);
		}
		return( (value1 < value2) ? -1 : 1);
	}
	
	private java.util.Date _stringToDate(
		String szDate) throws IdASException
	{
		java.text.SimpleDateFormat	sdf = new java.text.SimpleDateFormat();
		String[]							formats = new String[]
													{
													"yyyy-MM-dd'T'HH:mm:ss.S'Z'",
													"yyyy-MM-dd'T'HH:mm:ss'Z'",
													"yyyy-MM-dd'T'HH:mm:ss.SZ",							
													"yyyy-MM-dd'T'HH:mm:ssZ"
													};
		java.util.Date					date = null;
		int								iFormat;
		
		for (iFormat = 0; iFormat < formats.length; iFormat++)
		{
			try
			{
				sdf.applyPattern( formats[iFormat]);
				date = sdf.parse( szDate);
			}
			catch(java.text.ParseException e)
			{
				// try next pattern
				date = null;
				continue;
			}
			break;
		}
		if (date == null)
		{
			throw new IdASException( "Unable to parse date value: " + szDate);
		}
		return( date);
	}
	
	private int compareDateToString(
		java.util.Date	value1,
		String			value2) throws Exception
	{
		return( value1.compareTo( _stringToDate( value2)));
	}
	
	private int compareStringToDate(
		String			value1,
		java.util.Date	value2) throws Exception
	{
		java.util.Date	v1 = _stringToDate( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareDateToLong(
		java.util.Date	value1,
		long				value2)
	{
		return( value1.compareTo( new java.util.Date( value2)));
	}
	
	private int compareLongToDate(
		long				value1,
		java.util.Date	value2)
	{
		java.util.Date	v1 = new java.util.Date( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareStringToUri(
		String			value1,
		java.net.URI	value2) throws Exception
	{
		java.net.URI	v1 = new java.net.URI( value1);
		
		return( v1.compareTo( value2));
	}
	
	private int compareUriToString(
		java.net.URI	value1,
		String			value2) throws Exception
	{
		return( value1.compareTo( new java.net.URI( value2)));
	}
	
	/**
	 */
	int compareSimpleObjects(
		Object	obj1,
		Object	obj2) throws IdASException
	{
		String	szErrMsg;
		
		try
		{
			if (obj1 instanceof String)
			{
				if (obj2 instanceof String)
				{
					return( ((String)obj1).compareTo( (String)obj2));
				}
				else if (obj2 instanceof Boolean)
				{
					return( ((String)obj1).compareTo( ((Boolean)obj2).toString()));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( ((String)obj1).compareTo( ((java.math.BigDecimal)obj2).toString()));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( ((String)obj1).compareTo( ((java.math.BigInteger)obj2).toString()));
				}
				else if (obj2 instanceof Float)
				{
					return( ((String)obj1).compareTo( ((Float)obj2).toString()));
				}
				else if (obj2 instanceof Double)
				{
					return( ((String)obj1).compareTo( ((Double)obj2).toString()));
				}
				else if (obj2 instanceof Long)
				{
					return( ((String)obj1).compareTo( ((Long)obj2).toString()));
				}
				else if (obj2 instanceof Integer)
				{
					return( ((String)obj1).compareTo( ((Integer)obj2).toString()));
				}
				else if (obj2 instanceof Short)
				{
					return( ((String)obj1).compareTo( ((Short)obj2).toString()));
				}
				else if (obj2 instanceof Byte)
				{
					return( ((String)obj1).compareTo( ((Byte)obj2).toString()));
				}
				else if (obj2 instanceof java.nio.ByteBuffer)
				{
					return( ((String)obj1).compareTo( ((java.nio.ByteBuffer)obj2).toString()));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( compareStringToDate( (String)obj1, (java.util.Date)obj2));
				}
				else if (obj2 instanceof java.net.URI)
				{
					return( compareStringToUri( (String)obj1, (java.net.URI)obj2));
				}
			}
			else if (obj1 instanceof Boolean)
			{
				if (obj2 instanceof String)
				{
					return( ((Boolean)obj1).toString().compareTo( (String)obj2));
				}
				else if (obj2 instanceof Boolean)
				{
					boolean	b1 = ((Boolean)obj1).booleanValue();
					boolean	b2 = ((Boolean)obj2).booleanValue();
					
					if (b1 == b2)
					{
						return( 0);
					}
					else
					{
						return( b1 ? 1 : -1);
					}
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareLongToBigDecimal( ((Boolean)obj1).booleanValue() ? (long)1 : (long)0, (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareLongToBigInteger( ((Boolean)obj1).booleanValue() ? (long)1 : (long)0, (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareFloats( ((Boolean)obj1).booleanValue() ? (float)1 : (float)0, ((Float)obj2).floatValue()));  
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( ((Boolean)obj1).booleanValue() ? (double)1 : (double)0, ((Double)obj2).doubleValue()));  
				}
				else if (obj2 instanceof Long)
				{
					return( compareLongs( ((Boolean)obj1).booleanValue() ? (long)1 : (long)0, ((Long)obj2).longValue()));  
				}
				else if (obj2 instanceof Integer)
				{
					return( compareIntegers( ((Boolean)obj1).booleanValue() ? (int)1 : (int)0, ((Integer)obj2).intValue()));  
				}
				else if (obj2 instanceof Short)
				{
					return( compareIntegers( ((Boolean)obj1).booleanValue() ? (short)1 : (short)0, ((Short)obj2).shortValue()));  
				}
				else if (obj2 instanceof Byte)
				{
					return( compareBytes( ((Boolean)obj1).booleanValue() ? (byte)1 : (byte)0, ((Byte)obj2).byteValue()));  
				}
			}
			else if (obj1 instanceof java.math.BigDecimal)
			{
				if (obj2 instanceof String)
				{
					return( ((java.math.BigDecimal)obj1).compareTo( new java.math.BigDecimal( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					compareBigDecimalToLong( (java.math.BigDecimal)obj1, ((Boolean)obj2).booleanValue() ? (long)1 : (long)0);
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( ((java.math.BigDecimal)obj1).compareTo( (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareBigDecimalToBigInteger( (java.math.BigDecimal)obj1, (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareBigDecimalToDouble( (java.math.BigDecimal)obj1, (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareBigDecimalToDouble( (java.math.BigDecimal)obj1, ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareBigDecimalToLong( (java.math.BigDecimal)obj1, ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareBigDecimalToLong( (java.math.BigDecimal)obj1, (long)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareBigDecimalToLong( (java.math.BigDecimal)obj1, (long)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareBigDecimalToLong( (java.math.BigDecimal)obj1, (long)(((Byte)obj2).byteValue())));
				}
			}
			else if (obj1 instanceof java.math.BigInteger)
			{
				if (obj2 instanceof String)
				{
					return( ((java.math.BigInteger)obj1).compareTo( new java.math.BigInteger( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareBigIntegerToLong( (java.math.BigInteger)obj1, ((Boolean)obj2).booleanValue() ? (long)1 : (long)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareBigIntegerToBigDecimal( (java.math.BigInteger)obj1, (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( ((java.math.BigInteger)obj1).compareTo( (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareBigIntegerToDouble( (java.math.BigInteger)obj1, (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareBigIntegerToDouble( (java.math.BigInteger)obj1, ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareBigIntegerToLong( (java.math.BigInteger)obj1, ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareBigIntegerToLong( (java.math.BigInteger)obj1, (long)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareBigIntegerToLong( (java.math.BigInteger)obj1, (long)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareBigIntegerToLong( (java.math.BigInteger)obj1, (long)(((Byte)obj2).byteValue())));
				}
			}
			else if (obj1 instanceof Float)
			{
				if (obj2 instanceof String)
				{
					return( ((Float)obj1).compareTo( new Float( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareFloats( ((Float)obj1).floatValue(), ((Boolean)obj2).booleanValue() ? (float)1 : (float)0)); 
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareDoubleToBigDecimal( (double)(((Float)obj1).floatValue()), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareDoubleToBigInteger( (double)(((Float)obj1).floatValue()), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareFloats( ((Float)obj1).floatValue(), ((Float)obj2).floatValue()));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( (double)(((Float)obj1).floatValue()), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareDoubles( (double)(((Float)obj1).floatValue()), (double)(((Long)obj2).longValue())));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareDoubles( (double)(((Float)obj1).floatValue()), (double)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareDoubles( (double)(((Float)obj1).floatValue()), (double)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareDoubles( (double)(((Float)obj1).floatValue()), (double)(((Byte)obj2).byteValue())));
				}
			}
			else if (obj1 instanceof Double)
			{
				if (obj2 instanceof String)
				{
					return( ((Double)obj1).compareTo( new Double( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), ((Boolean)obj2).booleanValue() ? (double)1 : (double)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareDoubleToBigDecimal( ((Double)obj1).doubleValue(), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareDoubleToBigInteger( ((Double)obj1).doubleValue(), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), (double)(((Long)obj2).longValue())));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), (double)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), (double)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareDoubles( ((Double)obj1).doubleValue(), (double)(((Byte)obj2).byteValue())));
				}
			}
			else if (obj1 instanceof Long)
			{
				if (obj2 instanceof String)
				{
					return( ((Long)obj1).compareTo( new Long( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareLongs( ((Long)obj1).longValue(), ((Boolean)obj2).booleanValue() ? (long)1 : (long)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareLongToBigDecimal( ((Long)obj1).longValue(), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareLongToBigInteger( ((Long)obj1).longValue(), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareDoubles( (double)(((Long)obj1).longValue()), (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( (double)(((Long)obj1).longValue()), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareLongs( ((Long)obj1).longValue(), ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareLongs( ((Long)obj1).longValue(), (long)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareLongs( ((Long)obj1).longValue(), (long)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareLongs( ((Long)obj1).longValue(), (long)(((Byte)obj2).byteValue())));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( compareLongToDate( ((Long)obj1).longValue(), (java.util.Date)obj2));
				}
			}
			else if (obj1 instanceof Integer)
			{
				if (obj2 instanceof String)
				{
					return( ((Integer)obj1).compareTo( new Integer( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareIntegers( ((Integer)obj1).intValue(), ((Boolean)obj2).booleanValue() ? (int)1 : (int)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareLongToBigDecimal( (long)(((Integer)obj1).intValue()), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareLongToBigInteger( (long)(((Integer)obj1).intValue()), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareDoubles( (double)(((Integer)obj1).intValue()), (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( (double)(((Integer)obj1).intValue()), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareLongs( (long)(((Integer)obj1).intValue()), ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareIntegers( ((Integer)obj1).intValue(), ((Integer)obj2).intValue()));
				}
				else if (obj2 instanceof Short)
				{
					return( compareIntegers( ((Integer)obj1).intValue(), (int)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareIntegers( ((Integer)obj1).intValue(), (int)(((Byte)obj2).byteValue())));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( compareLongToDate( (long)(((Integer)obj1).intValue()), (java.util.Date)obj2));
				}
			}
			else if (obj1 instanceof Short)
			{
				if (obj2 instanceof String)
				{
					return( ((Short)obj1).compareTo( new Short( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareShorts( ((Short)obj1).shortValue(), ((Boolean)obj2).booleanValue() ? (short)1 : (short)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareLongToBigDecimal( (long)(((Short)obj1).shortValue()), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareLongToBigInteger( (long)(((Short)obj1).shortValue()), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareDoubles( (double)(((Short)obj1).shortValue()), (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( (double)(((Short)obj1).shortValue()), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareLongs( (long)(((Short)obj1).shortValue()), ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareIntegers( (int)(((Short)obj1).shortValue()), ((Integer)obj2).intValue()));
				}
				else if (obj2 instanceof Short)
				{
					return( compareShorts( ((Short)obj1).shortValue(), ((Short)obj2).shortValue()));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareShorts( ((Short)obj1).shortValue(), (short)(((Byte)obj2).byteValue())));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( compareLongToDate( (long)(((Short)obj1).shortValue()), (java.util.Date)obj2));
				}
			}
			else if (obj1 instanceof Byte)
			{
				if (obj2 instanceof String)
				{
					return( ((Byte)obj1).compareTo( new Byte( (String)obj2)));
				}
				else if (obj2 instanceof Boolean)
				{
					return( compareBytes( ((Byte)obj1).byteValue(), ((Boolean)obj2).booleanValue() ? (byte)1 : (byte)0));
				}
				else if (obj2 instanceof java.math.BigDecimal)
				{
					return( compareLongToBigDecimal( (long)(((Byte)obj1).byteValue()), (java.math.BigDecimal)obj2));
				}
				else if (obj2 instanceof java.math.BigInteger)
				{
					return( compareLongToBigInteger( (long)(((Byte)obj1).byteValue()), (java.math.BigInteger)obj2));
				}
				else if (obj2 instanceof Float)
				{
					return( compareDoubles( (double)(((Byte)obj1).byteValue()), (double)(((Float)obj2).floatValue())));
				}
				else if (obj2 instanceof Double)
				{
					return( compareDoubles( (double)(((Byte)obj1).byteValue()), ((Double)obj2).doubleValue()));
				}
				else if (obj2 instanceof Long)
				{
					return( compareLongs( (long)(((Byte)obj1).byteValue()), ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareIntegers( (int)(((Byte)obj1).byteValue()), ((Integer)obj2).intValue()));
				}
				else if (obj2 instanceof Short)
				{
					return( compareShorts( (short)(((Byte)obj1).byteValue()), ((Short)obj2).shortValue()));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareBytes( ((Byte)obj1).byteValue(), ((Byte)obj2).byteValue()));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( compareLongToDate( (long)(((Byte)obj1).byteValue()), (java.util.Date)obj2));
				}
			}
			else if (obj1 instanceof java.nio.ByteBuffer)
			{
				if (obj2 instanceof java.nio.ByteBuffer)
				{
					return( ((java.nio.ByteBuffer)obj1).compareTo( (java.nio.ByteBuffer)obj2));
				}
			}
			else if (obj1 instanceof java.util.Date)
			{
				if (obj2 instanceof String)
				{
					return( compareDateToString( (java.util.Date)obj1, (String)obj2));
				}
				else if (obj2 instanceof Long)
				{
					return( compareDateToLong( (java.util.Date)obj1, ((Long)obj2).longValue()));
				}
				else if (obj2 instanceof Integer)
				{
					return( compareDateToLong( (java.util.Date)obj1, (long)(((Integer)obj2).intValue())));
				}
				else if (obj2 instanceof Short)
				{
					return( compareDateToLong( (java.util.Date)obj1, (long)(((Short)obj2).shortValue())));
				}
				else if (obj2 instanceof Byte)
				{
					return( compareDateToLong( (java.util.Date)obj1, (long)(((Byte)obj2).byteValue())));
				}
				else if (obj2 instanceof java.util.Date)
				{
					return( ((java.util.Date)obj1).compareTo( (java.util.Date)obj2));
				}
			}
			else if (obj1 instanceof java.net.URI)
			{
				if (obj2 instanceof String)
				{
					return( compareUriToString( (java.net.URI)obj1, (String)obj2));
				}
				else if (obj2 instanceof java.net.URI)
				{
					return( ((java.net.URI)obj1).compareTo( (java.net.URI)obj2));
				}
			}
		}
		catch (Exception e)
		{
			szErrMsg = "Exception " + e.getClass().getName() + " comparing value " +
							obj1.getClass().getName() + "={" + obj1.toString() + "} to value " +
							obj2.getClass().getName() + "={" + obj2.toString() + "}: " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		szErrMsg = "Cannot compare value " +
						obj1.getClass().getName() + "={" + obj1.toString() + "} to value " +
						obj2.getClass().getName() + "={" + obj2.toString() + "}";
		_log.debug( szErrMsg);
		throw new IdASException( szErrMsg);
	}
	
	/**
	 */
	boolean testEntity(
		SharedEntity	sharedEntity,
		IFilter					filter) throws IdASException
	{
		SharedAttribute					sharedAttr;
		int									iFoundIndex;
		SharedAttributeValue				sharedAttrValue;
		java.util.ArrayList				attrList;
		BasicFilterAssertion				assertion;
		String								szAttrID;
		String								szComparator;
		String								szOperator;
		String								szErrMsg;
		BasicFilterAttributeAssertion	attributeAssertion;
		IAttributeValue					assertionValue;
		boolean								bGE;
		boolean								bNotted;
		int									iCmp;
		java.util.Iterator				filterIter;
		boolean								bAndOp;
		boolean								bPassed;
		
		// A null filter means that all of the Entities should pass.
		
		if (filter == null)
		{
			return( true);
		}
		
		// See if there is a NOT operator
		
		szOperator = ((BasicFilter)filter).getOperator();
		if ((assertion = (BasicFilterAssertion)(((BasicFilter)filter).getAssertion())) == null)
		{
			bAndOp = (szOperator != null && szOperator.equals( IFilter.OP_AND)) ? true : false;
			
			filterIter = ((BasicFilter)filter).getFilters();
			while (filterIter.hasNext())
			{
				bPassed = testEntity( sharedEntity, (IFilter)filterIter.next());
				if (!bPassed && bAndOp)
				{
					// No need to evaluate anymore if we are AND the filters and one fails.
					return( false);
				}
				else if (bPassed && !bAndOp)
				{
					// No need to evaluate anymore if we are ORing the filters and one passes.
					return( true);
				}
			}
			return( false);
		}
		bNotted = (szOperator != null && szOperator.equals( IFilter.OP_NOT)) ? true : false;
		szAttrID = assertion.getID().toString();
		sharedAttr = sharedEntity.getAttribute( szAttrID);
		szComparator = assertion.getComparator();
		
		if (!(assertion instanceof BasicFilterAttributeAssertion))
		{
			throw new NotImplementedException(
					"Only attribute assertions supported");
		}
		
		// Assertion is an attribute assertion
		
		attributeAssertion = (BasicFilterAttributeAssertion) assertion;
		if ((assertionValue = attributeAssertion.getAssertionValue()) == null)
		{
			if (!szComparator.equals( IFilterAttributeAssertion.COMP_ATTR_PRESENT))
			{
				szErrMsg = "Comparison operator '" + szComparator + "' not supported when no assertion value is given.";
				_log.debug( szErrMsg);
				throw new NotImplementedException( szErrMsg);
			}
			
			// See if the attribute is present on the Entity.
			
			if (bNotted)
			{
				bPassed = (sharedAttr != null ? false : true);
			}
			else
			{
				bPassed = (sharedAttr != null ? true : false);
			}
			return( bPassed);
		}
		
		// Assertion value is non-null
		
		if (!assertionValue.isSimple())
		{
			throw new NotImplementedException(
				"Complex assertion values in filters not supported.");
		}
		
		// Assertion value is a simple value
		
		if (sharedAttr == null)
		{
			return( false);
		}
		if (sharedAttr.getAttrDataType() == null || !sharedAttr.isSimpleDataType())
		{
			return( false);
		}
		if (szComparator.equals( IFilterAttributeAssertion.COMP_ATTR_EQ))
		{
			iFoundIndex = sharedAttr.findAttrValue( assertionValue);
			if (bNotted)
			{
				return( iFoundIndex == -1 ? true : false);
			}
			else
			{
				return( iFoundIndex != -1 ? true : false);
			}
		}
		if (!szComparator.equals( IFilterAttributeAssertion.COMP_ATTR_GE) &&
			 !szComparator.equals( IFilterAttributeAssertion.COMP_ATTR_LE))
		{
			szErrMsg = "Comparison operator '" + szComparator + "' not supported.";
			_log.debug( szErrMsg);
			throw new NotImplementedException( szErrMsg);
		}
		
		// Operator is >= or <=
		
		bGE = szComparator.equals( IFilterAttributeAssertion.COMP_ATTR_GE);
		attrList = sharedAttr.getValueList();
		for (int iLoop = 0; iLoop < attrList.size(); iLoop++)
		{
			sharedAttrValue = (SharedAttributeValue)attrList.get( iLoop);
			if (((SharedSimpleAttrValue)sharedAttrValue).getData() != null)
			{
				iCmp = compareSimpleObjects( ((SharedSimpleAttrValue)sharedAttrValue).getData(), ((ISimpleAttrValue)assertionValue).getData());
			}
			else
			{
				iCmp = -1;
			}
			if (bNotted)
			{
				
				// ALL must match the criteria in the NOT case, so if we find one that
				// does not match, we return FALSE.
				
				if ((bGE && iCmp < 0) || (!bGE && iCmp > 0))
				{
					return( false);
				}
			}
			else
			{
				if ((bGE && iCmp >= 0) || (!bGE && iCmp <= 0))
				{
					return( true);
				}
			}
		}
		
		// If we get here, we didn't find any value that satisfied the criteria
		return( bNotted ? true : false);
	}
	
	private SharedEntity _findEntity(
		String	szSimpleValue,
		String	szAttrID) throws IdASException
	{
		SharedEntity		sharedEntity;
		SharedAttribute			sharedAttr;
		java.util.ArrayList		valueList;
		SharedSimpleAttrValue	simpleValue;
		
		for (int iLoop = 0; iLoop < _entityList.size(); iLoop++)
		{
			sharedEntity = (SharedEntity)_entityList.get( iLoop);
			
			// See if the entity has the specified attribute
			
			if ((sharedAttr = (SharedAttribute)sharedEntity.getAttrMap().get( szAttrID)) != null)
			{
				if (sharedAttr.getAttrDataType() != null && sharedAttr.isSimpleDataType())
				{
					valueList = sharedAttr.getValueList();
					
					for (int iLoop2 = 0; iLoop2 < valueList.size(); iLoop2++)
					{
						simpleValue = (SharedSimpleAttrValue)valueList.get( iLoop2);
						if (simpleValue.getCanonical() != null && szSimpleValue.equals( simpleValue.getCanonical()))
						{
							return( sharedEntity);
						}
					}
				}
			}
		}
		return( null);
	}
				
	/**
	 */
	String open(
		Object	identity) throws IdASException
	{
		String						szEntityID;
		SharedEntity		sharedEntity;
		String						szErrMsg;
		SharedAttribute			userNameAttr;
		SharedAttribute			passwordAttr;
		java.util.ArrayList		valueList;
		SharedSimpleAttrValue	simpleValue;
		
		if (identity instanceof AuthNAnonymousMaterials)
		{
			szEntityID = "";
		}
		else if (identity instanceof AuthNNamePasswordMaterials)
		{
			szEntityID = ((AuthNNamePasswordMaterials) identity).getUsername();
			if ((sharedEntity = (SharedEntity)_entityMap.get( szEntityID)) == null)
			{
				szErrMsg = "Could not authenticate entity '" + szEntityID + "', entity not found";
				_log.debug( szErrMsg);
				throw new NoSuchEntityException( "Could not authenticate entity '" + szEntityID + "'");
			}
			if ((passwordAttr = sharedEntity.getAttribute( _szPasswordAttr)) == null)
			{
				szErrMsg = "No password attribute '" + _szPasswordAttr + "' found on Entity '" + szEntityID + "'";
				_log.debug( szErrMsg);
				throw new AuthenticationException( "Could not authenticate entity '" + szEntityID + "'");
			}
			if (passwordAttr.getAttrDataType() == null)
			{
				szErrMsg = "Type for password attribute '" + _szPasswordAttr + "' on Entity '" + szEntityID + "' is not known, attribute has no values";
				_log.debug( szErrMsg);
				throw new AuthenticationException( "Could not authenticate entity '" + szEntityID + "'");
			}
			if (!passwordAttr.isSimpleDataType())
			{
				szErrMsg = "Password attribute '" + _szPasswordAttr + "' on Entity '" + szEntityID + "' is not a simple type";
				_log.debug( szErrMsg);
				throw new AuthenticationException( "Could not authenticate entity '" + szEntityID + "'");
			}
			valueList = passwordAttr.getValueList();
			for (int iLoop = 0; iLoop < valueList.size(); iLoop++)
			{
				simpleValue = (SharedSimpleAttrValue)valueList.get( iLoop);
				if (simpleValue.getCanonical() != null && simpleValue.getCanonical().equals(((AuthNNamePasswordMaterials)identity).getPassword()))
				{
					break;
				}
				else if (iLoop == valueList.size() - 1)
				{
					szErrMsg = "Password value does not match password on for entity '" + szEntityID + "'";
					_log.debug( szErrMsg);
					throw new AuthenticationException( "Could not authenticate entity '" + szEntityID + "'");
				}
			}
		}
		else if (identity instanceof AuthNDigestMaterials)
		{
			BasicValueBase64Binary	base64Value = new BasicValueBase64Binary( ((AuthNDigestMaterials)identity).getSHA1Digest(), null);

			if ((sharedEntity = _findEntity( base64Value.getCanonical(), _szCardKeyHashAttr)) == null)
			{
				szErrMsg = "AuthN materials produced no Entity ID, hash='" + base64Value.getCanonical() + "'";
				_log.debug( szErrMsg);
				throw new NoSuchEntityException( szErrMsg);
			}
			else
			{
				if ((userNameAttr = sharedEntity.getAttribute( _szUserNameAttr)) == null)
				{
					szErrMsg = "No user name attribute specified on Entity that was found, hash='" + base64Value.getCanonical() + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				
				if (!userNameAttr.isSimpleDataType())
				{
					szErrMsg = "User name attribute on Entity that was found is not a simple type, hash='" + base64Value.getCanonical() + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				if (userNameAttr.getValueList().size() != 1)
				{
					szErrMsg = "User name attribute on Entity that was found is not single valued, hash='" + base64Value.getCanonical() + "'";
					_log.debug( szErrMsg);
					throw new IdASException( szErrMsg);
				}
				szEntityID = ((SharedSimpleAttrValue)userNameAttr.getValueList().get( 0)).getCanonical();
			}
		}
		else
		{
			szErrMsg = "Unsupported AuthN materials: " + identity.toString();
			_log.debug( szErrMsg);
			throw new NotImplementedException( szErrMsg);
		}
		return( szEntityID);
	}
	
	SharedEntity getEntity(
		String	szEntityID) throws IdASException
	{
		SharedEntity	sharedEntity = (SharedEntity)_entityMap.get( szEntityID);
		if (sharedEntity == null)
		{
			String	szErrMsg = "sharedContext:getEntity(entityID): No such entity '" + szEntityID + "'";
			_log.debug( szErrMsg);
			throw new NoSuchEntityException( szErrMsg);
		}
		return( sharedEntity);
	}
	
	void removeEntity(
		String	szEntityID) throws IdASException
	{
		String					szErrMsg;
		SharedEntity	sharedEntity;
		
		try
		{
			if ((sharedEntity = (SharedEntity)_entityMap.remove( szEntityID)) == null)
			{
				szErrMsg = "sharedContext:removeEntity(entityID): No such entity '" + szEntityID + "'";
				_log.debug( szErrMsg);
				throw new NoSuchEntityException( szErrMsg);
			}
			else
			{
				
				// Find and remove the entity in the list
				
				sharedEntity.setDeleted();
				for (int iLoop = 0; iLoop < _entityList.size(); iLoop++)
				{
					if (sharedEntity == (SharedEntity)_entityList.get( iLoop))
					{
						_entityList.remove( iLoop);
						break;
					}
				}
			}
		}
		catch (Exception e)
		{
			szErrMsg = "sharedContext:removeEntity(entityID): Exception " + e.getClass().getName() + " removing Entity '" + szEntityID + "': " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
	}
	
	SharedEntity addEntity(
		java.net.URI	type,
		String			szEntityID) throws IdASException
	{
		SharedEntity		sharedEntity;
		SharedAttribute			sharedAttr;
		String						szErrMsg;
		
		if ((sharedEntity = (SharedEntity)_entityMap.get( szEntityID)) != null)
		{
			szErrMsg = "sharedContext:addEntity(type,entityID): Attempting to add a Entity '" + szEntityID + "' that already exists";
			_log.debug( szErrMsg);
			throw new EntityExistsException( szErrMsg);
		}
		
		sharedEntity = new SharedEntity( this, type.toString(), szEntityID);
		
		sharedAttr = new SharedAttribute( sharedEntity, _szUserNameAttr, ITypedValue.STRING_ATTR_VALUE_TYPE_URI_STR, true);
		
		sharedAttr.addValueToList( new SharedSimpleAttrValue( sharedAttr, szEntityID));
		
		sharedEntity.addAttr( sharedAttr);
		
		_entityMap.put( szEntityID, sharedEntity);
		_entityList.add( sharedEntity);
		return( sharedEntity);
	}

	SharedEntity addEntity(
		IEntity	copyFrom) throws IdASException, EntityExistsException
	{
		SharedEntity	sharedEntity = null;
		IAttribute				attr;
		java.util.Iterator	attrIterator;
		
		sharedEntity = addEntity( copyFrom.getModel().getType(), copyFrom.getEntityID());
		
		// Copy all of the attributes
		
		attrIterator = copyFrom.getAttributes();
		while (attrIterator.hasNext())
		{
			attr = (IAttribute)attrIterator.next();
			sharedEntity.addAttribute( attr);
		}
		return( sharedEntity);
	}
	
	private org.w3c.dom.Element _createChildElement(
		org.w3c.dom.Document	doc,
		org.w3c.dom.Element	parentElement,
		String					szElementName,
		String					szAttrName1,
		String					szAttrValue1,
		String					szAttrName2,
		String					szAttrValue2,
		String					szElementValue)
	{
		org.w3c.dom.Element	childElement;
	
		childElement = doc.createElement( szElementName);
		if (szElementValue != null)
		{
			childElement.appendChild( doc.createTextNode( szElementValue));
		}
		if (szAttrName1 != null && szAttrValue1 != null)
		{
			childElement.setAttribute( szAttrName1, szAttrValue1);
		}
		if (szAttrName2 != null && szAttrValue2 != null)
		{
			childElement.setAttribute( szAttrName2, szAttrValue2);
		}
		parentElement.appendChild( childElement);
		return( childElement);
	}
	
	private void _createValueElements(
		org.w3c.dom.Document	doc,
		org.w3c.dom.Element	attrElement,
		SharedAttribute		sharedAttr) throws IdASException
	{
		SharedSimpleAttrValue	simpleValue;
		SharedComplexAttrValue	complexValue;
		boolean						bIsSimpleDataType;
		java.util.ArrayList		valueList = sharedAttr.getValueList();
		org.w3c.dom.Element		attrValueElement;
		
		if (sharedAttr.getAttrDataType() != null)
		{
			bIsSimpleDataType = sharedAttr.isSimpleDataType();
			
			for (int iLoop = 0; iLoop < valueList.size(); iLoop++)
			{
				if (bIsSimpleDataType)
				{
					simpleValue = (SharedSimpleAttrValue)valueList.get( iLoop);
					attrValueElement = _createChildElement( doc, attrElement, XML_ATTRIBUTE_VALUE_ELEMENT,
						null, null, null, null, simpleValue.getCanonical());
				}
				else
				{
					complexValue = (SharedComplexAttrValue)valueList.get( iLoop);
					attrValueElement = _createChildElement( doc, attrElement, XML_ATTRIBUTE_VALUE_ELEMENT,
						null, null, null, null, null);
					_createAttributeElements( doc, attrValueElement, complexValue);
				}
			}
		}
	}

	private void _createAttributeElements(
		org.w3c.dom.Document	doc,
		org.w3c.dom.Element	containerElement,
		SharedAttrContainer	sharedContainer) throws IdASException
	{
		SharedAttribute		sharedAttr;
		java.util.ArrayList	attrList = sharedContainer.getAttrList();
		org.w3c.dom.Element	attrElement;
		
		for (int iLoop = 0; iLoop < attrList.size(); iLoop++)
		{
			sharedAttr = (SharedAttribute)attrList.get( iLoop);
			
			attrElement = _createChildElement( doc, containerElement, XML_ATTRIBUTE_ELEMENT,
				XML_ATTR_ID_ATTR, sharedAttr.getAttrID(),
				XML_DATA_TYPE_ATTR, sharedAttr.getAttrDataType(), null);
				
			_createValueElements( doc, attrElement, sharedAttr);
		}
	}

	private void _createEntityElements(
		org.w3c.dom.Document	doc,
		org.w3c.dom.Element	contextElement) throws IdASException
	{
		SharedEntity			sharedEntity;
		org.w3c.dom.Element	entityElement;
		
		for (int iLoop = 0; iLoop < _entityList.size(); iLoop++)
		{
			sharedEntity = (SharedEntity)_entityList.get( iLoop);
			
			entityElement = _createChildElement( doc, contextElement, XML_ENTITY_ELEMENT,
				XML_ENTITY_TYPE_ATTR, sharedEntity.getEntityType(), null, null, null);
				
			_createAttributeElements( doc, entityElement, sharedEntity);
		}
	}

	void applyUpdates() throws IdASException
	{
		javax.xml.parsers.DocumentBuilder			documentBuilder = null;
		javax.xml.parsers.DocumentBuilderFactory	documentBuilderFactory = null;
		String												szErrMsg;
		org.w3c.dom.Document								doc;
		org.w3c.dom.Element								rootElement;

		_log.debug("Applying updates ...");
		
		// Create the document tree
		
		try
		{
			documentBuilderFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
			documentBuilderFactory.setNamespaceAware( true);
			documentBuilder = documentBuilderFactory.newDocumentBuilder();
			doc = documentBuilder.newDocument();
			rootElement = doc.createElement( XML_CONTEXT_ELEMENT);
			doc.appendChild( rootElement);
			
			// Add the Entities
			
			_createEntityElements( doc, rootElement);
			
		}
		catch (IdASException e)
		{
			throw e;
		}
		catch (Exception e)
		{
			szErrMsg = "sharedContext:applyUpdates(): Exception " + e.getClass().getName() + " creating XML DOM: " + e.getMessage();
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		
		// Write out the document tree
		
		_writeXMLDocument( doc);
		_lLastModifiedTime = _file.lastModified();
	}
	
}

