/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import java.net.URI;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.higgins.audit.api.AuditEvents;
import org.eclipse.higgins.audit.api.AuditException;
import org.eclipse.higgins.audit.api.AuditOutcomes;
import org.eclipse.higgins.audit.api.AuditRecord;
import org.eclipse.higgins.configuration.common.ConfigurableComponentFactoryHelper;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;

import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.spi.BasicContext;

import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;

/**
 * 
 * @author dsanders@novell.com
 */

public class NonsharedContext extends BasicContext
{
	private Log					_log = LogFactory.getLog( NonsharedContext.class.getName());
	private Log					_auditLog = LogFactory.getLog( AuditEvents.HIGGINS_AUDIT_EVENT_LOGGER);
	private AuditRecord		_auditRecord = null;
	private SharedContext	_readOnlyContext;
	private SharedContext	_updateContext;
	private String				_szEntityID;
	private Object				_identity = null;
	private int					_iLockCount = 0;	
	private boolean			_bIsOpen = false;
	private long				_lLastModifiedTime;
	private long				_lSyncCount = 0;
	private IContextId 		_contextID;
	private String				_contextURIs = "";
	private String				_contextTypes = "";
	private Map					_contextSettings;

	/**
	 */
	public NonsharedContext(
		SharedContext	readOnlyContext,
		IContextId contextID) throws IdASException
	{
		_contextID = contextID;
		URI contextURIs[] = _contextID.getUris();
		int iLoop;
		for (iLoop = 0; iLoop < contextURIs.length; ++iLoop)
			_contextURIs = _contextURIs + contextURIs[iLoop].toString() + ", ";

		String contextTypes[] = _contextID.getTypes();
		for (iLoop = 0; iLoop < contextTypes.length; ++iLoop)
			_contextTypes = _contextTypes + contextTypes[iLoop] + ", ";

		_contextSettings = _contextID.getConfiguration();

		_readOnlyContext = readOnlyContext;
		_lLastModifiedTime = _readOnlyContext.getLastModifiedTime();
		_updateContext = null;
		_bIsOpen = false;
		_iLockCount = 0;
		_identity = null;
		_szEntityID = null;

		String strAuditRecordClass = (String)_contextSettings.get( "AuditRecordClass");
		if (strAuditRecordClass != null)
		{
			try
			{
				_auditRecord = (AuditRecord)ConfigurableComponentFactoryHelper.getInstanceFromClassName( strAuditRecordClass);
			}
			catch (Exception e)
			{
				throw new IdASException( e);
			}
		}
	}

	long getSyncCount()
	{
		return( _lSyncCount);
	}

	boolean sync(
		boolean	bForceSync) throws IdASException
	{
		boolean	bSynced = false;
		
		if (_updateContext != null)
		{
			
			// _updateContext is private to this nonsharedContext object so
			// it will never have been read back in from a file.
			
			bSynced = bForceSync;
		}
		else
		{
			
			// See if the _readOnlyContext has changed its modification time.
			
			if (bForceSync ||
				_lLastModifiedTime != _readOnlyContext.getLastModifiedTime())
			{
				_lLastModifiedTime = _readOnlyContext.getLastModifiedTime();
				bSynced = true;
			}
		}
		if (bSynced)
		{
			_lSyncCount++;
		}
		return( bSynced);
	}

	void lockSharedContext() throws IdASException
	{
		
		// Always lock the read only context, as that is the one that is shared between threads.
		// This has the negative, but unavoidable, side effect of making read operations
		// wait on update operations.
		
		if (_iLockCount == 0)
		{
			_readOnlyContext.lock();
			_readOnlyContext.syncFile();
		}
		_iLockCount++;
	}

	void unlockSharedContext()
	{
		_iLockCount--;
		if (_iLockCount == 0)
		{
			_readOnlyContext.unlock();
		}
	}

	/**
	 */
	public String open(
		Object	identity) throws IdASException
	{
		if (_bIsOpen)
		{
			throw new ContextOpenException( "nonsharedContext:open(): Context already open");
		}

		try
		{
			lockSharedContext();
			_szEntityID = _readOnlyContext.open( identity);
			_bIsOpen = true;
			if (identity instanceof AuthNAnonymousMaterials)
			{
				_identity = "[ANONYMOUS]";
			}
			else
			{
				_identity = identity;
			}
		}
		finally
		{
			_emitAuditRecord(AuditEvents.AE_CREATE_SESSION,
				AuditOutcomes.AUDIT_OUT_SUCCESS, _identity.toString(), null, null);
			unlockSharedContext();
		}
		return( _szEntityID);
	}
	
	protected void finalize()
	{
		if (_bIsOpen)
		{
			try
			{
				close();
			}
			catch (IdASException e)
			{
			}
		}
	}

	/**
	 */
	public void close() throws IdASException
	{
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonSharedContext:close(): Context not open");
		}

		_emitAuditRecord(AuditEvents.AE_TERMINATE_SESSION,
			AuditOutcomes.AUDIT_OUT_SUCCESS, _identity.toString(), null, null);
		_log.debug("Context closed: " + _contextURIs + " as: " + _identity.toString());

		if (_readOnlyContext != null)
		{
			if (_iLockCount > 0)
			{
				_readOnlyContext.unlock();
				_iLockCount = 0;
			}
			synchronized( NonsharedContextFactory.SharedContextTable)
			{
				_readOnlyContext.decrOpenCount();
				_readOnlyContext.setLastCloseTime( System.currentTimeMillis());
			}
			_readOnlyContext = null;
		}
		_bIsOpen = false;
		_identity = null;
	}

	/**
	 */
	public boolean isOpen(
		Object identity) throws IdASException
	{
		return( _bIsOpen);
	}

	boolean isOpen()
	{
		return( _bIsOpen);
	}

	SharedContext getSharedContext()
	{
		return( _updateContext == null ? _readOnlyContext : _updateContext);
	}

	boolean beginTrans() throws IdASException
	{	
		lockSharedContext();
		if (_updateContext == null)
		{
			_updateContext = new SharedContext( _readOnlyContext);
			return( true);
		}
		return( false);
	}

	/**
	 */
	public String getSchema() throws IdASException
	{
		throw new NotImplementedException( "Not Implemented: nonSharedContext:getSchema()");
	}

	/**
	 */
	public IEntity getEntity(
		String	szEntityID) throws IdASException
	{
		return( getEntity( szEntityID, null));
	}

	/**
	 */
	public IEntity getEntity(
		String					szEntityID,
		java.util.Iterator	consumerSelectionList) throws IdASException
	{
		SharedEntity	sharedEntity = null;
		
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonsharedContext:getEntity(entityID,attrList): Context not open");
		}
		
		try
		{
			lockSharedContext();
			sharedEntity = getSharedContext().getEntity( szEntityID);
		}
		finally
		{
			unlockSharedContext();
		}
		
		return( sharedEntity == null ? null : new NonsharedEntity( this, sharedEntity));
	}
	
	/**
	 */
	public java.util.Iterator getEntities(
		IFilter	filter) throws IdASException
	{
		return( getEntities( filter, null));
	}

	/**
	 */
	public java.util.Iterator getEntities(
		IFilter					filter,
		java.util.Iterator	consumerSelectionList) throws IdASException
	{
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonsharedContext:getEntities(filter,attrlist): Context not open");
		}

		_emitAuditRecord(AuditEvents.AE_QUERY_ACCOUNT, AuditOutcomes.AUDIT_OUT_SUCCESS,
			_identity.toString(), filter.toString(), null);
		_log.debug("Searching for Entities matching filter: " + filter.toString()
			+ " as: " + _identity.toString() + " in context: " + _contextURIs);

		return( new EntityIterator( this, filter));			
	}

	/**
	 */
	public void removeEntity(
		String	szEntityID) throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonsharedContext:removeEntity(entityID): Context not open");
		}
		
		try
		{
			bStartedTrans = beginTrans();
			_updateContext.removeEntity( szEntityID);
		}
		finally
		{
			if (!bStartedTrans)
			{
				unlockSharedContext();
			}
		}
	}
	
	/**
	 * 
	 */
	public IEntity addEntity(
		java.net.URI	type,
		String			szEntityID) throws IdASException
	{
		boolean					bStartedTrans = false;
		SharedEntity	sharedEntity = null;
		
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonsharedContext:addEntity(type,entityID): Context not open");
		}
		
		try
		{
			bStartedTrans = beginTrans();
			sharedEntity = _updateContext.addEntity( type, szEntityID);
		}
		finally
		{
			if (!bStartedTrans)
			{
				unlockSharedContext();
			}
		}
		return( sharedEntity == null ? null : new NonsharedEntity( this, sharedEntity));
	}
	
	/**
	 */
	public IEntity addEntity(
		IEntity	copyFrom) throws IdASException, EntityExistsException
	{
		boolean					bStartedTrans = false;
		SharedEntity	sharedEntity = null;
		
		if (!_bIsOpen)
		{
			throw new ContextNotOpenException( "nonsharedContext:addEntity(copyFrom): Context not open");
		}
		
		try
		{
			bStartedTrans = beginTrans();
			sharedEntity = _updateContext.addEntity( copyFrom);
		}
		finally
		{
			if (!bStartedTrans)
			{
				unlockSharedContext();
			}
		}
		return( sharedEntity == null ? null : new NonsharedEntity( this, sharedEntity));
	}

	public void applyUpdates() throws IdASException
	{
		if (_updateContext != null)
		{
			_updateContext.applyUpdates();
			_readOnlyContext.setDataFromContext( _updateContext);
			_updateContext = null;
			unlockSharedContext();
		}
	}
	
	public void cancelUpdates() throws IdASException
	{
		if (_updateContext != null)
		{
			_updateContext = null;
			unlockSharedContext();
		}
	}

	/**
	 * @throws IdASException
	 */
	private void _emitAuditRecord(
		int iEventNumber,
		int iOutcome,
		String sInitiatorInfo,
		String sTargetInfo,
		String sEventInfo) throws IdASException
	{
		if (_auditRecord != null)
		{
			try
			{
				_auditRecord.clearRecord();
				_auditRecord.setOriginatorInfo( "Higgins IdAS JNDI Context Provider", null, null,
									null, null, null);
				if (sTargetInfo == null)
					sTargetInfo = "";
				if (sEventInfo == null)
					sEventInfo = "";
				_auditRecord.setInitiatorInfo( null, null, sInitiatorInfo);
				_auditRecord.setTargetInfo(null, _contextURIs, _contextTypes, null, null, sTargetInfo);
				_auditLog.info( _auditRecord);
			}
			catch (AuditException e)
			{
				throw new IdASException(e);
			}
		}
	}
}

