/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import java.net.URI;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.model.IAttributeModel;

/**
 */
public class NonsharedAttribute implements ISingleValuedAttribute
{
	private Log								_log = LogFactory.getLog( NonsharedAttribute.class.getName());
	private NonsharedAttrContainer	_nonsharedAttrContainer;
	private long							_lContainerSyncCount;
	private NonsharedContext			_nonsharedContext;
	private SharedAttribute				_sharedAttr;
	private String							_szAttrID;
	private java.net.URI					_attrID;
	private long							_lSyncCount;
	
	NonsharedAttribute(
		NonsharedAttrContainer	container,
		SharedAttribute			sharedAttr)
	{
		_nonsharedAttrContainer = container;
		_lContainerSyncCount = _nonsharedAttrContainer.getSyncCount();
		_nonsharedContext = _nonsharedAttrContainer.getNonSharedContext();
		_sharedAttr = sharedAttr;
		_szAttrID = _sharedAttr.getAttrID();
		_attrID = java.net.URI.create( _szAttrID);
		_lSyncCount = 0;
	}
	
	SharedAttribute getSharedAttribute()
	{
		return( _sharedAttr);
	}
	
	NonsharedAttrContainer getNonSharedAttrContainer()
	{
		return( _nonsharedAttrContainer);
	}
	
	long getSyncCount()
	{
		return( _lSyncCount);
	}
	
	boolean sync(
		boolean	bForceSync) throws IdASException
	{
		String					szErrMsg;
		boolean					bSynced = false;
		SharedAttrContainer	sharedAttributeContainer;
		
		if (_nonsharedAttrContainer.sync( bForceSync) || _nonsharedAttrContainer.getSyncCount() != _lContainerSyncCount)
		{
			bForceSync = true;
		}

		if (bForceSync || _sharedAttr == null || _sharedAttr.isDeleted())
		{
			if ((sharedAttributeContainer = _nonsharedAttrContainer.getSharedContainer()) != null)
			{
				_sharedAttr = sharedAttributeContainer.getAttribute( _szAttrID);
			}
			else
			{
				_sharedAttr = null;
			}
			bSynced = true;
			_lContainerSyncCount = _nonsharedAttrContainer.getSyncCount();
			_lSyncCount++;
		}
		if (_sharedAttr == null)
		{
			szErrMsg = "The " + _nonsharedAttrContainer.getContainerType() + " '" + _nonsharedAttrContainer.getContainerID() + "' no longer exists";
			
			_log.debug( szErrMsg);
			throw new IdASException( szErrMsg);
		}
		return( bSynced);
	}

	/**
	 */
	public java.net.URI getAttrID() throws IdASException
	{
		return( _attrID);
	}
	
	/**
	 */
	public java.util.Iterator getValues() throws IdASException
	{
		java.util.Iterator	iter = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			if (_sharedAttr.getAttrDataType() != null)
			{
				iter = new ValueIterator( this);
			}
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		
		return( iter);
	}
	
	/**
	 */
	public Iterator getValues(IExtension[] extensions) throws IdASException 
	{
		if (extensions != null) {
			for (int index = 0; index < extensions.length; index++) {
				if (extensions[index].failIfUnsupported() == true) {
					throw new UnhandledExtensionException(extensions[index].getClass().getName() + " not supported");
				}
			}
		}

		return getValues();
	}

	/**
	 */
	public IAttributeValue addValue(
		java.net.URI	type) throws IdASException, InvalidTypeException
	{
		boolean				bStartedTrans = false;
		String				szDataType = type.toString();
		IAttributeValue	attrValue = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			if (_sharedAttr.getAttrDataType() == null)
			{
				_sharedAttr.setAttrDataType( szDataType, NonsharedContextFactory.isSimpleDataType( szDataType));
			}
			if (_sharedAttr.isSimpleDataType())
			{
				attrValue = new NonsharedSimpleAttrValue( this, _sharedAttr.addSimpleValue( szDataType, null));
			}
			else
			{
				attrValue = new NonsharedComplexAttrValue( this, _sharedAttr.addComplexValue( szDataType)); 
			}
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( attrValue);
	}

	/**
	 */
	public IAttributeValue addValue(
		IAttributeValue	copyFrom) throws IdASException
	{
		boolean				bStartedTrans = false;
		IAttributeValue		attrValue = null;
		String				szDataType = copyFrom.getValueType().toString();
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			if (_sharedAttr.getAttrDataType() == null)
			{
				_sharedAttr.setAttrDataType( szDataType, NonsharedContextFactory.isSimpleDataType( szDataType));
			}
			if (_sharedAttr.isSimpleDataType())
			{
				attrValue = new NonsharedSimpleAttrValue( this, _sharedAttr.addSimpleValue( copyFrom));
			}
			else
			{
				attrValue = new NonsharedComplexAttrValue( this, _sharedAttr.addComplexValue( copyFrom)); 
			}
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( attrValue);
	}

	/**
	 */
	public ISimpleAttrValue addSimpleValue(
		java.net.URI	dataType,
		Object			data) throws IdASException, InvalidTypeException
	{
		String				szErrMsg;
		boolean				bStartedTrans = false;
		ISimpleAttrValue	attrValue = null;
		String				szDataType = dataType.toString();
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			
			if (!NonsharedContextFactory.isSimpleDataType( szDataType))
			{
				szErrMsg = "addSimpleValue(type,data): Data type '" + szDataType + "' specified for attribute '" + _szAttrID + "' on " + _nonsharedAttrContainer.getContainerType() + " '" +
									_nonsharedAttrContainer.getContainerID() + "' is NOT a supported simple type";
				_log.debug( szErrMsg);
				throw new InvalidTypeException( szErrMsg);
			}
			if (_sharedAttr.getAttrDataType() == null)
			{
				_sharedAttr.setAttrDataType( szDataType, true);
			}
			attrValue = new NonsharedSimpleAttrValue( this, _sharedAttr.addSimpleValue( szDataType, data));
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( attrValue);
	}
	
	/**
	 */
	public IComplexAttrValue addComplexValue(
		java.net.URI	dataType) throws IdASException, InvalidTypeException
	{
		String				szErrMsg;
		boolean				bStartedTrans = false;
		String				szDataType = dataType.toString();
		IComplexAttrValue	attrValue = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			if (NonsharedContextFactory.isSimpleDataType( szDataType))
			{
				szErrMsg = "addComplexValue(type): Value type '" + szDataType + "' specified for attribute '" + _szAttrID + "' on " + _nonsharedAttrContainer.getContainerType() + " '" +
									_nonsharedAttrContainer.getContainerID() + "' is a simple type";
				_log.debug( szErrMsg);
				throw new InvalidTypeException( szErrMsg);
			}
			if (_sharedAttr.getAttrDataType() == null)
			{
				_sharedAttr.setAttrDataType( szDataType, false);
			}
			attrValue = new NonsharedComplexAttrValue( this, _sharedAttr.addComplexValue( szDataType)); 
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( attrValue);
	}

	/**
	 */
	public IAttributeValue getValue() throws IdASException
	{
		IAttributeValue		attrValue = null;
		String					szErrMsg;
		java.util.ArrayList	valueList;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			
			valueList = _sharedAttr.getValueList();
			if (_sharedAttr.getAttrDataType() == null || valueList.size() == 0)
			{
				szErrMsg = "getValue: Attribute '" + _szAttrID + "' on " + _nonsharedAttrContainer.getContainerType() + " '" + _nonsharedAttrContainer.getContainerID() + "' has no value";
				_log.debug( szErrMsg);
				throw new NotSingleValuedAttributeException( szErrMsg);
			}
			if (valueList.size() > 1)
			{
				szErrMsg = "getValue: Attribute '" + _szAttrID + "' on " + _nonsharedAttrContainer.getContainerType() + " '" + _nonsharedAttrContainer.getContainerID() + "' is NOT single-valued";
				_log.debug( szErrMsg);
				throw new NotSingleValuedAttributeException( szErrMsg);
			}
				
			if (_sharedAttr.isSimpleDataType())
			{
				attrValue = new NonsharedSimpleAttrValue( this, (SharedSimpleAttrValue)valueList.get( 0));
			}
			else
			{
				attrValue = new NonsharedComplexAttrValue( this, (SharedComplexAttrValue)valueList.get( 0)); 
			}
			
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		
		return( attrValue);
	}
	
	
	/**
	 */
	public void remove() throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			_sharedAttr.remove();
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
	}
	
	/**
	 */
	public IAttributeModel getModel() throws IdASException
	{
		IAttributeModel	attrModel = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			attrModel = new NonsharedAttributeModel( this);
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( attrModel);
	}
	
	/**
	 */
	public boolean isSingleValued() throws IdASException
	{
		boolean	bIsSingleValued =  false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			
			if (_sharedAttr.getValueList().size() == 1)
			{
				bIsSingleValued = true;
			}
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( bIsSingleValued);
	}
	
	/**
	 */
	public boolean equals(
		IAttribute	attr) throws IdASException
	{
		boolean	bEqual = true;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "Context not open");
		}
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			bEqual = _sharedAttr.equals( attr);
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( bEqual);
	}
	

	public IAttribute addAttribute(URI attrID) throws IdASException,
			InvalidTypeException {
		throw new NotImplementedException();
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		throw new NotImplementedException();
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		throw new NotImplementedException();
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		throw new NotImplementedException();
	}

	public Iterator getAttributes() throws IdASException {
		return new Vector().iterator();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID)
			throws IdASException, NotSingleValuedAttributeException {
		throw new NotImplementedException();
	}

	public void removeAttribute(URI attrID) throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(URI attrID, Object value)
			throws IdASException {
		throw new NotImplementedException();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		throw new NotImplementedException();
	}

}

